#include "Image.h"

#include "_bmp.h"

void SaveBMP(Stream& stream, const Vector<Image>& image, int bpp, bool pal_grayscale, Size dot_size, bool icon)
{
	ASSERT(stream.IsOpen() && stream.IsStoring());
	int ico_fixup = -1;
	if(icon) {
		ico_fixup = (int)stream.GetPos();
		ICONDIR id;
		Zero(id);
		id.idType = 1;
		id.idCount = image.GetCount();
		id.EndianSwap();
		stream.Put(&id, sizeof(id));
		stream.Put0(sizeof(ICONDIRENTRY) * image.GetCount());
	}
	int count = (icon ? image.GetCount() : 1);
	for(int ii = 0; ii < count; ii++) {
		int fpos = (int)stream.GetPos();
		BMPHeader header;
		const Image& img = image[ii];
		Size size = img.GetSize();
		Zero(header);
		header.biSize = sizeof(BMP_INFOHEADER);
		header.biWidth = size.cx;
		header.biHeight = size.cy;
		if(icon)
			header.biHeight <<= 1;
		header.biBitCount = bpp;
		header.biPlanes = 1;
		header.biCompression = 0/* BI_RGB */;
		int ncolors = 0;
		PaletteCv pal_cv;
		RasterFormat fmt;
		RGBA palette[256];
		switch(bpp) {
		case 1:  fmt.Set1mf(); ncolors = 2; break;
		case 4:  fmt.Set4mf(); ncolors = 16; break;
		case 8:  fmt.Set8(); ncolors = 256; break;
		case 16: fmt.Set16le(0xf800, 0x07E0, 0x001f); break;
		case 32: fmt.Set32le(0xff0000, 0x00ff00, 0x0000ff); break;
		default: fmt.Set24le(0xff0000, 0x00ff00, 0x0000ff); break;
		}
		header.biClrUsed = ncolors;
		if(ncolors) {
			if(pal_grayscale)
				for(int i = 0; i < ncolors; i++) {
					BMP_RGB& p = header.palette[i];
					p.rgbRed = p.rgbGreen = p.rgbBlue = 255 * i / (ncolors - 1);
				}
			else {
			//	CreatePalette(~img, img.GetLength(), palette, ncolors, pal_cv);
				CreatePalette(~img, img.GetLength(), palette, ncolors);
				for(int i = 0; i < ncolors; i++) {
					BMP_RGB& p = header.palette[i];
					p.rgbRed = palette[i].r;
					p.rgbGreen = palette[i].g;
					p.rgbBlue = palette[i].b;
				}
				CreatePaletteCv(palette, ncolors, pal_cv);
			}
		}
		if(bpp == 16) {
			dword *bitfields = reinterpret_cast<dword *>(header.palette);
			bitfields[2] = 0x001f;
			bitfields[1] = 0x07E0;
			bitfields[0] = 0xf800;
			header.biCompression = 3/* BI_BITFIELDS */;
			ncolors = 3;
		}
		int row_bytes = (fmt.GetByteCount(size.cx) + 3) & ~3;
		header.biSizeImage = size.cy * row_bytes;
		if(dot_size.cx && dot_size.cy) {
			header.biXPelsPerMeter = fround(header.biWidth  * (1000.0 / 25.4 * 600.0) / dot_size.cx);
			header.biYPelsPerMeter = fround(header.biHeight * (1000.0 / 25.4 * 600.0) / dot_size.cy);
		}
		if(!icon) {
			BMP_FILEHEADER bmfh;
			Zero(bmfh);
			bmfh.bfType = 'B' + 256 * 'M';
			bmfh.bfOffBits = sizeof(bmfh) + sizeof(BMP_INFOHEADER) + sizeof(BMP_RGB) * ncolors;
			bmfh.bfSize = sizeof(bmfh) + sizeof(BMP_INFOHEADER) + ncolors + header.biSizeImage;
			bmfh.EndianSwap();
			stream.Put(&bmfh, sizeof(bmfh));
		}
		header.EndianSwap();
		stream.Put(&header, sizeof(BMP_INFOHEADER) + sizeof(BMP_RGB) * ncolors);
		Buffer<byte> scanline(row_bytes);
		for(int i = 0; i < size.cy; i++) {
			fmt.Write(scanline, img[size.cy - i - 1], size.cx, pal_grayscale ? NULL : &pal_cv);
			stream.Put(scanline, row_bytes);
		}
		if(icon) { //FIX!
			fmt.Set1mf();
			row_bytes = (fmt.GetByteCount(size.cx) + 3) & ~3;
			Buffer<RGBA> alpha(size.cx);
			for(int i = 0; i < size.cy; i++) {
				memcpy(alpha, img[size.cy - i - 1], size.cx * sizeof(RGBA));
				for(RGBA *a = ~alpha + size.cy; --a >= ~alpha;)
					a->r = a->g = a->b = a->a;
				fmt.Write(scanline, alpha, size.cx, pal_grayscale ? NULL : &pal_cv);
				stream.Put(scanline, row_bytes);
			}

			int newfpos = (int)stream.GetPos();
			ASSERT(ico_fixup >= 0);
			ICONDIRENTRY ide;
			Zero(ide);
			ide.bWidth = size.cx;
			ide.bHeight = size.cy;
			ide.bColorCount = (byte)header.biClrUsed;
//			ide.wPlanes = header.biPlanes;
//			ide.wBitCount = header.biBitCount;
//			ide.bReserved = 0;
			ide.wHotSpotX = img.GetHotSpot().x;
			ide.wHotSpotY = img.GetHotSpot().y;
			ide.dwBytesInRes = newfpos - fpos;
			ide.dwImageOffset = fpos - ico_fixup;
			ide.EndianSwap();
			stream.Seek(ico_fixup + sizeof(ICONDIR) + i * sizeof(ICONDIRENTRY));
			stream.Put(&ide, sizeof(ide));
			stream.Seek(newfpos);
		}
	}
}

void SaveBMP(const char *filename, const Image& img, int bpp, bool pal_grayscale)
{
	FileOut out(filename);
	SaveBMP(out, Vector<Image>() << img, bpp, pal_grayscale, Size(0, 0), false);
}
