#ifndef _Docking_Tree_h_
#define _Docking_Tree_h_

template <class N>
class Tree
{
public:	
	class Node : public N
	{
		Node *parent;
		Node *first, *last;
		Node *next, *prev;	
		
		friend class Tree;
	public:
		Node() : parent(NULL), first(NULL), last(NULL), next(NULL), prev(NULL) { }	
	
		bool 			IsLeaf() const			{ return first == NULL; }		
		bool			IsOnlyChild() const		{ return prev == next == NULL; }
		bool			HasOneChild() const		{ return first == last != NULL; }
		int 			ChildCount() const		{ int cnt = 0; for (Node *n = first; n; n = n->next) cnt++; return cnt; }
		int				ChildCountDeep() const  { int cnt = 0; for (Node *n = first; n; n = n->next) cnt += 1 + n->ChildCount(); return cnt; }				
	};
	
	class Traverse : public Moveable<Traverse>
	{
	protected:
		Tree<N> *t;
	public:	
		virtual void  						Reset() 	= 0;
		virtual typename Tree<N>::Node *	Next() 		= 0;
		virtual typename Tree<N>::Node *	Get()		= 0;
		typename Tree<N>::Node *			NextLeaf() 	{ Node *n = Next(); while (n && !n->IsLeaf()) n = Next(); return n; }
	};
private:
	class OutOfOrderT : public Traverse, public Moveable<OutOfOrderT>
	{
		int ix;
		Array<typename Tree<N>::Node> *node;		
	public:
		OutOfOrderT() : node(NULL), ix(0) 	{  } 
		void Set(Tree<N> *tree, Array<typename Tree<N>::Node> *_node, int n = 0) { ASSERT(ix >= 0); t = tree; node = _node; ix = n; }
		
		virtual void 		   			Reset()			{ ix = 0; }
		virtual typename Tree<N>::Node *Get()			{ return (ix < node->GetCount()) ? &(*node)[ix] : NULL; }		
		virtual typename Tree<N>::Node *Next()			{ return (ix < node->GetCount()) ? &(*node)[ix++] : NULL; } 
	};
	
	class PreOrderT : public Traverse, public Moveable<PreOrderT>
	{
		typename Tree<N>::Node *n, *p;
		typename Tree<N>::Node	*Left(typename Tree<N>::Node *root);
	public:		
		PreOrderT() : p(NULL), n(NULL)										{ } 
		void					 		Set(Tree<N> *tree, typename Tree<N>::Node *start) 	{t = tree; p = start; }
		void 		  	   				Reset()								{ n = NULL; }		
		virtual typename Tree<N>::Node *Get()								{ return n; }
		virtual typename Tree<N>::Node *Next();	
	};
	
	class PostOrderT : public PreOrderT, public Moveable<PostOrderT>
	{
	public:
		virtual	typename Tree<N>::Node *Next();
	};

	Array<Node>  node;
	Node		 rootnode;
	Node		*root;		 
	One<Traverse> ts;
	
	int		Find(Node *n) const			{ for (int i = 0; i < node.GetCount(); i++) if (&node[i] == n) return i; return -1; }
	void	DoRemove(Node *n);

	Node & 	AddFirst(Node *p, Node &n);
	Node & 	AddLast(Node *p, Node &n);
	Node & 	AddBefore(Node *s, Node &n);
	Node & 	AddAfter(Node *s, Node &n);	
public:
	Node & 	AddFirst(Node *p)			{ ASSERT(p); return AddFirst(p, node.Add()); }
	Node & 	AddLast(Node *p)			{ ASSERT(p); return AddLast(p, node.Add()); }			
	Node & 	AddBefore(Node *s)			{ ASSERT(s); return AddBefore(s, node.Add()); }
	Node & 	AddAfter(Node *s)			{ ASSERT(s); return AddAfter(s, node.Add()); }
	void 	RemoveDeep(Node *n);
	void 	Remove(Node *n);			

	Node *  Root() const				{ return root; }
	Node *	Parent(Node *n) const		{ ASSERT(n); return n->parent; }
	Node *	Next(Node *n) const			{ ASSERT(n); return n->next; }
	Node *	Prev(Node *n) const			{ ASSERT(n); return n->prev; }
	Node *	First(Node *n) const		{ ASSERT(n); return n->first; }
	Node *	Last(Node *n) const			{ ASSERT(n); return n->last; }

	Traverse &OutOfOrder() 		 { ts.Clear(); ts.Create<OutOfOrderT>().Set(this, &node, 0); return *ts; }
	Traverse &PreOrder() 		 { return PreOrder(root); }
	Traverse &PostOrder() 		 { return PostOrder(root); }
	Traverse &OutOfOrder(Node *n){ ASSERT(n); ts.Clear(); ts.Create<OutOfOrderT>().Set(this, node, Find(n)); return *ts; }
	Traverse &PreOrder(Node *p)  { ASSERT(p); ts.Clear(); ts.Create<PreOrderT>().Set(this, p); return *ts;  }
	Traverse &PostOrder(Node *p) { ASSERT(p); ts.Clear(); ts.Create<PostOrderT>().Set(this, p); return *ts;  }
	
	Tree<N>()					 { root = &rootnode; }
};

template <class N>
typename Tree<N>::Node *Tree<N>::PreOrderT::Left(typename Tree<N>::Node *root)
{ 
	n = t->First(t->Root()); 
	while (n && t->First(n)) n = t->First(n); 
	return n ? n : t->Root();
}

template <class N>
typename Tree<N>::Node *Tree<N>::PreOrderT::Next()
{ 
	if (n == p) n = NULL;
	else if (!n) n = Left(p); 
	else if (t->Next(n)) n = Left(t->Next(n)); 
	else n = t->Parent(n);
	return n;
}

template <class N>
typename Tree<N>::Node *Tree<N>::PostOrderT::Next()
{ 
	if (!n) n = p; 
	else if (t->First(n)) n = t->First(n); 
	else if (t->Next(n)) n = t->Next(n);
	else {
		n = t.Parent(n);
		while (n && !t->Next(n))
			n = t->Parent(n);
		n = n ? t->Next(n) : NULL;
	}
	return n;
}

template <class N>
typename Tree<N>::Node &Tree<N>::AddFirst(typename Tree<N>::Node *p, typename Tree<N>::Node &n)
{
	if (!p->IsLeaf()) return AddBefore(p->first, n);
	p->first = &n;
	p->last = &n;
	n.parent = p;
	n.next = n.prev = NULL;
	return n;
}

template <class N>
typename Tree<N>::Node &Tree<N>::AddLast(Node *p, typename Tree<N>::Node &n)
{
	if (!p->IsLeaf()) return AddAfter(p->last, n);
	p->first = &n;
	p->last = &n;
	n.parent = p;
	n.next = n.prev = NULL;
	return n;
}

template <class N>
typename Tree<N>::Node &Tree<N>::AddBefore(typename Tree<N>::Node *s, typename Tree<N>::Node &n)
{
	n.prev = s->prev;
	if (n.prev) n.prev->next = &n;
	n.next = s;
	s->prev = &n;
	n.parent = s->parent;
	if (s->parent && s->parent->first == s) s->parent->first = &n;
	return n;
}

template <class N>
typename Tree<N>::Node &Tree<N>::AddAfter(typename Tree<N>::Node *s, typename Tree<N>::Node &n)
{
	n.next = s->next;
	if (n.next) n.next->prev = &n;
	n.prev = s;
	s->next = &n;
	n.parent = s->parent;
	if (s->parent && s->parent->last == s) s->parent->last = &n;
	return n;
}

template <class N>
void Tree<N>::RemoveDeep(typename Tree<N>::Node *n)
{
	ASSERT(n && n->parent);
	int cnt;
	if (n->first && (cnt = n->ChildCountDeep()) > 5) {
		Vector<int> sorted_list;
		sorted_list.SetCount(cnt);
		PreOrderT t;
		t.Set(this, n);
		cnt = 0;
		while (t.Next())
			sorted_list.Set(cnt++, Find(t.Get()));
		Sort(sorted_list);
		DoRemove(n);	
		node.Remove(sorted_list);	
	}
	else {
		DoRemove(n);
		node.Remove(Find(n));
	}
}

template <class N>
void Tree<N>::Remove(typename Tree<N>::Node *n)
{
	ASSERT(n && n->parent);
	Node *t;
	for (Node *c = n->first; c; c = t) {
		t = c->next;
		AddAfter(n, *c);
	}
	DoRemove(n);
	node.Remove(Find(n));
}

template <class N>
void Tree<N>::DoRemove(typename Tree<N>::Node *n)
{
	if (n->next) n->next->prev = n->prev;
	if (n->prev) n->prev->next = n->next;
	if (n->parent->first == n) n->parent->first = n->next;
	if (n->parent->last == n) n->parent->last = n->prev;
}


#endif
