#include "Docking.h"

void SplitterTree::Paint(Draw &w)
{
	if (!IsTransparent())
		w.DrawRect(GetSize(), SColorFace());
	if (HasCapture() && drag)
		w.DrawRect(DragHlRect(), GUI_GlobalStyle() >= GUISTYLE_XP ? Blend(SColorHighlight, SColorFace) : SColorShadow);
}

void SplitterTree::LeftDown(Point p, dword keyflags)
{
	if (MouseInBar(p, tree.Root(), GetView(), true)) {
		SetCapture();
		dragpos = p;
		Refresh(DragHlRect());
	}
}

void SplitterTree::LeftUp(Point p, dword keyflags)
{
	if (HasCapture() && drag) {
		RefreshNode(tree.Parent(drag));
		ReleaseCapture();
		drag = NULL;
	}
}

void SplitterTree::MouseMove(Point p, dword keyflags)
{
	if (HasCapture() && drag) {
		int dif = 0;
		Node *parent = tree.Parent(drag);
		Node *next = tree.Next(drag);
		Rect prect = GetNodeRect(parent);
		if (parent->vert) {
			int h = prect.Height();
			if (!h) return;
			dif = p.y - dragpos.y;
			dif = minmax((dif * 10000) / h, -drag->sz, next->sz);	
		}
		else {
			int w = prect.Width();
			if (!w) return;
			dif = p.x - dragpos.x;
			dif = minmax((dif * 10000) / w, -drag->sz, next->sz);	
		}
		drag->sz += dif;
		next->sz -= dif;
		dragpos = p;
		LayoutNode(parent, prect);
		Refresh(prect);
	}
}

Image SplitterTree::CursorImage(Point p, dword keyflags)
{
	Node *n = drag ? tree.Parent(drag) : MouseInBar(p, tree.Root(), GetView(), false);
	if (n) return n->vert ? Image::SizeVert() : Image::SizeHorz();	
	return Image::Arrow();
}

SplitterTree::Node *SplitterTree::MouseInBar(const Point &pt, Node *p, Rect r, bool store_nodes)
{
	if (p->IsLeaf()) return false;
	Node *n;
	for (n = tree.First(p); n; n = tree.Next(n)) {
		Rect r = GetNodeRect(n);
		if (p->vert) {
			if (pt.y < r.top)
				break;
			else if (pt.y < r.bottom)
				return MouseInBar(pt, n, r, store_nodes);				
		}
		else {
			if (pt.x < r.left)
				break;
			else if (pt.x < r.right)
				return MouseInBar(pt, n, r, store_nodes);					
		}
		
	}
	if (!n || !tree.Prev(n)) return false;
	if (store_nodes) drag = tree.Prev(n);
	return p;
}

Rect SplitterTree::DragHlRect()
{
	Rect r = GetNodeRect(drag);
	if (tree.Parent(drag)->vert) {
		r.top = r.bottom;
		r.bottom += 4;	
	}
	else {
		r.left = r.right;
		r.right += 4;	
	}
	return r;	
}

void SplitterTree::LayoutNode(Node *p, Rect r)
{
	if (p->IsLeaf() && p != tree.Root()) {
		Ctrl *c = p->ctrl;
		ASSERT(c);
		c->SetRect(r);
	}
	int rsz = (p->vert ? r.Height() : r.Width()) - (p->ChildCount()-1) * 4;
	if (p->vert) {
		int bt = r.bottom;
		for (Node *n = tree.First(p); n; n = tree.Next(n)) {
			r.bottom = r.top + (rsz*n->sz)/10000;
			if (!tree.Next(n)) r.bottom = bt;
			LayoutNode(n, r);
			r.top = r.bottom + 4;
		}
	}
	else {
		int rt = r.right;
		for (Node *n = tree.First(p); n; n = tree.Next(n)) {
			r.right = r.left + (rsz*n->sz)/10000;
			if (!tree.Next(n)) r.right = rt;
			LayoutNode(n, r);
			r.left = r.right + 4;
		}
	}
}

Size SplitterTree::HVSize(bool vert, const Size &a, const Size &b) 
{
	return vert ? Size(max(a.cx, b.cx), a.cy + b.cy + 4) : Size(a.cx + b.cx + 4, max(a.cy, b.cy));
}

Size SplitterTree::GetNodeMinSize(Node *n) 
{
	if (n->IsLeaf()) return n->ctrl->GetMinSize();
	ASSERT(!n->IsEmpty());
	Size sz(0, 0);
	int cnt = -1;
	for (Node *q = tree.First(n); q; q = tree.Next(q)) {
		sz = HVSize(n->vert, GetNodeMinSize(q), sz);
		cnt++;
	}
	return sz + cnt*4;
}

Size SplitterTree::GetNodeMaxSize(Node *n) 
{
	if (n->IsLeaf()) return n->ctrl->GetMaxSize();
	Size sz(0, 0);
	int cnt = -1;
	for (Node *q = tree.First(n); q; q = tree.Next(q)) {
		sz = HVSize(n->vert, GetNodeMaxSize(q), sz);
		cnt++;
	}
	return sz + cnt*4;
}

void SplitterTree::FixChildSizes(Node *p)
{
	Node *first = tree.First(p);
	if (!first) return;
	int sum = 0;
	int cnt = 0;
	for (Node *n = first; n; n = tree.Next(n)) {
		sum += n->sz;	
		cnt++;
	}
	if (cnt == 1) { first->sz = 10000; return; }
	sum -= 10000;
	int rem = sum % cnt;
	sum /= cnt;
	for (Node *n = first; n; n = tree.Next(n))
		n->sz -= sum;
	first->sz += rem;
}

void SplitterTree::SetNodeSize(Node *n, const Size &sz)
{
	Node *p = tree.Parent(n);
	if (p->HasOneChild())
		n->sz = 10000;
	else if (sz.IsNullInstance() || !IsVisible())
		n->sz = 10000/(p->ChildCount());	
	else {
		int nsz = p->vert ? sz.cy : sz.cx;
		int msz = 0;
		int psz = p->vert ? GetNodeRect(p).Height() : GetNodeRect(p).Width(); // Get total size of parent
		int cnt = 0;

		for (Node *q = tree.First(p); q; q = tree.Next(q)) {
			q->sz = p->vert ? GetNodeMinSize(q).cy : GetNodeMinSize(q).cx;
			if (q == n)	q->sz = min(nsz, q->sz);	
			msz += q->sz;
			cnt++;
		}
		psz -= (cnt-1) * 4;
		if (psz <= 0) psz = msz;
		int dif = psz - msz;
		int rem = 0;
		if (dif < 0) {
			// Not enough space
			rem = dif % cnt;
			dif /= cnt;
			for (Node *q = tree.First(p); q; q = tree.Next(q))
				q->sz += dif;
			n->sz += rem;
		}
		else if (dif > 0) {
			// We have spare space
			int t = nsz - n->sz;
			if (t > dif) {
				n->sz += dif;
				dif = 0;
			}
			else if (t > 0) {
				n->sz += t;
				dif -= t;
			}		
			rem = dif % (cnt-1);	
			dif /= (cnt-1);	
			for (Node *q = tree.First(p); q; q = tree.Next(q))
				if (q != n) q->sz += dif; 
			if (p->HasOneChild())
				n->sz += rem;
			else {
				if (tree.First(p) == n)
					tree.Next(n)->sz += rem;
				else
					tree.First(p)->sz += rem;
			}
		}
		for (Node *q = tree.First(p); q; q = tree.Next(q))
			q->sz = minmax((q->sz * 10000) / psz, 0, 10000);
	}
}

Rect SplitterTree::GetNodeRect(Node *n)
{
	if (n->IsLeaf()) {
		ASSERT(n->ctrl);
		return n->ctrl->GetRect();	
	}
	else if (n->HasOneChild())
		return GetNodeRect(tree.First(n));
	return Rect(GetNodeRect(tree.First(n)).TopLeft(), GetNodeRect(tree.Last(n)).BottomRight()); 
}

SplitterTree::Node * SplitterTree::FindCtrl(Ctrl *c)
{
	Tree<CtrlNode>::Traverse &t = tree.OutOfOrder();
	while (Node *n = t.NextLeaf())
		if (n->ctrl == c)
			return n;
	return NULL;
}

void SplitterTree::CleanTree(Node *n)
{
	if (n->HasOneChild() && n != tree.Root()) {
		// If only one child left we should merge up
		Node *c = tree.First(n);
		c->sz = n->sz;
		tree.Remove(n);
		CleanTree(tree.Parent(c));		
	}		
	else if (n->IsLeaf() && n->IsEmpty() && n != tree.Root()) {
		// Remove empty nodes (without ctrls)
		Node *p = tree.Parent(n);
		Node *c = n;
		while (p->IsLeaf() && p->IsEmpty()) {
			c = p;
			p = tree.Parent(c);
		}
		tree.RemoveDeep(c);
		CleanTree(p);		
		FixChildSizes(p);
	}
	else 
		FixChildSizes(n);
}

Ctrl & SplitterTree::AddSibling(bool before, Ctrl &sibling, Ctrl &c, Size sz)
{
	Node *s = FindCtrl(&sibling);
	ASSERT(s);
	SetNodeSize(AddBA(before, s, c), sz);
	Layout();
	return c;
}

Ctrl & SplitterTree::AddSiblingOver(bool vert, bool before, Ctrl &sibling, Ctrl &c, Size sz)
{
	if (HasChild(&c)) return c;
	Node *s = FindCtrl(&sibling);
	ASSERT(s);
	Node *p = tree.Parent(s);
	if (p->vert == vert || p->HasOneChild()) { // Don't change orientation of root node
		p->vert = vert;
		SetNodeSize(AddBA(before, s, c), sz);
	}
	else {
		Node *pp = tree.Parent(p);
		if (pp) {
			ASSERT(vert == pp->vert);
			Node *n = AddBA(before, p, c);
			n->sz = 10000/max(p->ChildCount()-1, 1);
			FixChildSizes(pp);
		}
	}
	Layout();
	return c;
}

Ctrl & SplitterTree::AddSiblingUnder(bool vert, bool before, Ctrl &sibling, Ctrl &c, Size sz)
{
	if (HasChild(&c)) return c;
	Node *s = FindCtrl(&sibling);
	ASSERT(s);
	Node *p = tree.Parent(s);
	if (p->vert == vert || (p != tree.Root() && p->HasOneChild())) { // Don't change orientation of root node
		p->vert = vert;
		SetNodeSize(AddBA(before, s, c), sz);
	}
	else {
		Node *n = &tree.AddFirst(s);
		n->sz = 10000;		
		n->vert = s->vert;
		n->ctrl = s->ctrl;
		s->vert = vert;
		s->ctrl = NULL;
		SetNodeSize(AddBA(before, n, c), sz);
	}
	Layout();// Change to LayoutNode(p)?
	return c;
}

SplitterTree::Node * SplitterTree::AddBA(bool before, SplitterTree::Node *p, Ctrl &c)
{
	BlockAdd(c);
	Node & n = (before ? tree.AddBefore(p) : tree.AddAfter(p));	
	n.ctrl = &c;
	return &n;	
}

void SplitterTree::AddRoot0(Ctrl &ctrl, bool first, const Size &sz)
{
	Node &n = first ? tree.AddFirst(tree.Root()) : tree.AddLast(tree.Root());
	n.ctrl = &ctrl;
	BlockAdd(ctrl);
	SetNodeSize(&n, sz);
	Layout();
}

Ctrl & SplitterTree::Swap(Ctrl &oldctrl, Ctrl &newctrl)
{
	Node *l = FindCtrl(&oldctrl);
	ASSERT(l);	
	internal_block = true;
	l->ctrl->Remove();
	l->ctrl = &newctrl;
	Add(newctrl);
	internal_block = false;	
	newctrl.SetRect(oldctrl.GetRect());
	return newctrl;
}

void SplitterTree::RemoveFromTree(Ctrl &ctrl)
{
	Node *l = FindCtrl(&ctrl);
	ASSERT(l);
	Node *n = tree.Parent(l);	
	tree.Remove(l);
	CleanTree(n);
}

Ctrl * SplitterTree::GetNextSibling(Ctrl &c)
{ 
	Node *l = FindCtrl(&c); 
	return (l && tree.Next(l) ? tree.Next(l)->ctrl : NULL); 
}

Ctrl * SplitterTree::GetPrevSibling(Ctrl &c)
{
	Node *l = FindCtrl(&c); 
	return (l && tree.Prev(l) ? tree.Prev(l)->ctrl : NULL); 
}


SplitterTree::SplitterTree()
{
	internal_block = false;
	drag = NULL;
}

