#include "DialSlider.h"

#define IMAGECLASS DSImg
#define IMAGEFILE <DialSlider/DialSlider.iml>
#include <Draw/iml_source.h>

DialSlider::DialSlider()
{
	min = 0;
	max = INT_MAX;
	marker = true;
	labels = true;
	SetPos(0);
	BackPaint();
	SetStyle(StyleDefault());
}

void DialSlider::LeftDown(Point p, dword keyflags)
{
	if (PointInSlider(p)) {
		SetCapture();
		MouseMove(p, keyflags);
	}
}

void DialSlider::LeftUp(Point p, dword keyflags)
{
	ReleaseCapture();
}

void DialSlider::MouseMove(Point p, dword keyflags)
{
	if (HasCapture()) {
		int op = pos;
		int n = ClientToSlider(p);
		if (!IsNull(n))
			SetPos(n);
		if (op != pos)
			Action();
	}
}

void DialSlider::Paint(Draw &w)
{
	Size sz = GetImageSize();
	bool enabled = IsEnabled();
	// BG
	w.DrawRect(GetSize(), SColorFace);
	// Draw bar fill
	if (posimg.IsEmpty())
		posimg = GetPosImage();
	ChPaint(w, sz, posimg);
	// Draw center arrow
	if (marker && markerimg.IsEmpty())
		markerimg = GetMarkerImage();
	if (marker) {
		// Calculate correct positon for marker
		Size msz = markerimg.GetSize();
		Rect r(-msz/2, msz);
		// Scale rect
		double scale = (double)sz.cy / style->frame.GetSize().cy;
		r.left = fround(r.left * scale);
		r.top = fround(r.top * scale);
		r.right = fround(r.right * scale);
		r.bottom = fround(r.bottom * scale);
		// Draw
		r.Offset(sz.cx/2, sz.cy);		
		ChPaint(w, r, DisabledImage(markerimg, !enabled));
	}
	// Draw bar frame and labels
	ChPaint(w, sz, DisabledImage(style->frame, !enabled));
	if (labels)
		PaintLabels(w);
}

void DialSlider::PaintLabels(Draw &w)
{
	String minstr = AsString(min);
	String maxstr = AsString(max);
	Size isz = GetImageSize();
	int	inset = iscale(24, isz.cx, style->frame.GetSize().cx);
	Point p(inset, isz.cy);
	w.DrawText(p.x - GetTextSize(minstr, style->font).cx/2, p.y, minstr, style->font, SColorText());
	w.DrawText(isz.cx - p.x - GetTextSize(maxstr, style->font).cx/2, p.y, maxstr, style->font, SColorText());
}

void DialSlider::MinMax(int _min, int _max)
{
	min = _min;
	max = _max;
	SetPos(GetPos());
}

void DialSlider::SetPos(int _pos)
{
	_pos = minmax(_pos, min, max);
	if (pos != _pos) {
		pos = _pos;
		UpdateSync();
	}
}

void DialSlider::SetData(const Value& data)
{
	SetPos((int)data);
}

Image DialSlider::GetPosImage()
{
	Size sz = style->pos.GetSize();
	ImageDraw w(sz);
	
	// Force alpha channel init
	w.Alpha().DrawRect(sz, Black());
	// Set our mask
	w.Alpha().DrawImage(0, 0, style->mask);
	// Draw bar background
	w.DrawRect(sz, White());

	if (pos > min) {
		double 	cp = sz.cy/2.0f;
		Point 	centeri;
		double 	sin, cos;
		int 	a = GetAngle();
		
		// Rotates the drawing offset to the correct position (necessary because Rotate always 
		//	rotates around the images center point).
		// We only need to do one coord here (cp) as x is 0.
		Draw::SinCos(a, sin, cos);
		centeri.x = fround(sz.cx/2.0 + cp*sin);
		centeri.y = fround(sz.cy - cp*cos);

		// Rotate image
		Image p = Rotate(style->pos, a);
		sz = p.GetSize();
		// Draw new image in the correct position
		w.DrawImage(centeri.x-sz.cx/2, centeri.y-sz.cy/2, p);
	}
	return w;
}

Image DialSlider::GetMarkerImage()
{
	return Rotate(style->marker, GetAngle());
}

bool DialSlider::PointInSlider(Point p)
{
	Size csz = GetImageSize();
	Size isz = style->frame.GetSize();
	// Make sure p is inside the drawn area
	if (p.x > csz.cx || p.y > csz.cy) 
		return false;
	// Convert p into image coordinates
	Point ip(iscale(p.x, isz.cx, csz.cx), iscale(p.y, isz.cy, csz.cy));
	// Is p on a non-empty pixel?
	if (style->pos[ip.y][ip.x].a < 2)
		return false;	
	return true;
}

int DialSlider::ClientToSlider(Point p)
{
	Size isz = GetImageSize();

	// Calculate the angle of p relate to the marker
	p.x -= isz.cx / 2;
	p.y -= isz.cy;
	double rad = atan((double)p.y / p.x);
	if (p.x >= 0) // Handles right quadrant
		rad += M_PI;
	// Convert angle into pos
	return min + (int)((max-min) * rad/M_PI);
}

Size DialSlider::GetImageSize()
{
	// Returns a drawing area constrained by the proportions of our images
	Size csz = GetSize();
	int mcy = csz.cy - (labels ? style->font.GetHeight() + 2 : 0);
	double ratio = (double)style->frame.GetSize().cy / style->frame.GetSize().cx;
	int y = (int)(ratio * csz.cx);
	if (y > mcy)
		return Size((int)(mcy/ratio), mcy);
	return Size(csz.cx, y);		
}

void DialSlider::UpdateSync()
{
	posimg 		= Image();
	markerimg 	= Image();
	UpdateRefresh();
}

int DialSlider::GetAngle() const
{
	return 1800 + iscale(1800, pos-min, max-min);
}

CH_STYLE(DialSlider, Style, StyleDefault)
{
	frame 	= DSImg::frame();	
	mask 	= DSImg::mask();
	pos 	= DSImg::pos();
	marker 	= DSImg::marker();
	font	= StdFont();
}

