#include "RowList.h"

void RowList::MouseWheel(Point p, int zdelta, dword keyflags) {
	sb.Wheel(zdelta);
}

int  RowList::GetColumnCx(int i) const {
	return cx;
}

int RowList::GetColumnItems() const {
	if(cy == 0) return 0;
	return GetSize().cy / cy;
}

void RowList::ShiftSelect() {
	if(anchor < 0)
		anchor = cursor;
	int l = min(anchor, cursor);
	int h = max(anchor, cursor);
	for(int i = 0; i < GetCount(); i++) {
		bool sel = i >= l && i <= h;
		if(item[i].canselect && item[i].sel != sel) {
			item[i].sel = sel;
			RefreshItem(i);
		}
	}
	UpdateSelect();
}

Image RowList::CursorImage(Point p, dword)
{
	return HasCapture() ? Image::SizeHorz() : Image::Arrow();
}

int RowList::GetItem(Point p)
{
	int i = (p.y / cy) * ncl; 
	i += (p.x / cx);
	if(i < 0 || i >= GetPageItems())
		return -1;
	i += sb * ncl;
	return i >= 0 && i < GetCount() ? i : -1;
}

void RowList::PointDown(Point p) {
	int i = GetItem(p);
	if(i >= 0 && i < GetCount())
		SetCursor0(i, false);
	else
	if(clickkill)
		KillCursor();
}

void RowList::DoClick(Point p, dword flags)
{
	SetWantFocus();
	PointDown(p);
	p.y %= cy;
	p.x %= GetColumnCx(0);
	if(multi && cursor >= 0) {
		if(flags & K_SHIFT)
			ShiftSelect();
		else
		if(flags & K_CTRL)
			SelectOne(cursor, !IsSelected(cursor));
		else {
			ClearSelection();
			SelectOne(cursor, true);
			anchor = cursor;
		}
	}
	WhenLeftClick();
	WhenLeftClickPos(p);
}

void RowList::LeftDown(Point p, dword flags) {
	selclick = false;
	int i = GetItem(p);
	if(i >= 0 && i < GetCount() && multi && IsSel(i))
		selclick = true;
	else
		DoClick(p, flags);
}

void RowList::LeftUp(Point p, dword flags)
{
	if(selclick)
		DoClick(p, flags);
	selclick = false;
}

void RowList::LeftDrag(Point p, dword keyflags)
{
	if(!HasCapture())
		WhenDrag();
}

void RowList::RightDown(Point p, dword flags) {
	int i = GetItem(p);
	if(!(i >= 0 && i < GetCount() && multi && IsSel(i)))
		PointDown(p);
	MenuBar::Execute(WhenBar);
}

void RowList::LeftDouble(Point p, dword flags) {
	LeftDown(p, flags);
	WhenLeftDouble();
}

void RowList::SyncInfo()
{
	if((HasMouse() || info.HasMouse()) && !HasCapture() && popupex) {
		Point p = GetMouseViewPos();
		int i = GetItem(p);
		if(i >= 0) {
			Rect r = GetItemRect(i);
			if(p.x > r.left + 3 && p.x < r.right - 2) {
				Color ink, paper;
				dword style;
				GetItemStyle(i, ink, paper, style);
				Item& m = item[i];
				info.Set(this, r, m.value, m.display ? m.display : display, ink, paper, style);
				return;
			}
		}
	}
	info.Cancel();
}

void RowList::CancelMode()
{
	info.Cancel();
	dropitem = -1;
	anchor = -1;
	selclick = false;
}

void RowList::MouseLeave()
{
	if(!info.IsOpen())
		info.Cancel();
}

Rect RowList::GetItemRect(int i) const {
	if(i < 0) return Rect(0, 0, 0, 0);
	return RectC( (i % ncl) * cx, (i / ncl - sb) * cy, cx, cy);
}

void RowList::Page(bool down) {
	int q = sb;
	if(down)
		sb.NextPage();
	else
		sb.PrevPage();
	if(q == sb)
		SetCursor(down ? GetCount() - 1 : 0);
	else
		SetCursor(cursor - q + sb);
}

bool RowList::Key(dword _key, int count) {
	int c = cursor;
	bool sel = _key & K_SHIFT;
	int key = _key & ~K_SHIFT;
	switch(key) {
	case K_UP:
		if(c < 0)
			c = GetCount() - 1;
		else
		if(c > 0)
			c--;
		break;
	case K_DOWN:
		if(c < 0)
			c = 0;
		else
		if(c < GetCount() - 1)
			c++;
		break;
	case K_LEFT:
		c = max(c - GetColumnItems(), 0);
		break;
	case K_RIGHT:
		c = min(c + GetColumnItems(), GetCount() - 1);
		break;
	case K_PAGEUP:
		Page(false);
		return true;
	case K_PAGEDOWN:
		Page(true);
		return true;
	case K_HOME:
		c = 0;
		break;
	case K_END:
		c = GetCount() - 1;
		break;
	default:
		return MenuBar::Scan(WhenBar, _key);
	}
	if(sel) {
		SetCursor0(c, false);
		ShiftSelect();
	}
	else {
		SetCursor(c);
		anchor = c;
	}
	return true;
}

void RowList::RefreshSel()
{
	for(int i = 0; i < item.GetCount(); i++)
		if(item[i].sel)
			RefreshItem(i);
}

void RowList::ClearSelection() {
	bool upd = false;
	for(int i = 0; i < item.GetCount(); i++)
		if(item[i].sel) {
			item[i].sel = false;
			upd = true;
			RefreshItem(i);
		}
	if(upd)
		UpdateSelect();
}

void RowList::SelectOne(int i, bool sel) {
	if(!multi) return;
	if(item[i].canselect)
		item[i].sel = sel;
	UpdateSelect();
	RefreshItem(i);
}

bool RowList::IsSelected(int i) const {
	return item[i].sel;
}

bool RowList::IsSel(int i) const
{
	return IsMultiSelect() ? IsSelected(i) : GetCursor() == i;
}

void RowList::UpdateSelect() {
	selcount = 0;
	for(int i = 0; i < item.GetCount(); i++)
		if(item[i].sel)
			selcount++;
	SyncInfo();
	Action();
	WhenSelection();
	WhenSel();
}

void RowList::GetItemStyle(int i, Color& fg, Color& bg, dword& st)
{
	st = 0;
	bool hasfocus = HasFocusDeep() && !IsDragAndDropTarget();
	bool drop = i == dropitem && !insert;
	if(IsReadOnly())
		st |= Display::READONLY;
	if(IsSelected(i))
		st |= Display::SELECT;
	if(i == cursor)
		st |= Display::CURSOR;
	if(drop) {
		hasfocus = true;
		st |= Display::CURSOR;
	}
	if(hasfocus)
		st |= Display::FOCUS;
	fg = SColorText;
	bg = SColorPaper;
	if(nobg)
		bg = Null;
	if((st & Display::SELECT) ||
	   (!multi || !item[i].canselect && selcount == 0) && (st & Display::CURSOR) ||
	   drop) {
		fg = hasfocus ? SColorHighlightText : SColorText;
		bg = hasfocus ? SColorHighlight : Blend(SColorDisabled, SColorPaper);
	}
}

dword RowList::PaintItem(Draw& w, int i, const Rect& r)
{
	Color ink, paper;
	dword style;
	GetItemStyle(i, ink, paper, style);
	const Item& m = item[i];
	w.Clip(r);
	(m.display ? m.display : display)->Paint(w, r, m.value, ink, paper, style);
	w.End();
	return style;
}

void RowList::Paint(Draw& w) {
	Size sz = GetSize();
	if(GetColumnCx(0) == 0) return;
	int y = 0;
	int i = sb*ncl;
	int coli = 0;
	while(y <= sz.cy) {
		int x = 0;
		while (x + cx < sz.cx) {
			Rect rect = RectC(x, y, cx, cy);
			if(i < GetCount()) {
				if(w.IsPainting(rect)) {
					Rect r = rect;
					r.right--;
					dword style = PaintItem(w, i, r);
//					w.DrawRect(rect.right - 1, rect.top, 1, rect.Height(),
//					           x + cx < sz.cx ? SColorDisabled : SColorPaper);
					if(i == cursor && selcount != 1 && multi && item[i].canselect)
						DrawFocus(w, r, style & Display::SELECT ? SColorPaper() : SColorText());
				}
			}
			else
				w.DrawRect(rect, SColorPaper);
			if(i == dropitem && insert)
				DrawHorzDrop(w, x, y, cx);
			if(i + 1 == dropitem && insert && y + 2 * cy > sz.cy)
				DrawHorzDrop(w, x, y + cy - 2, cx);
			x += cx;
			i++;
		}
		w.DrawRect(x, y, cx, sz.cy - y, SColorPaper);
		y += cy;
	}
	scroller.Set(sb);
}

Image RowList::GetDragSample()
{
	Size sz = StdSampleSize();
	ImageDraw iw(sz);
	int y = 0;
	for(int i = 0; i < GetCount() && y < sz.cy; i++)
		if(IsSel(i)) {
			PaintItem(iw, i, RectC(0, y, sz.cx, cy));
			y += cy;
		}
	return Crop(iw, 0, 0, sz.cx, y);
}

int  RowList::GetPageItems() const {
	return ncl * GetColumnItems();
}

void RowList::SetSb() {
	int rows = GetCount()/ncl;
	sb.SetTotal(rows + ((GetCount() - rows*ncl) ? 1 : 0));
	sb.SetPage(GetColumnItems());
}

void RowList::Layout() {
	SetSb();
	SetColumns();
}

void RowList::SetColumns()
{
	int i = sb.Get() * ncl;
	ncl = max(1, GetSize().cx / max(cx, 1));	
	sb.Set(i / ncl);
}

void RowList::Scroll() {
	Size sz = GetSize();
	sz.cy = sz.cy / cy * cy;
	scroller.Scroll(*this, sz, sb, cy);
	info.Cancel();
}

void RowList::RefreshItem(int i, int ex)
{
	if(i >= 0) {
		Refresh(GetItemRect(i).InflatedVert(ex));
		SyncInfo();
	}
}

void RowList::SetCursor0(int c, bool sel)
{
	int c0 = cursor;
	c = minmax(c, 0, GetCount() - 1);
	if(c < 0 || cursor < 0)
		Refresh();
	else
		RefreshCursor();
	cursor = c;
	int q = sb;
	sb.ScrollInto(cursor/ncl);
	if(q != sb)
		Refresh();
	else
		RefreshCursor();
	if(sel && multi) {
		ClearSelection();
		if(cursor >= 0) {
			SelectOne(cursor, true);
			anchor = cursor;
		}
	}
	if(c0 != cursor) {
		if(cursor >= 0)
			WhenEnterItem();
		else
			WhenKillCursor();
		WhenSel();
	}
	SyncInfo();
	Action();
}

void RowList::SetCursor(int c)
{
	SetCursor0(c, true);
}

void RowList::SetSbPos(int y)
{
	SetSb();
	sb = minmax(y/cy, 0, GetCount()/ncl - GetColumnItems());
}

void RowList::KillCursor()
{
	if(cursor < 0) return;
	Refresh();
	SyncInfo();
	cursor = -1;
	sb.Begin();
	WhenKillCursor();
	WhenSel();
	Action();
}

void RowList::GotFocus()
{
	if(cursor < 0 && GetCount())
		SetCursor(sb);
	Refresh();
	SyncInfo();
}

void RowList::LostFocus()
{
	Refresh();
	SyncInfo();
}

int RowList::RoundedCy()
{
	Rect r = GetRect();
	Rect rr = r;
	frame->FrameLayout(r);
	return r.Height() / cy * cy + rr.Height() - r.Height();
}

void RowList::FrameLayout(Rect& r)
{
	r.bottom = r.top + RoundedCy();
	frame->FrameLayout(r);
}

void RowList::FrameAddSize(Size& sz)
{
	frame->FrameAddSize(sz);
}

void RowList::FramePaint(Draw& draw, const Rect& r)
{
	int ry = RoundedCy();
	Rect rr = r;
	rr.bottom = ry;
	frame->FramePaint(draw, rr);
	rr = r;
	rr.top = ry;
	draw.DrawRect(rr, SColorFace);
}

RowList& RowList::RoundSize(bool b)
{
	if(b)
		Ctrl::SetFrame(0, *this);
	else
		Ctrl::SetFrame(0, *frame);
	RefreshLayout();
	Refresh();
	SyncInfo();
	return *this;
}

void RowList::SetFrame(CtrlFrame& _frame)
{
	frame = &_frame;
	RefreshLayout();
	Refresh();
	SyncInfo();
}

struct RowList::ItemOrder {
	const ValueOrder *order;
	bool operator()(const Item& a, const Item& b) const {
		return (*order)(a.value, b.value);
	}
};

void RowList::Sort(const ValueOrder& order)
{
	ItemOrder itemorder;
	itemorder.order = &order;
	KillCursor();
	UPP::Sort(item, itemorder);
	sb.Begin();
	SyncInfo();
}

void RowList::Clear() {
	CancelMode();
	KillCursor();
	item.Clear();
	selcount = 0;
	Update();
	Refresh();
	SyncInfo();
	SetSb();
}

void RowList::Insert(int ii, const Value& val, bool canselect)
{
	int c = -1;
	if(cursor >= ii) {
		c = cursor + 1;
		KillCursor();
	}
	Item& m = item.Insert(ii);
	m.value = val;
	m.sel = false;
	m.canselect = canselect;
	m.display = NULL;
	Refresh();
	SyncInfo();
	SetSb();
	if(c >= 0)
		SetCursor(c);
}

void RowList::Insert(int ii, const Value& val, const Display& display, bool canselect)
{
	Insert(ii, val, canselect);
	item[ii].display = &display;
	SyncInfo();
}

void RowList::Set(int ii, const Value& val, bool canselect)
{
	Item& m = item[ii];
	m.value = val;
	m.sel = false;
	m.canselect = canselect;
	m.display = NULL;
	RefreshItem(ii);
	SyncInfo();
	SetSb();
}

void RowList::Set(int ii, const Value& val, const Display& display, bool canselect)
{
	Set(ii, val, canselect);
	item[ii].display = &display;
	SyncInfo();
}

void RowList::Remove(int ii)
{
	int c = -1;
	if(cursor >= ii) {
		c = max(ii, cursor - 1);
		KillCursor();
	}
	item.Remove(ii);
	Refresh();
	SyncInfo();
	SetSb();
	if(c >= 0)
		SetCursor(c);
}

void RowList::Add(const Value& val, bool canselect)
{
	Insert(item.GetCount(), val, canselect);
}

void RowList::Add(const Value& val, const Display& display, bool canselect)
{
	Add(val, canselect);
	item.Top().display = &display;
}

void RowList::DnD(int _drop, bool _insert)
{
	if(dropitem != _drop || insert != _insert) {
		RefreshItem(dropitem - 1);
		RefreshItem(dropitem);
		dropitem = _drop;
		insert = _insert;
		RefreshItem(dropitem - 1);
		RefreshItem(dropitem);
	}
}

bool RowList::DnDInsert(int i, int py, PasteClip& d, int q)
{
	Rect r = GetItemRect(i);
	int cy = r.GetHeight();
	if(py < r.top + cy / q) {
		WhenDropInsert(i, d);
		if(d.IsAccepted()) {
			DnD(i, true);
			return true;
		}
	}
	if(py >= r.top + (q - 1) * cy / q && i < GetCount()) {
		WhenDropInsert(i + 1, d);
		if(d.IsAccepted()) {
			DnD(i + 1, true);
			return true;
		}
	}
	return false;
}

void RowList::DragAndDrop(Point p, PasteClip& d)
{
	int i = GetItem(p);
	if(i >= 0) {
		if(DnDInsert(i, p.y, d, 4))
			return;
		WhenDropItem(i, d);
		if(d.IsAccepted()) {
			DnD(i, false);
			return;
		}
		if(DnDInsert(i, p.y, d, 2))
			return;
	}
	if(GetCount() == 0 && p.y < 4 || !WhenDrop) {
		WhenDropInsert(GetCount(), d);
		if(d.IsAccepted()) {
			DnD(GetCount(), true);
			return;
		}
	}
	WhenDrop(d);
	DnD(-1, false);
}

void RowList::DragRepeat(Point p)
{
	sb = sb + GetDragScroll(this, p, 1).y;
}

void RowList::DragEnter()
{
	RefreshSel();
}

void RowList::DragLeave()
{
	DnD(-1, false);
	RefreshSel();
}

void RowList::RemoveSelection()
{
	for(int i = GetCount() - 1; i >= 0; i--)
		if(IsSel(i))
			Remove(i); // Optimize!
}

void RowList::InsertDrop(int ii, const Vector<Value>& data, PasteClip& d, bool self)
{
	if(data.GetCount() == 0)
		return;
	if(d.GetAction() == DND_MOVE && self) {
		for(int i = GetCount() - 1; i >= 0; i--) {
			if(IsSel(i)) {
				Remove(i); // Optimize!
				if(i < ii)
					ii--;
			}
		}
		KillCursor();
		d.SetAction(DND_COPY);
	}
	item.InsertN(ii, data.GetCount());
	for(int i = 0; i < data.GetCount(); i++) {
		Item& m = item[ii + i];
		m.value = data[i];
		m.sel = false;
		m.canselect = true;
		m.display = NULL;
	}
	Refresh();
	SyncInfo();
	SetSb();
	ClearSelection();
	SetCursor(ii + data.GetCount() - 1);
	if(IsMultiSelect())
		for(int i = 0; i < data.GetCount(); i++)
			SelectOne(ii + i, true);
}

void RowList::InsertDrop(int ii, const RowList& src, PasteClip& d)
{
	Vector<Value> data;
	for(int i = 0; i < src.GetCount(); i++)
		if(src.IsSel(i))
			data.Add(src[i]);
	InsertDrop(ii, data, d, &src == this);
}

void RowList::InsertDrop(int ii, PasteClip& d)
{
	InsertDrop(ii, GetInternal<RowList>(d), d);
}

void RowList::Serialize(Stream& s) {
	int version = 0;
	s / version;
	s / ncl;
	Refresh();
	SyncInfo();
}

RowList::RowList() {
	clickkill = false;
	ncl = 1;
	cy = Draw::GetStdFontCy();
	cx = 50;
	cursor = -1;
	AddFrame(sb);
	sb.WhenScroll = THISBACK(Scroll);
	sb.AutoHide();
	Ctrl::SetFrame(ViewFrame());
	frame = &ViewFrame();
	RoundSize();
	display = &StdDisplay();
	multi = false;
	selcount = 0;
	nobg = false;
	popupex = true;
}