#include "Draw.h"

void Crop(RasterEncoder& tgt, Raster& img, const Rect& rc)
{
	Rect r = rc & img.GetSize();
	tgt.Create(r.Size(), img);
	for(int y = r.top; y < r.bottom; y++)
		tgt.WriteLine(~img[y] + r.left);
}

Image Crop(const Image& img, const Rect& rc)
{
	if(rc.left == 0 && rc.top == 0 && rc.Size() == img.GetSize())
		return img;
	if((rc & img.GetSize()).IsEmpty())
		return Image();
	ImageRaster  src(img);
	ImageEncoder tgt;
	Crop(tgt, src, rc);
	return tgt;
}

Image Crop(const Image& img, int x, int y, int cx, int cy)
{
	return Crop(img, RectC(x, y, cx, cy));
}

void CanvasSize(RasterEncoder& tgt, Raster& img, int cx, int cy)
{
	tgt.Create(cx, cy, img);
	int ccx = min(img.GetWidth(), cx);
	int ccy = min(img.GetHeight(), cy);
	for(int y = 0; y < ccy; y++) {
		memcpy(~tgt, img[y], ccx * sizeof(RGBA));
		memset(~tgt + ccx, 0, (cx - ccx) * sizeof(RGBA));
		tgt.WriteLine();
	}
	for(int y = cy - ccy; --y >= 0;) {
		memset(~tgt, 0, cx * sizeof(RGBA));
		tgt.WriteLine();
	}
}

Image CanvasSize(const Image& img, int cx, int cy)
{
	ImageRaster  src(img);
	ImageEncoder tgt;
	CanvasSize(tgt, src, cx, cy);
	return tgt;
}

Image AssignAlpha(const Image& img, const Image& alpha)
{
	Size sz = Size(min(img.GetWidth(), alpha.GetWidth()),
	               min(img.GetHeight(), alpha.GetHeight()));
	if(sz.cx == 0 || sz.cy == 0)
		return Image();
	ImageBuffer ib(sz);
	for(int y = 0; y < sz.cy; y++) {
		const RGBA *s = img[y];
		const RGBA *e = s + sz.cx;
		const RGBA *a = alpha[y];
		RGBA *t = ib[y];
		while(s < e) {
			*t = *s++;
			(t++)->a = (a++)->a;
		}
	}
	return ib;
}

Image Etched(const Image& img)
{
	if(!img)
		return Image();
	Size sz = img.GetSize();
	ImageBuffer b(sz);
	memset(~b, 0, sizeof(RGBA) * b.GetLength());
	for(int y = 0; y < sz.cy - 1; y++) {
		RGBA *gt = b[y];
		RGBA *wt = b[y + 1];
		const RGBA *s = img[y];
		const RGBA *e = s + sz.cx - 1;
		while(s < e) {
			if(s->a == 255 && s->r + s->g + s->b < 400) {
				gt->a = 255;
				gt->r = gt->g = gt->b = 127;
				wt->a = 255;
				wt->r = wt->g = wt->b = 255;
			}
			s++;
			gt++;
			wt++;
		}
	}
	return b;
}

int   EqualightCh(int c, int l, int h)
{
	return Saturate255((c - l) * 255 / (h - l) + l);
}

Image Equalight(const Image& img, int thold)
{
	int histogram[256];
	ZeroArray(histogram);
	const RGBA *s = ~img;
	const RGBA *e = s + img.GetLength();
	while(s < e) {
		histogram[Grayscale(*s)]++;
		s++;
	}
	int n = (thold * img.GetLength()) >> 8;
	int h = 255;
	int l = 0;
	while(l < h) {
		if(n < 0)
			break;
		n -= histogram[l++];
		if(n < 0)
			break;
		n -= histogram[h--];
	}
	if(l >= h)
		return img;
	ImageBuffer w(img.GetSize());
	RGBA *t = w;
	s = ~img;
	while(s < e) {
		t->r = EqualightCh(s->r, l, h);
		t->g = EqualightCh(s->g, l, h);
		t->b = EqualightCh(s->b, l, h);
		t->a = s->a;
		s++;
		t++;
	}
	return w;
}

Image Grayscale(const Image& img)
{
	const RGBA *s = ~img;
	const RGBA *e = s + img.GetLength();
	ImageBuffer w(img.GetSize());
	RGBA *t = w;
	while(s < e) {
		int q = Grayscale(*s);
		t->r = q;
		t->g = q;
		t->b = q;
		t->a = s->a;
		t++;
		s++;
	}
	return w;
}

Image Grayscale(const Image& img, int amount)
{
	const RGBA *s = ~img;
	const RGBA *e = s + img.GetLength();
	ImageBuffer w(img.GetSize());
	RGBA *t = w;
	int na = 256 - amount;
	while(s < e) {
		int q = Grayscale(*s);
		t->r = Saturate255((amount * q + na * s->r) >> 8);
		t->g = Saturate255((amount * q + na * s->g) >> 8);
		t->b = Saturate255((amount * q + na * s->b) >> 8);
		t->a = s->a;
		t++;
		s++;
	}
	return w;
}

Image Colorize(const Image& img, Color color, int alpha)
{
	const RGBA *s = ~img;
	const RGBA *e = s + img.GetLength();
	ImageBuffer w(img.GetSize());
	RGBA *t = w;
	byte r = color.GetR();
	byte g = color.GetG();
	byte b = color.GetB();
	alpha = alpha + (alpha >> 7);
	while(s < e) {
		int ga = Grayscale(*s);
		ga = ga + (ga >> 7);
		t->r = (alpha * (((ga * r) >> 8) - s->r) >> 8) + s->r;
		t->g = (alpha * (((ga * g) >> 8) - s->g) >> 8) + s->g;
		t->b = (alpha * (((ga * b) >> 8) - s->b) >> 8) + s->b;
		t->a = s->a;
		t++;
		s++;
	}
	return w;
}

inline
byte ContrastCh(int amount, int ch)
{
	return Saturate255(128 + (amount * (ch - 128) >> 8));
}

Image Contrast(const Image& img, int amount)
{
	const RGBA *s = ~img;
	const RGBA *e = s + img.GetLength();
	ImageBuffer w(img.GetSize());
	RGBA *t = w;
	amount += 256;
	int na = 256 - amount;
	while(s < e) {
		t->r = ContrastCh(amount, s->r);
		t->g = ContrastCh(amount, s->g);
		t->b = ContrastCh(amount, s->b);
		t->a = s->a;
		t++;
		s++;
	}
	return w;
}

struct RGBAI {
	int r, g, b, a;

	RGBAI() { r = g = b = a= 0; }
};

static void sGetS(RGBA q, RGBAI& p, int mul)
{
	p.r += mul * q.r;
	p.g += mul * q.g;
	p.b += mul * q.b;
	p.a += mul * q.a;
}

static void sSetP(RGBA& t, const RGBA& s, const RGBAI& q, int amount)
{
	int na = 256 + amount;
	t.b = Saturate255((na * (s.b << 6) - amount * q.b) >> 14);
	t.g = Saturate255((na * (s.g << 6) - amount * q.g) >> 14);
	t.r = Saturate255((na * (s.r << 6) - amount * q.r) >> 14);
	t.a = Saturate255((na * (s.a << 6) - amount * q.a) >> 14);
}

void SharpenLine(RGBA *line, int cx, const RasterLine l[3], int amount)
{
	if(cx > 1) {
		RGBAI q;
		sGetS(l[0][0], q, 7);
		sGetS(l[0][0], q, 9);
		sGetS(l[0][1], q, 7);
		sGetS(l[1][0], q, 9);
		sGetS(l[1][1], q, 9);
		sGetS(l[2][0], q, 7);
		sGetS(l[2][0], q, 9);
		sGetS(l[2][1], q, 7);
		sSetP(line[0], l[1][0], q, amount);
	}
	for(int i = 1; i < cx - 1; i++) {
		RGBAI q;
		sGetS(l[0][i - 1], q, 7);
		sGetS(l[0][i], q, 9);
		sGetS(l[0][i + 1], q, 7);
		sGetS(l[1][i - 1], q, 9);
		sGetS(l[1][i + 1], q, 9);
		sGetS(l[2][i - 1], q, 7);
		sGetS(l[2][i], q, 9);
		sGetS(l[2][i + 1], q, 7);
		sSetP(line[i], l[1][i], q, amount);
	}
	if(cx > 1) {
		RGBAI q;
		sGetS(l[0][cx - 2], q, 7);
		sGetS(l[0][cx - 1], q, 9);
		sGetS(l[0][cx - 1], q, 7);
		sGetS(l[1][cx - 2], q, 9);
		sGetS(l[1][cx - 1], q, 9);
		sGetS(l[2][cx - 2], q, 7);
		sGetS(l[2][cx - 1], q, 9);
		sGetS(l[2][cx - 1], q, 7);
		sSetP(line[cx - 1], l[1][cx - 1], q, amount);
	}
}

void Sharpen(RasterEncoder& target, Raster& src, int amount)
{
	Size sz = src.GetSize();
	target.Create(sz, src);
	if(sz.cx < 2) return;
	RasterLine l[3];
	l[0] = src[0];
	l[1] = src[0];
	l[2] = src[1];
	SharpenLine(target, sz.cx, l, amount);
	target.WriteLine();
	for(int y = 1; y < sz.cy - 1; y++) {
		l[0] = l[1];
		l[1] = l[2];
		l[2] = src[y + 1];
		SharpenLine(target, sz.cx, l, amount);
		target.WriteLine();
	}
	l[0] = l[1];
	l[1] = l[2];
	l[2] = src[sz.cy - 1];
	SharpenLine(target, sz.cx, l, amount);
	target.WriteLine();
}

Image Sharpen(const Image& img, int amount)
{
	ImageEncoder tgt;
	ImageRaster src(img);
	Sharpen(tgt, src, amount);
	return tgt;
}

Image SharpRescale(const Image& img, int cx, int cy)
{
	return img.GetSize() == Size(cx, cy) ? img : Sharpen(Rescale(img, cx, cy), 160);
}

Image SetColor(const Image& img, Color c)
{
	byte r = c.GetR();
	byte b = c.GetB();
	byte g = c.GetG();
	const RGBA *s = ~img;
	const RGBA *e = s + img.GetLength();
	ImageBuffer w(img.GetSize());
	RGBA *t = w;
	while(s < e) {
		t->r = r;
		t->g = g;
		t->b = b;
		t->a = s->a;
		t++;
		s++;
	}
	return w;
}

Image Over(const Image& img, const Image& over, byte alpha)
{//Improve!
	Size sz = img.GetSize();
	if(sz != over.GetSize())
		return Image();
	int n = img.GetLength();
	ImageBuffer b(img.GetSize());
	memcpy(~b, img, sizeof(RGBA) * n);
	AlphaBlend(b, ~over, n, alpha);
	return b;
}

Image Emboss(const Image& img, Color color, int amount)
{
	Size sz = img.GetSize();
	ImageBuffer b(sz);
	RGBA c = RGBAColor(color);
	Fill(~b, RGBAZero(), sz.cx);
	int na = 256 - amount;
	for(int y = 1; y < sz.cy; y++) {
		RGBA *t = b[y];
		const RGBA *s = img[y];
		const RGBA *e = s + sz.cx;
		*t = c;
		t->a = s->a;
		t++;
		s++;
		while(s < e) {
			int q = amount * (Grayscale(*s) - Grayscale(*(s - sz.cx - 1)));
			t->a = s->a;
			t->r = Saturate255(c.r + ((q * c.r) >> 16));
			t->g = Saturate255(c.g + ((q * c.g) >> 16));
			t->b = Saturate255(c.b + ((q * c.b) >> 16));
			t++;
			s++;
		}
	}
	return b;
}

Image EmbossTh(const Image& img, Color color, Color colorl, Color colord, int thold)
{
	Size sz = img.GetSize();
	ImageBuffer b(sz);
	RGBA c = RGBAColor(color);
	RGBA cl = RGBAColor(colorl);
	RGBA cd = RGBAColor(colord);
	FillColor(~b, c, sz.cx);
	for(int y = 1; y < sz.cy; y++) {
		RGBA *t = b[y];
		const RGBA *s = img[y];
		const RGBA *e = s + sz.cx;
		*t = c;
		t->a = s->a;
		t++;
		s++;
		while(s < e) {
			int q = Grayscale(*s) - Grayscale(*(s - sz.cx - 1));
			byte a = s->a;
			*t = q > thold ? cl : q < -thold ? cd : c;
			t->a = a;
			t++;
			s++;
		}
	}
	return b;
}
