#include <Core/Core.h>
#include "FireBird.h"

NAMESPACE_UPP

#ifdef PLATFORM_WIN32
#define DLLFILENAME "fbclient.dll"
#elif defined(PLATFORM_POSIX)
#define DLLFILENAME "libfbclient.so.2"
#endif
#define DLIMODULE   FBCLIENT
#define DLIHEADER   <plugin/firebird/FireBird.dli>
#include <Core/dli_source.h>

const char *FirebirdReadString(const char *s, String& stmt) {
	stmt.Cat(*s);
	int c = *s++;
	for(;;) {
		if(*s == '\0') break;
		else
		if(*s == '\'' && s[1] == '\'') {
			stmt.Cat('\'');
			s += 2;
		}
		else
		if(*s == c) {
			stmt.Cat(c);
			s++;
			break;
		}
		else
		if(*s == '\\') {
			stmt.Cat('\\');
			if(*++s)
				stmt.Cat(*s++);
		}
		else
			stmt.Cat(*s++);
	}
	return s;
}

class FireBirdConnection : public SqlConnection {
protected:
	virtual void SetParam(int i, const Value& r);
	virtual SqlSession& GetSession() const;
	virtual void Cancel();
	virtual bool Execute();
	virtual bool Fetch();
	virtual void GetColumn(int i, Ref f) const;	
	virtual String ToString() const;

private:
	FireBirdSession& session;
	isc_db_handle *DB;
	Vector<String> param;
	isc_tr_handle trans;
	XSQLDA *sqlda; /* Row data */
	ISC_STATUS_ARRAY status; /* status vector */	
    isc_stmt_handle stmt; /* statement handle */
    int num_cols; /* Number of columns */
    void clearData();

public:
	FireBirdConnection(FireBirdSession& session, isc_db_handle *DB);
	virtual ~FireBirdConnection();
	String GetUser() const;

public:
	T_FBCLIENT& fbclient;
};

/* --------------------------------------- */
/* FireBirdConnection Class                */
/* --------------------------------------- */

/* Constructor */
FireBirdConnection::FireBirdConnection(FireBirdSession& p_session, isc_db_handle *p_DB)
: session(p_session), DB(p_DB), sqlda(NULL), stmt(NULL), num_cols(0), fbclient(p_session.fbclient)
{	
}

/* Desctructor */
FireBirdConnection::~FireBirdConnection()
{
	Cancel();
}

/* Clear data */
void FireBirdConnection::clearData()
{
	/* We need to free sqlda data allocated */
	if (sqlda)
	{
		XSQLVAR *var;
		int i;
		
		for (var = sqlda->sqlvar, i = 0; i < num_cols; var++, i++)
		{
			if (var->sqltype & ~1)
			{
				delete var->sqldata;
				var->sqldata = NULL;
			}
			if (var->sqltype & 1)
			{
				delete var->sqlind;	
				var->sqlind = NULL;
			}
		}
		free ((XSQLDA *) sqlda);
		sqlda = NULL;
		num_cols = 0;
	}
}

/* Cancel */
void FireBirdConnection::Cancel()
{
	param.Clear();
	info.Clear();

	if (stmt)
	{
		if (fbclient.isc_dsql_free_statement(status, &stmt, DSQL_close))
    	{
			return;
    	}
    	stmt = NULL;
	}
	
	clearData();
}

/* Return session */
SqlSession& FireBirdConnection::GetSession() const
{
	return session;
}

/* Set Parameter */
void FireBirdConnection::SetParam(int i, const Value& r)
{
}

/* Execute statement */
bool FireBirdConnection::Execute()
{
	bool need_commit = false; /* If trans was not started, we need to commit */
	static char stmt_info[] = { isc_info_sql_stmt_type };
	char info_buffer[20];
	short l;
	int statement_type;
	XSQLVAR *var;
	int i;
	
	if(statement.GetLength() == 0) {
		session.SetError("Empty statement", statement);
		return false;
	}
	String query;
	int pi = 0;
	const char *s = statement;
	while(s < statement.End())
		if(*s == '\'' || *s == '\"')
			s = FirebirdReadString(s, query);
		else {
			if(*s == '?')
				query.Cat(param[pi++]);
			else
				query.Cat(*s);
			s++;
		}
	param.Clear();

	Stream *trace = session.GetTrace();
	dword time;
	if(session.IsTraceTime())
		time = GetTickCount();
	
	clearData();

	if (stmt)
	{
		if (fbclient.isc_dsql_free_statement(status, &stmt, DSQL_close))
    	{
			return false;
    	}
    	stmt = NULL;
	}
	
	/* Allocate SQLDA for first time */
	sqlda = (XSQLDA *) malloc(XSQLDA_LENGTH(10));
    sqlda->sqln = 10;
    sqlda->version = 1;

	/* Start transaction if it's not started */
	trans = session.getTrans();
	if (!trans)
	{
		session.Begin();
		trans = session.getTrans();
		need_commit = true;
	}
	else
	{
		/* If last stmt was select with own trans, we need to roolback */
		if ((session.GetTransType()) == TRANS_FETCH)
		{
			session.Rollback();
			session.Begin();
			trans = session.getTrans();
			need_commit = true;
		}
		else
		{
			need_commit = false;
		}
	}

	/* Allocate statement */
	fbclient.isc_dsql_allocate_statement(status, DB, &stmt);
	if (status[0] == 1 && status[1])
	{
		return false;
	}
	
	/* Prepare statement */
	fbclient.isc_dsql_prepare(status, &trans, &stmt, 0, query, session.GetSqlDialect(), sqlda);
	if (status[0] == 1 && status[1])
	{
		return false;
	}

	/* Identitify statement type */
	if (!fbclient.isc_dsql_sql_info(status, &stmt, sizeof (stmt_info), stmt_info,
        sizeof (info_buffer), info_buffer))
    {
        l = (short) fbclient.isc_vax_integer((char *) info_buffer + 1, 2);
        statement_type = fbclient.isc_vax_integer((char *) info_buffer + 3, l);
    }
	
	session.SetLastQueryType(statement_type);
	
    if (!sqlda->sqld)
    {			
		/* Execute non-select statement */
		session.SetTransType(TRANS_NORMAL);
		if (fbclient.isc_dsql_execute(status, &trans, &stmt, session.GetSqlDialect(), NULL))
        {
            if (need_commit)
            {
                session.Rollback();
            }
            return false;
        }

        /* Commit insert, update, delete and DDL statements if that is what sql_info says */
        if (trans && ((statement_type == isc_info_sql_stmt_insert) ||
        	(statement_type == isc_info_sql_stmt_update) ||
        	(statement_type == isc_info_sql_stmt_delete) ||
        	(statement_type == isc_info_sql_stmt_ddl)))
        {
            /* LOG ("\tCommitting...\n"); */
            if (need_commit)
            {
                session.Commit();
            }
            
            if (stmt)
			{
				if (fbclient.isc_dsql_free_statement(status, &stmt, DSQL_close))
		    	{
					return false;
		    	}
		    	stmt = NULL;
			}            
            return true;
        }
        return false;
	}
	
	/* Inform that last query was select */
	if (need_commit)
	{
		session.SetTransType(TRANS_FETCH);
	}
	else
	{
		session.SetTransType(TRANS_NORMAL);
	}
	
	/* Re-allocate SQLDA */
	num_cols = sqlda->sqld;
	
	if (sqlda->sqln < num_cols)
    {
        sqlda = (XSQLDA *) realloc(sqlda, XSQLDA_LENGTH (num_cols));
        sqlda->sqln = num_cols;
        sqlda->version = 1;

        if (fbclient.isc_dsql_describe(status, &stmt, session.GetSqlDialect(), sqlda))
        {
            return false;
        }        
    }

	/* Get column definitions */
    /* And allocate space for data */	
    info.SetCount(num_cols);
    for (var = sqlda->sqlvar, i = 0; i < num_cols; var++, i++)
    {
        SqlColumnInfo& f = info[i];
        f.name = var->sqlname;
        
		switch (var->sqltype & ~1)
		{
			case SQL_SHORT :
				{
					var->sqldata = (char*) new short(0);
					f.type = INT_V;
				}
				break;
			case SQL_LONG :
				{
					var->sqldata = (char*) new int(0);
					f.type = INT_V;
            		break; 
				}
				break;
			case SQL_FLOAT : 
				{
					var->sqldata = (char*) new float(0.0);
					f.type = DOUBLE_V;
				}
				break;
			case SQL_DOUBLE :
				{
					var->sqldata = (char*) new double(0.0);
					f.type = DOUBLE_V;
				}
				break;				
			case SQL_INT64 :
				{
					var->sqldata = (char*) new int64(0);
					f.type = INT64_V;
				}
				break;
			case SQL_ARRAY :
			case SQL_BLOB :
				{
					var->sqldata = (char*) new ISC_QUAD;
					memset(var->sqldata, 0, sizeof(ISC_QUAD));
					f.type = VALUEARRAY_V;
				}
				break;
			case SQL_TYPE_DATE :
				{
					var->sqldata = (char*) new ISC_DATE;
					memset(var->sqldata, 0, sizeof(ISC_DATE));
					f.type = DATE_V;
				}
				break;
			case SQL_TIMESTAMP :
				{
					var->sqldata = (char*) new ISC_TIMESTAMP;
					memset(var->sqldata, 0, sizeof(ISC_TIMESTAMP));
					f.type = TIME_V;
				}
				break;
			case SQL_TYPE_TIME :
				{
					var->sqldata = (char*) new ISC_TIME;
					memset(var->sqldata, 0, sizeof(ISC_TIME));
					f.type = TIME_V;
				}
				break;
			case SQL_TEXT :
				{
					var->sqldata = new char[var->sqllen+1];
					memset(var->sqldata, ' ', var->sqllen);
					var->sqldata[var->sqllen] = '\0';
					f.type = STRING_V;
				}
				break;
			case SQL_VARYING :
				{
					var->sqldata = new char[var->sqllen+3];
					memset(var->sqldata, 0, 2);
					memset(var->sqldata+2, ' ', var->sqllen);
					var->sqldata[var->sqllen+2] = '\0';
					f.type = STRING_V;
				}
				break;
		}
		if (var->sqltype & 1) var->sqlind = new short(-1);	// 0 indicator		
    }

	/* Execute select statement */
	if (fbclient.isc_dsql_execute(status, &trans, &stmt, session.GetSqlDialect(), NULL))
    {
        return false;
    }
    
    return true;
}

/* Fetch next row if exists */
bool FireBirdConnection::Fetch()
{
	if (!fbclient.isc_dsql_fetch(status, &stmt, session.GetSqlDialect(), sqlda))
	{
		return true;
	}
	else
	{
		return false;
	}
}

/* Get column data */
void FireBirdConnection::GetColumn(int i, Ref f) const
{
	XSQLVAR *var;
	
	var = sqlda->sqlvar;
	const char *s = var[i].sqldata;
	if(s == NULL)
		f = Null;
	else {
		switch(info[i].type)
		{
			case INT_V:
				{
					switch(var[i].sqltype & ~1)
					{
						case SQL_SHORT:
							{
								short value = 0;
								memcpy(&value, s, sizeof(short));
								f.SetValue(value);
							}
							break;
						default:
							{
								int value = 0;
								memcpy(&value, s, sizeof(int));
								f.SetValue(value);
							}
							break;
					}
				}
				break;
			case DOUBLE_V:
				{
					switch(var[i].sqltype & ~1)
					{
						case SQL_FLOAT:
							{
								float value = 0;
								memcpy(&value, s, sizeof(float));
								f.SetValue(value);
							}
							break;
						default:
							{
								double value = 0;
								memcpy(&value, s, sizeof(double));
								f.SetValue(value);
							}
							break;
					}
				}
				break;
			case INT64_V:
				{
					int64 value = 0;
					memcpy(&value, s, sizeof(int64));
					f.SetValue(value);
				}
				break;
			case DATE_V:
				{
					switch(var[i].sqltype & ~1)
					{
						case SQL_TIMESTAMP:
							{
								struct tm tm_value;
								Time dt_value;
								fbclient.isc_decode_timestamp((ISC_TIMESTAMP *)s, &tm_value);
								dt_value.year = tm_value.tm_year + 1900;
								dt_value.month = tm_value.tm_mon + 1;
								dt_value.day = tm_value.tm_mday;
								dt_value.hour = tm_value.tm_hour;
								dt_value.minute = tm_value.tm_min;
								dt_value.second = tm_value.tm_sec;
								f = Value(dt_value);							
							}
							break;
						default:
							{
								struct tm tm_value;
								Date dt_value;
								fbclient.isc_decode_sql_date((ISC_DATE *)s, &tm_value);
								dt_value.year = tm_value.tm_year + 1900;
								dt_value.month = tm_value.tm_mon + 1;
								dt_value.day = tm_value.tm_mday;
								f = Value(dt_value);
							}
							break;
					}
				}
				break;
			case TIME_V:
				{
					struct tm tm_value;
					Time dt_value;
					fbclient.isc_decode_sql_time((ISC_TIME *)s, &tm_value);
					dt_value.year = 0;
					dt_value.month = 0;
					dt_value.day = 0;
					dt_value.hour = tm_value.tm_hour;
					dt_value.minute = tm_value.tm_min;
					dt_value.second = tm_value.tm_sec;
					f = Value(dt_value);
				}
				break;
			case STRING_V:
				switch(var[i].sqltype & ~1)
				{
					case SQL_VARYING:
						{
							f = Value(String(s + 2, (short)*(int *)s));
						}
						break;
					default:
						{
							f = Value(String(s, var[i].sqllen));
						}
						break;
				}
				break;
			default:
				f = Value(String(s, var[i].sqllen));
				break;
		}
	}
}

/* Get user */
String FireBirdConnection::GetUser() const
{
	return session.GetUser();
}

/*  ToString() */
String FireBirdConnection::ToString() const
{
	return statement;
}

/* --------------------------------------- */
/* FireBirdSession Class                */
/* --------------------------------------- */

/* Constructor */
FireBirdSession::FireBirdSession()
{
	DB = new isc_db_handle;
	trans = NULL;
	SQL_DIALECT = SQL_DIALECT_CURRENT; /* SQL_DIALECT_V6 in actual version */
	trans_type = TRANS_NORMAL;
	Dialect(FIREBIRD);
}

/* Destructor */
FireBirdSession::~FireBirdSession()
{
	delete DB;
}

/* Open database sesion */
bool FireBirdSession::Open(const char *connect)
{		
	String connstring;
	String dbfullname;
	
	Vector<String> connprop = Split(String(connect), ';');
	servername = connprop[0];
	dbname = connprop[1];
	username = connprop[2];
	password = connprop[3];
	
	connstring << (char) isc_dpb_version1
		<< (char) isc_dpb_user_name
		<< (char) strlen(username)
		<< username
		<< (char) isc_dpb_password
		<< (char) strlen(password)
		<< password;
	
	dbfullname << servername << ":" << dbname;
	
	*DB = NULL;
	trans = NULL;
	
	fbclient.Load();
	
	fbclient.isc_attach_database(status,
		strlen(dbfullname), dbfullname, DB, strlen(connstring), connstring);
	
	if (status[0] == 1 && status[1])
	{
		return false;
	}
	else
	{
		return true;
	}
}

/* Close database sesion */
void FireBirdSession::Close()
{
	if (DB)
	{
		fbclient.isc_detach_database(status, DB);
		*DB = NULL;
	}
}

/*  Create connection */
SqlConnection * FireBirdSession::CreateConnection()
{
	return new FireBirdConnection(*this, DB);
}

/* Begin transaction */
void FireBirdSession::Begin()
{
	if (!trans)
	{
		fbclient.isc_start_transaction(status, &trans, 1, DB, 0, NULL);
		SetTransType(TRANS_NORMAL);
	}
	else if (GetTransType() == TRANS_FETCH)
		{
			Rollback();
			fbclient.isc_start_transaction(status, &trans, 1, DB, 0, NULL);
			SetTransType(TRANS_NORMAL);
		}
}

/* Commit transaction */
void FireBirdSession::Commit()
{
	if (trans)
	{
		fbclient.isc_commit_transaction (status, &trans);
		SetTransType(TRANS_NORMAL);
		trans = NULL;
	}
}

/* Rollback transaction */
void FireBirdSession::Rollback()
{
	if (trans)
	{	
		fbclient.isc_rollback_transaction (status, &trans);
		SetTransType(TRANS_NORMAL);
		trans = NULL;
	}
}

/* Fetch list */
static Vector<String> FetchList(Sql& cursor, bool upper = false)
{
	Vector<String> out;
	String s;
	while(cursor.Fetch(s))
		out.Add(upper ? ToUpper(s) : s);
	return out;
}

/* Get tables list */
Vector<String> FireBirdSession::EnumTables(String database)
{
	Vector<String> out;
	Sql cursor(*this);
	if(cursor.Execute("SELECT RDB$RELATION_NAME FROM RDB$RELATIONS WHERE RDB$SYSTEM_FLAG = 0;"))
		out = FetchList(cursor); // 06-09-12 cxl: was false; In Linux, names are case sensitive
	return out;
}

END_UPP_NAMESPACE