#include "UppLogging.hpp"
#include <Core/Core.h>
#include <plugin/bz2/bz2.h>
using namespace Upp;

namespace UppLog{
	
	Ptr<Logger> Logger::_logger = 0;
	
	Logger::Logger() : _logs_directory(String().Cat() << GetFileDirectory(GetExeFilePath()) << "logs"),
					   _incr_offset(Time(0,0,1,0,0,0)), _use_timestamp(true),					   
					   _release_log_title(""), _debug_log_title("") {} //_release_log(""), _debug_log(""),
	
	Ptr<Logger> Logger::Instance(){
		static Logger _log_inst;
		_logger = &_log_inst;	
		return _logger;	
	}
	
	//begin debug/release logs with default options.
	void Logger::BeginLogs(){
		#ifdef flagDEBUG
		BeginDebug();
		#endif	
		BeginRelease();
	}
	
	void Logger::BeginRelease(String release_log_title, String release_log_dir){
		_release_log_title = release_log_title.IsEmpty() ? (String().Cat() << GetExeTitle() << ".release") : release_log_title;
		release_log_dir = release_log_dir.IsEmpty() ? _logs_directory : release_log_dir;		
		CreateLog(_release_log_title, release_log_dir);
	}
	
	void Logger::BeginDebug(String debug_log_title, String debug_log_dir){
		_debug_log_title = debug_log_title.IsEmpty() ? (String().Cat() << GetExeTitle() << ".debug") : debug_log_title;
		debug_log_dir = debug_log_dir.IsEmpty() ? _logs_directory : debug_log_dir;
		CreateLog(_debug_log_title, debug_log_dir);
	}
	
	void Logger::CreateLog(String id, String log_file_dir){
		if(id.IsEmpty() || _custom_logs.Find(id)>=0){return;} // already exists, do nothing		
		
		log_file_dir = log_file_dir.IsEmpty() ? _logs_directory : log_file_dir;
		String log_file_path = "";
		log_file_path.Cat() << log_file_dir << "/";
		log_file_path.Cat( GetLogFileName( id ) );
		EndLog(id);
		
		RealizeDirectory(log_file_dir);
		CompressLogFiles(log_file_dir, id);
		PrintLogFileStart(log_file_path);
		_custom_logs.Add(id, log_file_path);		
	}
	
	bool Logger::IsActive(String id){
		return _custom_logs.Find(id)>=0;
	}
	
	bool Logger::IsActive(LogType lt){
		switch(lt){
			case RELEASE:
				return IsActive(_release_log_title);
			case DEBUG:
				return IsActive(_debug_log_title);
			case BOTH:				
				return IsActive(_release_log_title) && IsActive(_debug_log_title);
		}
		return false;
	}
	
	void Logger::PrintLogFileStart(String log_file){
		if(_use_timestamp){
			FileAppend fa(log_file);
			fa.PutLine( String().Cat() << " ### Begin Log - " << AsString(GetSysTime()) << " ### " );			
			fa.Close();
		}
	}	
	
	Logger::~Logger(){
		EndLogs();	
	}
	
	void Logger::EndLogs(){
		while(_custom_logs.GetCount() > 0){
			EndLog(_custom_logs.GetKey(0));
		}		
	}
	
	void Logger::EndLog(LogType lt){
		switch(lt){
			case RELEASE:				
				EndLog(_release_log_title);	break;
			case DEBUG:
				EndLog(_debug_log_title); break;
			case BOTH:
				EndLog(_release_log_title);
				EndLog(_debug_log_title);
		}
	}
	
	void Logger::EndLog(String id){
		int pos = -1;
		if( (pos=_custom_logs.Find(id)) >=0 ){
			PrintLogFileEnd(_custom_logs.Get(id));
			_custom_logs.RemoveKey(id);
		}
	}
	
	void Logger::PrintLogFileEnd(String log_file){
		if(_use_timestamp){
			FileAppend fa(log_file);
			fa.PutLine( String().Cat() << " ### End Log - " << AsString(GetSysTime()) << " ### ");
			fa.PutLine(" ");
			fa.Close();
		}
	}
	
	void Logger::Log(String id, String const & message){
		if(_custom_logs.Find(id)>=0){
			CheckLogFile(id);
			String full_message = "";
			if(_use_timestamp){
				full_message.Cat(String().Cat() << FormatTime(GetSysTime(), GetFileIncrementFormatter()) << "   "); //"hh:mm:ss"
			}
			full_message.Cat(message);			
			FileAppend filea(_custom_logs.Get(id));							
			filea.PutLine(full_message);
			filea.Close();			
			if(_cout_custom.Get(id,false)){ Cout() << full_message << "\n"; }
		}
	}
	
	void Logger::Log(LogType lt, String const & message){
		switch(lt){
			case RELEASE:
				ReleaseLog(message); break;
			case DEBUG:
				DebugLog(message); break;
			case BOTH:
				ReleaseLog(message);
				DebugLog(message);
		}
	}
	
	void Logger::ReleaseLog(String const & message){
		if(_use_upp_rls_log){
			RLOG(message);
		}else{
			Log(_release_log_title, message);
		}
	}
	
	void Logger::DebugLog(String const & message){
		if(_use_upp_dbg_log){
			LOG(message);
		}else{
			Log(_debug_log_title, message);
		}
	}
	
	void Logger::EnableCout(LogType lt, bool tf){
		switch(lt){
			case RELEASE:
				EnableCout(_release_log_title, tf); //_cout_rls = tf;
				break;
			case DEBUG:
				EnableCout(_debug_log_title, tf); //_cout_dbg = tf;			
				break;
			case BOTH:
				EnableCout(_release_log_title, tf); //_cout_rls = tf;
				EnableCout(_debug_log_title, tf); //_cout_dbg = tf;
		}		
	}
	
	void Logger::EnableCout(String id, bool tf){
		if(_custom_logs.Find(id)>=0){
			_cout_custom.GetAdd(id)=tf;		
		}
	}
	
	void Logger::DefaultDirectory(String dir, bool create){
		if(create){RealizeDirectory(dir);}
		if(DirectoryExists(dir)){_logs_directory = dir;}
	}
	String Logger::DefaultDirectory(void){return _logs_directory;}
		
	void Logger::UseUppDefaultLog(LogType lt, bool tf){
		switch(lt){
			case RELEASE:
				_use_upp_rls_log = tf;
				break;
			case DEBUG:
				_use_upp_dbg_log = tf;
				break;
			case BOTH:
				_use_upp_rls_log = tf;
				_use_upp_dbg_log = tf;
		}	
	}
	
	Logger::Logger(const Logger&){}
	void Logger::IncrementalLogCutOff(Time t){_incr_offset=t;}	
	Time Logger::IncrementalLogCutOff(){return _incr_offset;}
	
	void Logger::CheckLogFile(LogType lt){
		switch(lt){
			case RELEASE:
				CheckLogFile(_release_log_title); break;
			case DEBUG:
				CheckLogFile(_debug_log_title); break;				
			case BOTH:
				CheckLogFile(_release_log_title);
				CheckLogFile(_debug_log_title);
		}
	}

	void Logger::CheckLogFile(String id){
		if(IsActive(id)){
			String logfile = GetLogFileName(id);
			String dir = GetFileFolder(_custom_logs.Get(id));
			if( !FileExists(dir+"/"+logfile) ){
				EndLog(id);
				CreateLog(id, dir);
			}
		}
	}
	
	String Logger::GetLogFileName(String title){
		return String().Cat() << title << "." << FormatTime(GetSysTime(),GetFileIncrementFormatter()) << ".log";
	}
	
	String Logger::GetFileIncrementFormatter(){
		String ft;		
		if(_incr_offset.second>0){ft="MM-DD-YY-hhmmss";}
		else if(_incr_offset.minute>0){ft="MM-DD-YY-hhmm";}
		else if(_incr_offset.hour>0){ft="MM-DD-YY-hh";}
		else if(_incr_offset.day>0){ft="MM-DD-YY";}
		else if(_incr_offset.month>0){ft="MM-YY";}
		else if(_incr_offset.year>0){ft="YY";}
		return ft;
	}	
	
	void Logger::CompressLogFiles(String log_file_path){
		String filetitle = GetFileTitle(log_file_path);
		String dir = GetFileFolder(log_file_path);
		CompressLogFiles(dir, filetitle);
	}
	
	int64 Logger::GetSeconds(Time t){
		int64 seconds = 0;
		if(t.second){seconds += t.second;}
		if(t.minute){seconds += t.minute * 60;}
		if(t.hour){seconds += t.hour * 60 * 60;}
		if(t.day){seconds += t.day * 24 * 60 * 60;}		
		if(t.month){seconds += t.month * 30 * 24 * 60 * 60;}
		if(t.year){seconds += t.year * 365 * 24 * 60 * 60;}
		return seconds;
	};
	
	void Logger::CompressLogFiles(String dir, String filetitle){
		Vector<String> files = FetchDir(dir, String().Cat() << "*" << filetitle << "*");
		for(int i=0; i<files.GetCount(); ++i){
			// loop through each log file seen and bzip2 it into its month-year log archive.
			// only compress the file if it is older than the specified offset.
			Time filetime = FindFile(files[i]).GetLastWriteTime();
			Time dectime;
			dectime.Set(GetSysTime().Get()-GetSeconds(_incr_offset));
			if( dectime.Get() > filetime.Get() ){ //(GetSysTime()-_incr_offset)
				String bzlogfilename = String().Cat() << filetitle << "." << Format("%Mon", filetime.month)
														<< "-" << FormatDate(filetime, "YYYY" ) << ".log.bz2";
				FileAppend fout(String().Cat() << dir << "/" << bzlogfilename);		
				FileIn fin(files[i]);
				BZ2Compress(fout, fin);
				fin.Close();
				fout.Close();
				FileDelete(files[i]);				
			}
		}		
	}
	
	Vector<String> Logger::FetchDir(String dir, String file_match){
	    FindFile ff;
	    String match;
	    Vector<String> files;    
	    match <<= (file_match.GetCount() <= 0) ? String("/*") : file_match;
	    //bool dire = DirectoryExists(dir);
	    //String s1 = AppendFileName(dir, match);
	    if( DirectoryExists(dir) && ff.Search(AppendFileName( dir, match )) ){		    
		    do{
		        if( ff.IsFile() ){
		            String file_loc( String().Cat() << dir << "/" << ff.GetName() );
		            files.Add( file_loc );	            
		        }
		    }while(ff.Next());
	    }
	    return files;
	}

}
	
	/*int64 Logger::GetSeconds(int years, int months, int days, int hours, int minutes, int secs){
		int64 seconds = secs;
		if(minutes){seconds += minutes * 60;}
		if(hours){seconds += hours * 60 * 60;}
		if(days){seconds += days * 24 * 60 * 60;}		
		if(months){seconds += months * 30 * 24 * 60 * 60;}
		if(years){seconds += years * 365 * 24 * 60 * 60;}
		return seconds;	
	}*/