#ifndef UPP_UPP_LOGGING_HPP
#define UPP_UPP_LOGGING_HPP

#include <Core/Core.h>
#include <plugin/bz2/bz2.h>
using namespace Upp;

namespace UppLog{

#define LOGGER	Logger::Instance()

/** Singleton Class for Logging 
	Debug logs rely on flagDEBUG
	--> though you can enable them all.
	All release logs are always enabled.
	--> though you can separate/disable any one of them.	
*/

enum LogType{
	RELEASE,
	DEBUG,
	BOTH
};

class Logger:public Pte<Logger>{
	private:		
		String _logs_directory;
		String _release_log_title, _debug_log_title;
		bool _use_upp_dbg_log, _use_upp_rls_log, _use_timestamp;
		Time _incr_offset;
		//int _max_file_size;
		//int _verbosity;
		VectorMap<String,String> 	_custom_logs;
		VectorMap<String,bool>		_cout_custom;
		
	protected:				
		static Ptr<Logger> _logger;		///< The Singleton Object pointer. (to be used later for possible MT)
		
		Logger();
		~Logger();
		
		void PrintLogFileStart(String log_file);
		void PrintLogFileEnd(String log_file);
		
		void CheckLogFile(LogType lt);
		void CheckLogFile(String id);
		String GetLogFileName(String title);
		String GetFileIncrementFormatter();
		
		/** Compress all .log files that are not compressed in the logs directory and that are dated later than today. 
		Keep all of today's log file in its .log file. Compressed log files have the format: 'ApplicationName.Month-YYYY.log.bz2'. */		
		void CompressLogFiles(String log_file_path);
		void CompressLogFiles(String dir, String filetitle);
		Vector<String> FetchDir(String dir, String file_match);
		
		int64 GetSeconds(Time t); /**< Generates seconds count from values in Time object. 
									   Note: only estimates seconds in year and seconds in month. */
		
	public:		
		static Ptr<Logger> Instance();		///< Get a Ptr to instance of the Logger
		
		Logger(const Logger&);
		/** Call this method to open release and debug log file handles.
		If you leave the strings blank, the path will default to exe directory and
		file name to .exe name with a .log extension.
		 */
		void BeginLogs();
		void BeginRelease(String release_log_title="", String release_log_dir="");
		void BeginDebug(String debug_log_title="", String debug_log_dir="");
		
		void EndLogs();					///< Add closing messages to log files and turn off all logging.
		void EndLog(LogType);			///< Add closing message to RELEASE/DEBUG log file(s) and turn off logging for that file.
		void EndLog(String);			///< Add closing message to CUSTOM log file and turn off logging for that file.
		
		void CreateLog(String id, String log_file_dir="");	///< Create custom log file
		
		void Log(String, String const & );		///< Log to custom log file
		void Log(LogType, String const & );		///< Log to release and/or debug log files.
		void ReleaseLog(String const &);		/**< Send text string to Release Log output file. */		
		void DebugLog(String const &);			/**< Send text string to Debug Log output file. Debug log 
											 	file is only created when running a debug build. */
		
		bool IsActive(LogType lt);		///< Check if debug/release logs are active
		bool IsActive(String id);		///< Check if custom log is active
		
		void DefaultDirectory(String,bool=true);	/**< Set logger's default directory. Will create (realize) 
													     directory if it does not exist (overridable as 2nd argument) */
		String DefaultDirectory(void);				///< Get logger's default directory.
		
		/** use LOG() macro from Upp instead of DebugLog()
			use RLOG() macro from Upp instead of ReleaseLog() */
		void UseUppDefaultLog(LogType, bool=true);
		
		void IncrementalLogCutOff(Time);			/**< automatically compress all logs before the 
													   current one according to date/time difference. */
		Time IncrementalLogCutOff();				///< get compression cut off.		
		void EnableCout(LogType, bool=true);		///< send log messages to Cout.
		void EnableCout(String, bool=true);			///< enable Cout output of custom user-defined logs.
		void UseTimeStamp(bool=true);				///< place timestamp before each log entry.
		
		//void Verbosity(int);						///< set verbosity level.
		//int  Verbosity();							///< get verbosity level
};

inline void UPPLOG(String title, String const & message){
	Logger::Instance()->Log(title,message);
};

inline void UPPLOG(LogType lt, String const & message){
	Logger::Instance()->Log(lt, message);
};

inline void UPPRLSLOG(String const & message){
	Logger::Instance()->ReleaseLog(message);
};

inline void UPPDBGLOG(String const & message){
	Logger::Instance()->DebugLog(message);
};

}
//#define UPPLOG(x,y) Logger::Instance()->Log(x,y) <-- has been replaced with inline void functions
#endif