#ifndef UPP_CMDLINEARGPROCESSOR_TARG_HPP
#define UPP_CMDLINEARGPROCESSOR_TARG_HPP

#include <Core/Core.h>

using namespace Upp;

class ArgExc : public Exc
{
	public:
		ArgExc(String error_str) : Exc(error_str){};
};

/**
Abstract base class for all arguments. Non-templated for dynamic use in Upp containers.
*/
class TArg : public Pte<TArg>, public Moveable<TArg>
{
	friend class ArgProcessor;
	protected:		
		String _short_flag, _long_flag, _title, _description;
		Value _val;
		bool _required, _requires_value;
		virtual void Val(String val)=0;
		
	public:
		TArg(String short_flag, String long_flag, String title, String description, bool required, bool requires_value) :
			_short_flag(short_flag), _long_flag(long_flag), \
			_title(title), _description(description), \
			_required(required), _requires_value(requires_value) {};
			
		TArg(String title, String description, bool required, bool requires_value) : \
			_title(title), _description(description), \
			_required(required), _requires_value(requires_value) 
		{
			SetFlagsFromTitle();
		};
		
		TArg(const TArg& arg)
		{
			_short_flag = arg._short_flag;
			_long_flag = arg._long_flag;
			_title = arg._title;
			_description = arg._description;
			_val = arg._val;
			_required = arg._required;
			_requires_value = arg._requires_value;
		};
		
		String GetTitle()		{ return _title; };
		String GetLongFlag()	{ return _long_flag; };
		String GetShortFlag()	{ return _short_flag; };
		String GetDescription() { return _description; };
		bool RequiresValue()	{ return _requires_value; };
		bool IsRequired()		{ return _required; };
		virtual bool IsSet()	{ return !this->Val().IsNull() && !this->Val().IsVoid(); };
				
		virtual Value Val(void)=0;
		
		static int CharFilterNotWhitespace(int c)
		{
			return IsSpace(c) ? 0 : c;
		};
		
		void SetFlagsFromTitle()
		{
			_short_flag = ToLower(_title.Left(1));
			_long_flag = ToLower(FilterWhile(_title, CharFilterNotWhitespace));
		};
};

template<typename T>
class Arg : public TArg
{
	protected:
		void Val(String val)	
		{
			this->_val = Value(val); 
		};
	public:
		Arg(String short_flag, String long_flag, String title, String description, bool required=false) : \
			TArg(short_flag, long_flag, title, description, required, true){};
		Arg(String title, String description, bool required=false) : \
			TArg(title, description, required, true){};		
		Value Val(void)			{ return _val; };		
};

template<> Arg<bool>::Arg(String short_flag, String long_flag, String title, String description, bool required) : \
	TArg(short_flag, long_flag, title, description, required, false){}

template<> Arg<bool>::Arg(String title, String description, bool required) : \
	TArg(title, description, required, false){}	

template<> void Arg<bool>::Val(String val)
{
	this->_val = Value(true);
}

template<> void Arg<int>::Val(String val)
{
	this->_val = Value(StrInt(val));
	if(this->_val.IsVoid()){ throw ArgExc("Argument is not an integer. Argument: " + val); }
}

template<> void Arg<int64>::Val(String val)
{
	this->_val = StrIntValue(val);
	if(this->_val.IsVoid()){ throw ArgExc("Argument is not an integer. Argument: " + val); }
}

template<> void Arg<double>::Val(String val)
{
	this->_val = StrDblValue(val);
	if(this->_val.IsVoid()){ throw ArgExc("Argument is not a decimal number. Argument: " + val); }
}

//*/

template<typename T>
class MultiArg : public TArg, public Vector<Value>
{
	protected:
		void Val(String val)
		{
			this->Add(Value(val));
		};
	public:
		MultiArg(String short_flag, String long_flag, String title, String description, bool required=false) : \
			TArg(short_flag, long_flag, title, description, required, true){};
		MultiArg(String title, String description, bool required=false) : \
			TArg(title, description, required, true){};
		
		Value Val(void)
		{
			return Value( this->GetCount() );
		};
		
		bool IsSet()
		{
			return this->GetCount() > 0;
		};	
};

template<> MultiArg<bool>::MultiArg(String short_flag, String long_flag, String title, String description, bool required) : \
	TArg(short_flag, long_flag, title, description, required, false){};

template<> MultiArg<bool>::MultiArg(String title, String description, bool required) : \
	TArg(title, description, required, false){}

template<> void MultiArg<bool>::Val(String val)
{
	this->Add(Value(true));
}

template<> void MultiArg<int>::Val(String val)
{
	Value tmp_val = Value(StrInt(val));
	if(tmp_val.IsVoid() || tmp_val.IsNull()) { return; }
	this->Add(tmp_val);
}

template<> void MultiArg<int64>::Val(String val)
{
	Value tmp_val = StrIntValue(val);
	if(tmp_val.IsVoid() || tmp_val.IsNull()) { return; }
	this->Add(tmp_val);
}

template<> void MultiArg<double>::Val(String val)
{
	Value tmp_val = StrDblValue(val);
	if(tmp_val.IsVoid() || tmp_val.IsNull()) { return; }
	this->Add(tmp_val);
}


/*int pos = -1;
pos = _title.Find(" ");			
//String whitespace(" \n\t");
//pos = _title.FindFirstOf(whitespace);
/*if(pos>=0)
{
	_long_flag = ToLower(_title.Left(pos));
}
else
{
	_long_flag = ToLower(_title);
}*/

#endif


		
		