#ifndef _ConveyorThread_h_
#define _ConveyorThread_h_

#include <Core/Core.h>
#include <windows.h>

using namespace Upp;

template<class T> class ConveyorThread : protected Thread
{ typedef ConveyorThread CLASSNAME;
public:
	ConveyorThread(bool _enabled = true)
		:enabled(_enabled)
		,clearNotExit(false)
	{
		finishEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
		queueEvent  = CreateEvent(NULL, TRUE, FALSE, NULL);
		
		Run(THISBACK(RunQueue));
	}
	
	virtual ~ConveyorThread()
	{
		CloseHandle(finishEvent);
		CloseHandle(queueEvent);
	}
	
	void Request(Callback1<const T &>handler, const T &request)
	{
		queueMutex.Enter();
		queue.AddTail(RequestElement(request, handler));
		queueMutex.Leave();
		if (enabled)
			SetEvent(queueEvent);
	}
	
	int GetQueueLength()
	{
		int l = 1;
		queueMutex.Enter();
		l = queue.GetCount();
		queueMutex.Leave();
		return l;
	}	
	void ClearQueue()
	{
		queueMutex.Enter();
		queue.Clear();
		queueMutex.Leave();
	}
	void ClearQueueWaitSuspended()
	{
		ClearQueue();
		clearNotExit = true;
		RequestFinish();
		while (working)
			Sleep(50);
	}
	
	void Enable(bool enable = true)
	{
		if (enable && !enabled && !queue.IsEmpty())
			SetEvent(queueEvent);
		enabled = enable;
	}
	
	void SharedEnter() {sharedDataMutex.Enter();}
	void SharedLeave() {sharedDataMutex.Leave();}

	void RequestFinish()     {SetEvent(finishEvent);}
	int  WaitFinished()      {return Wait();}
	bool IsFinished()        {return !IsOpen();}
	bool IsRequestedFinish() {return (WaitForSingleObject(finishEvent, 0) == WAIT_OBJECT_0);}
	
protected:
	HANDLE GetFinishEvent() {return finishEvent;}
	
private:
	void RunQueue()
	{
		const static int WAITINGEVENTS = 2;
		HANDLE waitingEvents[WAITINGEVENTS] = {finishEvent, queueEvent};
		const static DWORD SIGNALED_EVENT_FINISH = WAIT_OBJECT_0;
		const static DWORD SIGNALED_EVENT_QUEUE  = WAIT_OBJECT_0+1;
		
		DWORD signaledEvent;
		while ((signaledEvent = WaitForMultipleObjects(WAITINGEVENTS, waitingEvents, FALSE, INFINITE)) != SIGNALED_EVENT_FINISH || clearNotExit)
		{
			if (!enabled)
				continue;
			
			queueMutex.Enter();
			bool requests = !queue.IsEmpty();
			queueMutex.Leave();
			
			working = true;
			while (requests)
			{
				if (!enabled)
					break;
				
				queueMutex.Enter();
				const RequestElement headRequest = queue.Head(); //copy due to ::ClearRequests
				queue.DropHead();
				requests = !queue.IsEmpty();
				queueMutex.Leave();
				
				headRequest.handler.Execute(headRequest.t);
				
				if (requests && (WaitForSingleObject(finishEvent, 0) == WAIT_OBJECT_0))
					return;
			}
			working      = false;
			clearNotExit = false;
		}
	}
		
	bool   enabled;
	Mutex  queueMutex;
	Mutex  sharedDataMutex;
	HANDLE finishEvent;
	HANDLE queueEvent;
	bool   working;
	bool   clearNotExit;
	
	struct RequestElement : Moveable<RequestElement>
	{
		const T  t;
		Callback1<const T &> handler;
		RequestElement(const T &_t, Callback1<const T &> _c) :t(_t) , handler(_c) {}
	};
	typedef BiVector<RequestElement> RequestQueue;
	RequestQueue queue;
};

#endif
