#ifndef _RS232Thread_h_
#define _RS232Thread_h_

#include "ConveyorThread.h"

#ifdef LOG
	#undef LOG
	#define LOG(x)
#endif

using namespace Upp;

typedef String       RS232Data;
typedef Callback2<bool *, const void *>               RS232SendData;
typedef Callback4<DWORD, RS232Data *, bool *, void *> RS232ReceiveData;
typedef Callback3<bool, int, const void *>            RS232PacketReceived;

struct RS232RequestElement : public Moveable<RS232RequestElement>
{
	DCB       dcb;
	int       id;
	DWORD     timeout;
	RS232Data inData;
	
	RS232SendData       sendHandler;
	RS232ReceiveData    receiveHandler;
	RS232PacketReceived packetHandler;

	//MASTER	
	RS232RequestElement (RS232SendData onSend, RS232ReceiveData onReceive, RS232PacketReceived onPacket,
	                     DWORD _timeout, int _id, const DCB & _dcb,
	                     void *_inData, int _inDataSize)
		:timeout(_timeout), id(_id)
		,sendHandler(onSend), receiveHandler(onReceive), packetHandler(onPacket)
		,inData((const char *) _inData, _inDataSize)
	{ memcpy(&dcb, &_dcb, sizeof(dcb)); }
	
	//SLAVE
	RS232RequestElement (RS232ReceiveData onReceive, RS232PacketReceived onPacket,
	                     DWORD _timeout, int _id, const DCB & _dcb)
		:timeout(_timeout), id(_id)
		,receiveHandler(onReceive), packetHandler(onPacket)
	{ memcpy(&dcb, &_dcb, sizeof(dcb)); }
};

class RS232Thread : public ConveyorThread<RS232RequestElement>
{ typedef RS232Thread CLASSNAME;
	
public:
	RS232Thread()
		: h(INVALID_HANDLE_VALUE)
	{
		ioEvent    = CreateEvent(NULL, TRUE, FALSE, NULL);
		ZeroMemory(&o, sizeof(o));
		o.hEvent = ioEvent;
	}
	
	~RS232Thread()
	{
		Close();
		CloseHandle(ioEvent);
	}
	
	bool Open(int n)
	{
		Close();
	
		char comName[0xFF];
		sprintf(comName, "\\\\.\\COM%d", n);
		
		h = CreateFile (comName, GENERIC_READ | GENERIC_WRITE, 0, 0, OPEN_EXISTING, FILE_FLAG_OVERLAPPED, 0);
		if (h == INVALID_HANDLE_VALUE)
			return false;
		
		ZeroMemory(&curDCB, sizeof(curDCB));
		//GetCommState (h, &curDCB);
		SetStandardTimeouts();
		//SetCommMask(h, EV_RXCHAR);
	
		return true;
	}
	
	void Close()
	{
		if (h != INVALID_HANDLE_VALUE)
		{
			CloseHandle(h);
			h = INVALID_HANDLE_VALUE;
		}
	}
	
	void String2DCB(const String &s, DCB *dcb)
	{
		memcpy(dcb, &curDCB, sizeof(curDCB));
		BuildCommDCB(s, dcb);
	}


	template<class Tout> void RequestIOMaster(const RS232RequestElement &req)
	{ Request(THISBACK(IOMasterHandler<Tout>), req); }
	
	template<class Tout> void RequestIOSlave(const RS232RequestElement &req)
	{ Request(THISBACK(IOSlaveHandler<Tout>), req); }
	
	bool SendData(const RS232Data &data, DWORD timeout = INFINITE)
	{
		const static int WAITINGEVENTS = 2;
		HANDLE waitingEvents[WAITINGEVENTS] = {GetFinishEvent(), ioEvent};
		const static DWORD SIGNALED_EVENT_FINISH = WAIT_OBJECT_0;
		const static DWORD SIGNALED_EVENT_IO     = WAIT_OBJECT_0+1;
		DWORD written;
		WriteFile(h, data, data.GetLength(), &written, &o);
		DWORD result = WaitForMultipleObjects(WAITINGEVENTS, waitingEvents, FALSE, timeout);
		switch (result)
		{
		case SIGNALED_EVENT_IO:
			return true;
		case SIGNALED_EVENT_FINISH:
			CancelIo(h);
			SetEvent(GetFinishEvent());
			return false;
		default:
		case WAIT_TIMEOUT:
			CancelIo(h);
			return false;
		}
	}
	
	bool ReceiveData(DWORD size, RS232Data *data, DWORD timeout = INFINITE)
	{
		const static int WAITINGEVENTS = 2;
		HANDLE waitingEvents[WAITINGEVENTS] = {GetFinishEvent(), ioEvent};
		const static DWORD SIGNALED_EVENT_FINISH = WAIT_OBJECT_0;
		const static DWORD SIGNALED_EVENT_IO     = WAIT_OBJECT_0+1;
		
		const static int STATIC_BUF_SIZE = 512;
		char staticBuf[STATIC_BUF_SIZE];
		
		bool   success = false;
		DWORD  read    = 0;
		char  *buf     = (size < STATIC_BUF_SIZE) ? staticBuf : new char[size];
		
		//SetStandardTimeouts(10000/*timeout == INFINITE ? 0 : timeout*/);
		int io = ReadFile(h, buf, size, &read, &o);
		/*if (io)
		{
			CancelIo(h);
			SetEvent(GetFinishEvent());
			LOG("SIGNALED_EVENT_FINISH");
		}
		else
		*/{
			DWORD result = /*SIGNALED_EVENT_IO;*/ WaitForMultipleObjects(WAITINGEVENTS, waitingEvents, FALSE, timeout);
			switch (result)
			{
			case SIGNALED_EVENT_IO:
				GetOverlappedResult(h, &o, &read, FALSE);
				if (read)
				{
					data->Cat(buf, read);
					success = true;
				}
				CancelIo(h);
				LOG(Format("[%d] SIGNALED_EVENT_IO (%d/%d finished)", (int) io, (int) read, (int) size));
				break;
			case SIGNALED_EVENT_FINISH:
				CancelIo(h);
				SetEvent(GetFinishEvent());
				LOG("SIGNALED_EVENT_FINISH");
				break;
			case WAIT_TIMEOUT:
				CancelIo(h);
				GetOverlappedResult(h, &o, &read, FALSE);
				LOG(Format("WAIT_TIMEOUT (%d finished)", (int) read));
				break;
			default:
				CancelIo(h);
				LOG("default");
			}
		}
		
		if (size >= STATIC_BUF_SIZE)		
			delete[] buf;
		return success;
	}
	
protected:
	template<class Tout> void IOMasterHandler (const RS232RequestElement &req)
	{
		bool succ;
		bool rcv;
		RS232Data rcvData;
		Tout outStruct;

		SetDCB(req.dcb);

		do
		{
			PurgeComm(h, PURGE_TXCLEAR|PURGE_RXCLEAR);
	
			req.sendHandler.Execute(&succ, (const void *) ((const char *) req.inData));
			if (!succ)
				return;

			req.receiveHandler.Execute(req.timeout, &rcvData, &rcv, &outStruct);
			
			if (WaitForSingleObject(GetFinishEvent(), 0) == WAIT_OBJECT_0)
				return;
		}
		while (!rcv);
		
		req.packetHandler.Execute(true, req.id, &outStruct);
	}

	template<class Tout> void IOSlaveHandler (const RS232RequestElement &req)
	{
		bool rcv;
		RS232Data rcvData;
		Tout outStruct;

		SetDCB(req.dcb);

		do
		{
			PurgeComm(h, PURGE_TXCLEAR|PURGE_RXCLEAR);
			
			req.receiveHandler.Execute(req.timeout, &rcvData, &rcv, &outStruct);
			req.packetHandler.Execute(rcv, req.id, &outStruct);
			
			if (WaitForSingleObject(GetFinishEvent(), 0) == WAIT_OBJECT_0)
				return;
		}
		while (!rcv);
		
	}

	void SetDCB(const DCB &dcb)
	{
		if (!memcmp(&dcb, &curDCB, sizeof(dcb)))
			return;
		
		memcpy(&curDCB, &dcb, sizeof(dcb));
		if (!SetCommState(h, &curDCB))
			LOG("SetCommState -");
		else
			LOG(Format("SetCommState + (baud:%d, parity:%d, stop:%d)", (int) dcb.BaudRate, (int) dcb.Parity, (int) dcb.StopBits));
	}
	
	void SetStandardTimeouts(DWORD readTimeout = 0)
	{
		COMMTIMEOUTS curTimeouts;
		
		curTimeouts.ReadIntervalTimeout         = 0;
		curTimeouts.ReadTotalTimeoutMultiplier  = 0;
		curTimeouts.ReadTotalTimeoutConstant    = readTimeout;
		
		curTimeouts.WriteTotalTimeoutMultiplier = 0;
		curTimeouts.WriteTotalTimeoutConstant   = 0;
		
		SetCommTimeouts(h, &curTimeouts);
	}
	
private:
	HANDLE       h;
	HANDLE       ioEvent;
	DCB          curDCB;
	OVERLAPPED   o;
};

#endif
