/*
    U++ Command Line Argument Processor 0.3.0
    Author: Chris (captainc)
    Date: 11/03/08
    Description:
    This package aggregates and validates command line input. 
    It allows for multiple different command line argument combinations.    
    Users define their arguments, conditions, and allowable combinations; 
    the package will ensure that the arguments are assigned the correct values 
    and occur in the correct combinations. Usage information is automatically 
    generated and ready for output to command line (or other stream).
    
    This U++ package was inspired by TCLAP the Templatized Command Line Parser 
    (tclap.sourceforge.net). TCLAP is not friendly with U++, so it was decided 
    to create somewhat similar functionality using U++ types. Though, much 
    functionality has been added as well.
*/

#ifndef UPP_CMDLINEARGPROCESSOR_ARGPROCESSOR_HPP
#define UPP_CMDLINEARGPROCESSOR_ARGPROCESSOR_HPP

#include <Core/Core.h>
#include <CmdLineArgProcessor/TArgument.hpp>
#include <CmdLineArgProcessor/ArgSet.hpp>

using namespace Upp;

class ArgProcessor
{
	protected:
		String _program_title, _version;
		String _short_delim, _long_delim, _error_str;
		Vector< Ptr<ArgSet> > _arg_sets;
		Vector< Ptr<TOption> > _arg_list;
		Vector< Ptr<TArg> > _base_arg_list;
		Stream * _out_stream;
		bool _error, _print_usage, _found_match;
		
		bool IsHelpFlag(String flag, bool long_flag);
		bool IsVersionFlag(String flag, bool long_flag=false);
		
		void AddOption(Ptr<TOption> parg);
		void AddBaseArg(Ptr<TArg> parg);
		static bool IsIn(Ptr<TOption> parg, const Vector< Ptr<TOption> > & args);
		static bool IsIn(Ptr<TArg> parg, const Vector< Ptr<TArg> > & args);
		int GetArgIdx(String id, bool long_flag);
		
		bool MatchesBaseArgs(Ptr<ArgSet> & arg_set, const Vector<String> & cmdline, int idx);
		bool MatchesOptions(Ptr<ArgSet> arg_set);
		
		String GetArgUsage(Ptr<TOption> pTDArg);
		String GetArgUsage(Ptr<TArg> parg);
		String GetArgDetails(Ptr<TOption> pTDArg, const Vector<int> & widths);
		String GetArgDetails(Ptr<TArg> pTBArg, const Vector<int> & widths);
		
		String GetHelpUsage();

		Vector<int> GetGlobalWidths();
		Vector<int> GetArgSetWidths(Ptr<ArgSet> arg_set);

	public:
		ArgProcessor(String program_title, String version) : 
			_program_title(program_title), _version(version), _short_delim("-"), _long_delim("--"), \
			_out_stream(&(Cout())), _error(false), _error_str(""), _print_usage(false), _found_match(false) {};
		
		void AddArgSet(ArgSet& arg_set);
				
		ArgProcessor& operator<<(ArgSet& arg_set)
		{
			AddArgSet(arg_set);
			return *this;
		};
		
		bool HasMatch(){ return _found_match; };
		
		void ProcessCmdLine(const Vector<String> & cmdline);
		void SetArgDelim(String short_delim, String long_delim);
				
		String UsageInformation();
		String UsageInformationCondensed();
		
		void PrintUsageInformation(bool condensed=true);
		
		String VersionInformation();
		void PrintVersionInformation();		
		
		String GetError() { return _error_str; }
		bool IsError() { return _error; };
		void PrintUsageOnError(bool print_usage=true){ _print_usage=print_usage; };
		void PrintToStream(Stream & ss=Cout());
};

#endif
