/*
All Unit Tests for UCLAP and Print Output Tests (eyeball tests)
*/

#include <Core/Core.h>
#include <CmdLineArgProcessor/ArgProcessor.hpp>

using namespace Upp;

#define TEST1_SETUP \
ArgProcessor ap("Command Line Argument Processor Test", "1");\
Arg<int> argA ("integer", "integer arg description.", true);\
Arg<String> argB ("string", "string arg description.", true);\
MultiArg<int> argC ("numbers", "minimum of 5 numbers must be specified.", true, 5);\
ArgSet test1_set1;\
test1_set1 << argA << argB << argC;\
ap << test1_set1;\
Vector<String> cmdline_sim;

#define TEST1_TESTPASS(TEST_NO) \
ap.ProcessCmdLine(cmdline_sim);\
if(ap.IsError() || !test1_set1.IsMatch())\
{\
	Cout() << "Test " << #TEST_NO << " Failed\n";\
	Cout() << "\t" << ap.GetError() << "\n";\
	all_good = false;\
}\
else{\
	Cout() << "Test " << #TEST_NO << " Passed\n";\
	Cout() << "\targA val: " << argA.Val() << "\n";\
	Cout() << "\targB val: " << argB.Val() << "\n";\
	Cout() << "\targC count: " << argC.GetCount();\
	String argc_vals("");\
	for(int i=0; i<argC.GetCount();++i){argc_vals << argC[i] << " ";}\
	Cout() << "\targC vals: " << argc_vals << "\n";\
}

#define GENERIC_TESTFAIL(TEST_NO) \
ap.ProcessCmdLine(cmdline_sim);\
if(ap.IsError() || !test1_set1.IsMatch())\
{\
	Cout() << "Test " << #TEST_NO << " Passed\n";\
	Cout() << "\t" << ap.GetError() << "\n";\
}\
else{\
	Cout() << "Test " << #TEST_NO << " Failed\n";\
	all_good = false;\
}

bool Test1() // Only Base Args
{
	// Test format: myproject Arg<int> Arg<String> MultiArg<int> <int> ... <int>
	bool all_good = true;
	// Test 1a: Valid Input, MultiArg at Minimum value
	{
		TEST1_SETUP
		cmdline_sim << "5" << "TestString" << "1" << "2" << "3" << "4" << "5";
		TEST1_TESTPASS(1a)
	}
	
	// Test 1b: Valid Input with dash (delimiter) as first argument, MultiArg > Minimum value
	{
		TEST1_SETUP
		cmdline_sim << "-5" << "TestString" << "1" << "2" << "3" << "4" << "5" << "6";
		TEST1_TESTPASS(1b)
	}
	
	// Test 1c: Invalid Input - Too Few Arguments
	{
		TEST1_SETUP
		cmdline_sim << "5" << "TestString" << "1" << "2" << "3";
		GENERIC_TESTFAIL(1c)
	}
	
	// Test 1d: Invalid Input - Missing / Wrong / Misplaced Argument
	{
		TEST1_SETUP
		cmdline_sim << "1" << "1" << "2" << "3";
		GENERIC_TESTFAIL(1d)
	}
	
	// Test 1e: Invalid Input - Wrong Argument Type In MultiArg
	{
		TEST1_SETUP
		cmdline_sim << "1" << "TestString" << "1" << "2" << "3" << "4" << "five" << "6";
		GENERIC_TESTFAIL(1e)
	}
	
	return all_good;
};

#define TEST2_SETUP \
ArgProcessor ap("Command Line Argument Processor Test", "2");\
Arg<String> argB ("string", "string arg description.", true);\
MultiArg<int> argC ("numbers", "maximum of 5 numbers can be specified.", true, 1, 5);\
ArgSet test1_set1;\
test1_set1 << argC << argB;\
ap << test1_set1;\
Vector<String> cmdline_sim;

#define TEST2_SETUP2 \
ArgProcessor ap("Command Line Argument Processor Test", "2");\
Arg<String> argB ("string", "string arg description.", true);\
MultiArg<int> argC ("numbers", "exactly 2 numbers must be specified.", true, 2, 2);\
ArgSet test1_set1;\
test1_set1 << argC << argB;\
ap << test1_set1;\
Vector<String> cmdline_sim;

#define TEST2_TESTPASS(TEST_NO) \
ap.ProcessCmdLine(cmdline_sim);\
if(ap.IsError() || !test1_set1.IsMatch())\
{\
	Cout() << "Test " << #TEST_NO << " Failed\n";\
	Cout() << "\t" << ap.GetError() << "\n";\
	all_good = false;\
}\
else{\
	Cout() << "Test " << #TEST_NO << " Passed\n";\
	Cout() << "\targC count: " << argC.GetCount();\
	String argc_vals("");\
	for(int i=0; i<argC.GetCount();++i){argc_vals << argC[i] << " ";}\
	Cout() << "\targC vals: " << argc_vals << "\n";\
	Cout() << "\targB val: " << argB.Val() << "\n";\
}

#define TEST2_TESTFAIL(TEST_NO) \
ap.ProcessCmdLine(cmdline_sim);\
if(ap.IsError() || !test1_set1.IsMatch())\
{\
	Cout() << "Test " << #TEST_NO << " Passed\n";\
	Cout() << "\t" << ap.GetError() << "\n";\
}\
else\
{\
	Cout() << "Test " << #TEST_NO << " Failed\n";\
	Cout() << "\targC count: " << argC.GetCount();\
	String argc_vals("");\
	for(int i=0; i<argC.GetCount();++i){argc_vals << argC[i] << " ";}\
	Cout() << "\targC vals: " << argc_vals << "\n";\
	Cout() << "\targB val: " << argB.Val() << "\n";\
	all_good = false;\
}

bool Test2() // Only Base Args, MultiArg now has Min=1, Max=5
{
	// Test format: myproject MultiArg<int> <int> ... <int> Arg<String>
	bool all_good = true;
	// Test 2a: Valid input, MultiArg at Max=5, explicit switch to argument of new type.
	{
		TEST2_SETUP
		cmdline_sim << "1" << "2" << "3" << "4" << "5" << "TestString";
		TEST2_TESTPASS(2a)
	}
	
	// Test 2b: Valid input, MultiArg below Max, implicit switch to argument of new type. 
	{
		TEST2_SETUP
		cmdline_sim << "1" << "2" << "3" << "TestString";
		TEST2_TESTPASS(2b)
	}
	
	// Test 2c: Invalid input, MultiArg above Max, Leaving extra arguments on the command line
	{
		TEST2_SETUP
		cmdline_sim << "1" << "2" << "3" << "4" << "5" << "6" << "TestString";
		TEST2_TESTFAIL(2c)
	}
	
	// Test 2d: Valid input, MultiArg at required count
	{
		TEST2_SETUP2
		cmdline_sim << "1" << "2" << "TestString";
		TEST2_TESTPASS(2d)
	}
	
	// Test 2e: Invalid input, MultiArg below required count
	{
		TEST2_SETUP2
		cmdline_sim << "1" << "TestString";
		GENERIC_TESTFAIL(2e)
	}
	
	return all_good;
};

#define TEST3_SETUP \
ArgProcessor ap("Command Line Argument Processor Test", "3");\
Option<int> argA ("integer", "integer arg description.", true);\
Option<String> argB ("string", "string arg description.", true);\
MultiOption<double> argC ("numbers", "any number of numbers can be specified", false);\
ArgSet test1_set1;\
test1_set1 << argA << argB << argC;\
ap << test1_set1;\
Vector<String> cmdline_sim;

#define TEST3_TESTPASS(TEST_NO) \
ap.ProcessCmdLine(cmdline_sim);\
if(ap.IsError() || !test1_set1.IsMatch())\
{\
	Cout() << "Test " << #TEST_NO << " Failed\n";\
	Cout() << "\t" << ap.GetError() << "\n";\
	all_good = false;\
}else{\
	Cout() << "Test " << #TEST_NO << " Passed\n";\
	Cout() << "\targA val: " << argA.Val() << "\n";\
	Cout() << "\targB val: " << argB.Val() << "\n";\
	String argc_vals("");\
	for(int i=0; i<argC.GetCount();++i){argc_vals << argC[i] << " ";}\
	Cout() << "\targC count: " << argC.GetCount();\
	Cout() << "\targC vals: " << argc_vals << "\n";\
}

bool Test3() // Only Delimited Args
{
	// Test format: myproject -a argA -b argB1 -b argB2
	bool all_good = true;
	
	// Test 3a: Valid input, all args
	{
		TEST3_SETUP
		cmdline_sim << "-i" << "1" << "--string" << "'Test String With Spaces'" << "-n" << "1" << "-n" << "2";
		TEST3_TESTPASS(3a)
	}
	
	// Test 3b: Valid input, -n not specified
	{
		TEST3_SETUP
		cmdline_sim << "-i" << "1" << "--string" << "'Test String With Spaces'";
		TEST3_TESTPASS(3b)
	}
	
	// Test 3c: Invalid input, -string specified instead of --string
	{
		TEST3_SETUP
		cmdline_sim << "-i" << "1" << "-string" << "'Test String With Spaces'";
		GENERIC_TESTFAIL(3c)
	}
	
	return all_good;
};

#define TEST4_SETUP \
ArgProcessor ap("Command Line Argument Processor Test", "3");\
Option<bool> dargA ("switch", "true or false");\
MultiOption<double> dargC ("numbers", "any number of numbers can be specified", false);\
Arg<int> bargA ("base-integer", "just an integer", true);\
MultiArg<double> bargC ("base-multi-double", "list of reals", true);\
ArgSet test1_set1;\
test1_set1 << dargA << dargC << bargA << bargC;\
ap << test1_set1;\
Vector<String> cmdline_sim;

#define TEST4_TESTPASS(TEST_NO) \
ap.ProcessCmdLine(cmdline_sim);\
if(ap.IsError() || !test1_set1.IsMatch())\
{\
	Cout() << "Test " << #TEST_NO << " Failed\n";\
	Cout() << "\t" << ap.GetError() << "\n";\
	all_good = false;\
}else{\
	Cout() << "Test " << #TEST_NO << " Passed\n";\
	Cout() << "\tdargA val: " << dargA.Val() << "\n";\
	String dargc_vals("");\
	for(int i=0; i<dargC.GetCount();++i){dargc_vals << dargC[i] << " ";}\
	Cout() << "\tdargC count: " << dargC.GetCount();\
	Cout() << "\tdargC vals: " << dargc_vals << "\n";\
	Cout() << "\tbargA val: " << bargA.Val() << "\n";\
	String bargc_vals("");\
	for(int i=0; i<bargC.GetCount();++i){bargc_vals << bargC[i] << " ";}\
	Cout() << "\tbargC count: " << bargC.GetCount();\
	Cout() << "\tbargC vals: " << bargc_vals << "\n";\
}

bool Test4() // Combination switch then normal
{
	// Test format: myproject -s -n dargA bargA bargB1 ... bargBn
	bool all_good = true;
	
	// Test 4a: Valid input, all args
	{
		TEST4_SETUP
		cmdline_sim << "-n" << "1" << "--switch" << "2" << "1.1" << "1.2" << "1.3" << "1.4";
		TEST4_TESTPASS(4a)
	}
	
	// Test 4b: Valid input, no Option's
	{
		TEST4_SETUP
		cmdline_sim << "2" << "1.1" << "1.2" << "1.3" << "1.4";
		TEST4_TESTPASS(4b)
	}

	// Test 4c: Invalid input, missing required argument
	{
		TEST4_SETUP
		cmdline_sim << "-n" << "1" << "-n" << "2";
		GENERIC_TESTFAIL(4c)
		
		/*Cout() << "\n";
		ap.PrintUsageInformation();
		Cout() << "\n";*/
	}
	
	return all_good;
};

#define TEST5_SETUP \
ArgProcessor ap("Command Line Argument Processor Test", "5");\
Option<int> argA ("integer", "integer arg description.", true);\
Option<String> argB ("string", "string arg description.", true);\
MultiArg<double> argC ("numbers", "any number of numbers can be specified", false);\
ArgSet test_set1, test_set2;\
test_set1 << argA << argC;\
test_set2 << argB << argC;\
ap << test_set1 << test_set2;\
Vector<String> cmdline_sim;

bool Test5() // Ensure that the correct argset is matched when multiple argsets specified
{
	bool all_good = true;
	
	// Test 5a: match test_set1
	{
		TEST5_SETUP
		cmdline_sim << "-i" << "1" << "1.1";
		ap.ProcessCmdLine(cmdline_sim);
		if(!test_set1.IsMatch() || ap.IsError())
		{
			all_good=false;
			Cout() << "Test " << "5a" << " Failed\n";
		}else{
			Cout() << "Test " << "5a" << " Passed\n";
		}
		
		if(ap.IsError())
		{
			Cout() << "\t" << ap.GetError() << "\n";
		}
		
	}
	
	// Test 5b: match test_set2
	{
		TEST5_SETUP
		cmdline_sim << "-s" << "TestString1" << "1.1" << "1.2";
		ap.ProcessCmdLine(cmdline_sim);
		if(!test_set2.IsMatch() || ap.IsError())
		{
			all_good=false;
			Cout() << "Test " << "5b" << " Failed\n";
		}else{
			Cout() << "Test " << "5b" << " Passed\n";
		}
		
		if(ap.IsError())
		{
			Cout() << "\t" << ap.GetError() << "\n";
		}
	}
	
	// Test 5c: match none
	{
		TEST5_SETUP
		cmdline_sim << "test";
		ap.ProcessCmdLine(cmdline_sim);
		if(test_set1.IsMatch() || test_set2.IsMatch() || !ap.IsError())
		{
			all_good=false;
			Cout() << "Test " << "5c" << " Failed\n";
		}else{
			Cout() << "Test " << "5c" << " Passed\n";
		}
		
		if(ap.IsError())
		{
			Cout() << "\t" << ap.GetError() << "\n";
		}
	}
	
	return all_good;
};

void PrintUsageTest()
{
	ArgProcessor ap("Command Line Arg Processor Test", "0.3");
	ArgSet set1, set2, set3, set4, set5, set6, set7, set8, common;
	Option<String> dargA ("string", "An optional delimited string input");
	Option<int> dargB ("integer", "A required delimited integer input", true);
	MultiOption<String> dargC ("multi-string", "An optional arguement that can be specified up to 5 times", false, 1, 5);
	Arg<String> bargA ("base-string", "A required string input. Must be the 1st specified base argument.", true);
	Arg<double> bargB ("base-double", "A non-required double input. 2nd on command line if specified.");
	
	common << dargB << dargC << bargB;
	
	int min;
	min=3;
	MultiArg<String> bargC1 ("base-multi-str", "A non-required list of strings", false, min, min);
	MultiArg<String> bargC2 ("base-multi-str", "A non-required list of strings", false, min, min+1);
	MultiArg<String> bargC3 ("base-multi-str", "A non-required list of strings", false, min, min+2);
	MultiArg<String> bargC4 ("base-multi-str", "A non-required list of strings", false, min, min+3);
	MultiArg<String> bargC5 ("base-multi-str", "A non-required list of strings", false, min);
	
	//set1 << dargA << dargB << dargC << bargA << bargB << bargC;
	set1 << bargC1 << common;
	set2 << bargC2 << common;
	set3 << bargC3 << common;
	set4 << bargC4 << common;
	set5 << bargC5 << common;
		
	ap << set1 << set2 << set3 << set4 << set5;
	Cout() << "\nCondensed Output:\n\n";
	ap.PrintUsageInformation(true);
	Cout() << "\n\nFull Output:\n\n";
	ap.PrintUsageInformation(false);
}

CONSOLE_APP_MAIN
{
	
	if( Test1() && Test2() && Test3() && Test4() && Test5() )
	{
		Cout() << "\nRESULT: All Tests Passed\n\n";
	}
	else
	{
		Cout() << "\nRESULT: Something Failed\n\n";
	}
	
	PrintUsageTest();
	Cout() << "\n";
}

