#ifndef UPP_CMDLINEARGPROCESSOR_TARG_HPP
#define UPP_CMDLINEARGPROCESSOR_TARG_HPP

#include <Core/Core.h>

using namespace Upp;

class ArgExc : public Exc
{
	public:
		ArgExc(String error_str) : Exc(error_str){};
};

/**
Virtual base class for all arguments.
*/
class TArgument //: public Pte<TArgument>, public Moveable<TArgument>
{
	friend class ArgProcessor;
	friend class ArgSet;
	protected:		
		String _title, _description;
		Value _val;
		bool _required, _requires_value;
		int _max, _min;
		virtual void Val(String val)=0;
		//virtual void Reset(){ _val = Null; };
		
	public:
		TArgument() : \
			_title(""), _description(""), \
			_required(false), _requires_value(false), \
			_min(1), _max(1) {};
		
		TArgument(String title, String description, bool required, bool requires_value, int min=1, int max=1) : \
			_title(title), _description(description), \
			_required(required), _requires_value(requires_value), \
			_min(min), _max(max) 
			{
				if(_min<1){_min=1;}
				if(_max<1){_max=1;}
				if(_max<_min){_max=_min;}
			};
		
		TArgument(const TArgument& arg)
		{
			_title = arg._title;
			_description = arg._description;
			_val = arg._val;
			_required = arg._required;
			_requires_value = arg._requires_value;
			_min = arg._min;
			_max = arg._max;
		};
		
		String GetTitle()		{ return _title; };		
		String GetDescription() { return _description; };
		bool RequiresValue()	{ return _requires_value; };
		bool IsRequired()		{ return _required; };
		virtual bool IsSet()	{ return !this->Val().IsNull() && !this->Val().IsVoid(); };
		int Min()				{ return _min; };
		int Max()				{ return _max; };
		
		virtual Value Val(void)=0;
};

////////////////////////////////
//* Start Delimited Arguments */
////////////////////////////////

class TOption : public virtual TArgument, public Pte<TOption>, public Moveable<TOption>
{
	protected:
		String _short_flag, _long_flag;
	public:
		TOption() : _short_flag(""), _long_flag("")
		{
			SetFlagsFromTitle();
		};
		
		TOption(String short_flag, String long_flag) : _short_flag(short_flag), _long_flag(long_flag)	{};
		
		TOption(const TOption& arg)
		{
			_short_flag = arg._short_flag;
			_long_flag = arg._long_flag;
		};
		
		String GetLongFlag()	{ return _long_flag; };
		String GetShortFlag()	{ return _short_flag; };
		
		void SetFlagsFromTitle()
		{
			_short_flag = ToLower(this->GetTitle().Left(1));
			_long_flag = ToLower(FilterWhile(this->GetTitle(), CharFilterNotWhitespace));
		};
		
		static int CharFilterNotWhitespace(int c){ return IsSpace(c) ? 0 : c; };
};

template<typename T>
class Option : public TOption
{
	protected:
		void Val(String val){ this->_val = Value(val); };
		
	public:
		Option(String short_flag, String long_flag, String title, String description, bool required=false) : \
			TArgument(title, description, required, true), \
			TOption(short_flag, long_flag){};
		
		Option(String title, String description, bool required=false) : \
			TArgument(title, description, required, true){};
		
		Value Val(void)	{ return _val; };
};

template<> inline Option<bool>::Option(String short_flag, String long_flag, String title, String description, bool required) : \
	TArgument(title, description, required, false), \
	TOption(short_flag, long_flag){}

template<> inline Option<bool>::Option(String title, String description, bool required) : \
	TArgument(title, description, required, false){}

template<> inline void Option<bool>::Val(String val)
{
	this->_val = Value(true);
}

template<> inline void Option<int>::Val(String val)
{
	this->_val = Value(StrInt(val));
	if(!this->IsSet()){ throw ArgExc("Argument is not an integer. Argument: " + val); } // this->_val.IsVoid()
}

template<> inline void Option<int64>::Val(String val)
{
	this->_val = StrIntValue(val);
	if(!this->IsSet()){ throw ArgExc("Argument is not an integer. Argument: " + val); }
}

template<> inline void Option<double>::Val(String val)
{
	this->_val = StrDblValue(val);
	if(!this->IsSet()){ throw ArgExc("Argument is not a decimal number. Argument: " + val); }
}

template<typename T>
class MultiOption : public TOption, public Vector<Value>
{
	protected:
		void Val(String val)
		{
			this->Add(Value(val));
		};
		
		/*void Reset()
		{
			TArgument::Reset();
			this->Clear();
		};*/
		
	public:
		MultiOption(String short_flag, String long_flag, String title, String description, bool required=false, int min=1, int max=INT_MAX) : \
			TArgument(title, description, required, true, min, max), \
			TOption(short_flag, long_flag){};
		MultiOption(String title, String description, bool required=false, int min=1, int max=INT_MAX) : \
			TArgument(title, description, required, true, min, max){};
		
		Value Val(void)
		{
			return Value( this->GetCount() );
		};
		
		bool IsSet()
		{
			return this->GetCount() > 0 && this->GetCount() >= _min;
		};	
};

template<> inline MultiOption<bool>::MultiOption(String short_flag, String long_flag, String title, String description, bool required, int min, int max) : \
	TArgument(title, description, required, false, min, max), \
	TOption(short_flag, long_flag){};

template<> inline MultiOption<bool>::MultiOption(String title, String description, bool required, int min, int max) : \
	TArgument(title, description, required, false, min, max){}

template<> inline void MultiOption<bool>::Val(String val)
{
	this->Add(Value(true));
}

template<> inline void MultiOption<int>::Val(String val)
{
	Value tmp_val = Value(StrInt(val));
	if(tmp_val.IsVoid() || tmp_val.IsNull()) //{ return; }
	{ throw ArgExc("Argument is not an integer. Argument: " + val); }
	this->Add(tmp_val);
}

template<> inline void MultiOption<int64>::Val(String val)
{
	Value tmp_val = StrIntValue(val);
	if(tmp_val.IsVoid() || tmp_val.IsNull()) //{ return; }
	{ throw ArgExc("Argument is not an integer. Argument: " + val); }
	this->Add(tmp_val);
}

template<> inline void MultiOption<double>::Val(String val)
{
	Value tmp_val = StrDblValue(val);
	if(tmp_val.IsVoid() || tmp_val.IsNull()) //{ return; }
	{ throw ArgExc("Argument is not a decimal number. Argument: " + val); }
	this->Add(tmp_val);
}

////////////////////////////
//* Start Basic Arguments */
////////////////////////////

class TArg : public virtual TArgument, public Pte<TArg>, public Moveable<TArg> 
{};

template<typename T>
class Arg : public TArg
{
	protected:
		void Val(String val){ this->_val = Value(val); };
	public:
		Arg(String title, String description, bool required=false, int min=1, int max=1): \
			TArgument(title, description, required, true){};
		
		Value Val(void)	{ return _val; };
};

template<> inline void Arg<bool>::Val(String val)
{
	this->_val = Value(true);
}

template<> inline void Arg<int>::Val(String val)
{
	this->_val = Value(StrInt(val));
	if(!this->IsSet()){ throw ArgExc("Argument is not an integer. Argument: " + val); }
}

template<> inline void Arg<int64>::Val(String val)
{
	this->_val = StrIntValue(val);
	if(!this->IsSet()){ throw ArgExc("Argument is not an integer. Argument: " + val); }
}

template<> inline void Arg<double>::Val(String val)
{
	this->_val = StrDblValue(val);
	if(!this->IsSet()){ throw ArgExc("Argument is not a decimal number. Argument: " + val); }
}

template<typename T>
class MultiArg : public TArg, public Vector<Value>
{
	protected:
		void Val(String val){ this->Add(Value(val)); };
		
		/*void Reset()
		{
			TArgument::Reset();
			this->Clear();
		};*/
	public:
		Value Val(void)
		{
			return Value( this->GetCount() );
		};
		
		bool IsSet()
		{
			return this->GetCount() > 0 && this->GetCount() >= _min;
		};	
	
		MultiArg(String title, String description, bool required=false, int min=1, int max=INT_MAX): \
			TArgument(title, description, required, true, min, max){};
		
};

template<> inline void MultiArg<bool>::Val(String val)
{
	this->Add(Value(true));
}

template<> inline void MultiArg<int>::Val(String val)
{
	Value tmp_val = Value(StrInt(val));
	if(tmp_val.IsVoid() || tmp_val.IsNull()) //{ return; }
	{ throw ArgExc("Argument is not an integer. Argument: " + val); }
	this->Add(tmp_val);
}

template<> inline void MultiArg<int64>::Val(String val)
{
	Value tmp_val = StrIntValue(val);
	if(tmp_val.IsVoid() || tmp_val.IsNull()) //{ return; }
	{ throw ArgExc("Argument is not an integer. Argument: " + val); }
	this->Add(tmp_val);
}

template<> inline void MultiArg<double>::Val(String val)
{
	Value tmp_val = StrDblValue(val);
	if(tmp_val.IsVoid() || tmp_val.IsNull()) //{ return; }
	{ throw ArgExc("Argument is not a decimal number. Argument: " + val); }
	this->Add(tmp_val);
}

#endif

