#include <Core/Core.h>
#include <CmdLineArgProcessor/TArgument.hpp>
#include <CmdLineArgProcessor/ArgSet.hpp>
#include <CmdLineArgProcessor/ArgProcessor.hpp>

using namespace Upp;

bool ArgProcessor::IsIn(Ptr<TOption> parg, const Vector< Ptr<TOption> > & args)
{
	for(int i=0; i<args.GetCount(); i++)
	{
		if(args[i]==parg)
		{
			return true;
		}
	}
	return false;
}

void ArgProcessor::AddOption(Ptr<TOption> parg)
{
	if(!IsIn(parg, _arg_list))
	{
		_arg_list.Add(parg);
	}
}

//\TODO: refactor these functions to work with Base Class of TArg and TOption
bool ArgProcessor::IsIn(Ptr<TArg> parg, const Vector< Ptr<TArg> > & args)
{
	for(int i=0; i<args.GetCount(); i++)
	{
		if(args[i]==parg)
		{
			return true;
		}
	}
	return false;
}

void ArgProcessor::AddBaseArg(Ptr<TArg> parg)
{
	if(!IsIn(parg, _base_arg_list))
	{
		_base_arg_list.Add(parg);
	}
}

void ArgProcessor::AddArgSet(ArgSet& arg_set)
{
	for(int i=0; i<arg_set._options.GetCount(); i++)
	{
		AddOption(arg_set._options[i]);
	}
	for(int i=0; i<arg_set._base_args.GetCount(); i++)
	{
		AddBaseArg(arg_set._base_args[i]);
	}
	_arg_sets.Add(&arg_set);
}

int ArgProcessor::GetArgIdx(String id, bool long_flag)
{
	for(int i=0; i<_arg_list.GetCount(); i++){
		if( (long_flag && _arg_list[i]->GetLongFlag().IsEqual(id) ) || _arg_list[i]->GetShortFlag().IsEqual(id) )
		{ return i; }
	}
	return -1;
}

bool ArgProcessor::IsHelpFlag(String flag, bool long_flag)
{
	return (long_flag && flag.IsEqual("help")) || flag.IsEqual("h");	
}

bool ArgProcessor::IsVersionFlag(String flag, bool long_flag)
{
	return (long_flag && flag.IsEqual("version")) || flag.IsEqual("v");
}

void ArgProcessor::ProcessCmdLine(const Vector<String> & cmdline)
{
	// Parse cmdline and assign values to arguments.
	try
	{
		int i=0;
		for(i; i < cmdline.GetCount(); i++)
		{
			if(cmdline[i].StartsWith(_long_delim) || cmdline[i].StartsWith(_short_delim))
			{
				int pos = cmdline[i].StartsWith(_long_delim) ? \
						cmdline[i].Find(_long_delim)+(_long_delim.GetCount()-1) : \
						cmdline[i].Find(_short_delim)+(_short_delim.GetCount()-1);
				String arg_flag = cmdline[i].Mid(pos+1); //ToLower
				bool is_long_flag = pos>(_short_delim.GetCount()-1);
				if(IsHelpFlag(arg_flag, is_long_flag))
				{
					PrintUsageInformation();
					return;
				}
				else if(IsVersionFlag(arg_flag, is_long_flag))
				{
					PrintVersionInformation();
					return;
				}
				else
				{
					int arg_pos=GetArgIdx(arg_flag, is_long_flag);
					if(arg_pos>=0)
					{
						if(_arg_list[arg_pos]->RequiresValue())
						{
							i++;
							if( i < cmdline.GetCount() )
							{
								_arg_list[arg_pos]->Val(cmdline[i]);
							}
							else
							{
								String used_flag = is_long_flag ? \
									_long_delim+_arg_list[arg_pos]->GetLongFlag() : \
									_short_delim+_arg_list[arg_pos]->GetShortFlag();
								throw ArgExc("Expected value for argument: '" + \
										_arg_list[arg_pos]->GetTitle() + "', flag: '" + used_flag + "'.");
							}
						}
						else // assume it is a switch option (they require no values)
						{
							//TODO: i'd like to refactor the use of blank string in Val.
							_arg_list[arg_pos]->Val("");
						}
					}
					else // Unknown flag used,
					{
						// throw ArgExc("Unknown flag used: '" + cmdline[i] + "'");
						// don't fail, rather let it try to match it to the BaseArgs (Ie. negative numbers "-2")
						break;
					}
				}
			}
			else
			{
				//throw ArgExc("Invalid argument switch: '" + cmdline[i] + "'");
				// if it doesn't start with a delimiter, then assume base args
				break;
			}			
		}
		
		for(int j=0; j<_arg_sets.GetCount(); j++)
		{
			if(MatchesOptions(_arg_sets[j]) && MatchesBaseArgs(_arg_sets[j], cmdline, i))
			{
				_found_match = true;
				_arg_sets[j]->_is_match = true;
			}
		}
		
		if(!HasMatch())
		{
			throw ArgExc("Invalid argument specification.");
		}
		
	}
	catch(ArgExc &err)
	{
		_error=true;
		_error_str = "Error: " + err;		
		if(_print_usage)
		{
			(*_out_stream) << _error_str + " Reference program usage information...\n";	
			PrintUsageInformation(); 
		}
	}
}

bool ArgProcessor::MatchesBaseArgs(Ptr<ArgSet> & arg_set,const Vector<String> & cmdline, int idx)
{
	for(int i=0; i<arg_set->_base_args.GetCount(); i++)
	{
		Ptr<TArg> & pTBArg = arg_set->_base_args[i];
		for(int j=0; j<pTBArg->Max(); j++)
		{
			if(idx>=cmdline.GetCount())
			{
				if(pTBArg->IsRequired() && !pTBArg->IsSet())
				{
					return false;
				}
				break;
			}
			
			try
			{
				pTBArg->Val(cmdline[idx]);
			}
			catch(ArgExc&)
			{
				if( pTBArg->IsSet() || (j==0 && !pTBArg->IsRequired()) ) { break; }
				else { return false; }
			}
			idx++;
		}		
	}
	if(idx==cmdline.GetCount())
	{
		return true;
	}
	return false;
}

bool ArgProcessor::MatchesOptions(Ptr<ArgSet> arg_set)
{
	for(int i=0; i<arg_set->_options.GetCount(); i++)
	{
		for(int j=0; j<_arg_list.GetCount(); j++)
		{
			if(arg_set->_options[i] == _arg_list[j])
			{
				return (arg_set->_options[i]->IsRequired() && !arg_set->_options[i]->IsSet() ? false : true);
			}
			else if(_arg_list[j]->IsSet())
			{
				return false;
			}
		}		
	}
	return true;
}

String ArgProcessor::UsageInformation()
{
	String usage_info("");
	usage_info << VersionInformation() << "\nUsage Information:\n\n";
	for(int i=0;i<_arg_sets.GetCount();i++)
	{
		usage_info << Format("%d. %s ", i+1, GetFileName(GetExeFilePath())); //GetExeTitle()
		//String details("\nArgument Details:\n");
		Vector<int> widths = GetArgSetWidths(_arg_sets[i]);
		String details("");
		if(_arg_sets[i]->_options.GetCount()>0)
		{
			details << "   Options:\n";
			for(int j=0; j<_arg_sets[i]->_options.GetCount(); j++)
			{
				Ptr<TOption> pTDArg = _arg_sets[i]->_options[j];
				details << GetArgDetails(pTDArg, widths);
				usage_info << GetArgUsage(pTDArg);
				usage_info << " ";
			}
		}
		if(_arg_sets[i]->_base_args.GetCount()>0)
		{
			details << "   Arguments:\n";
			for(int k=0; k<_arg_sets[i]->_base_args.GetCount(); k++)
			{
				Ptr<TArg> pTBArg = _arg_sets[i]->_base_args[k];
				details << GetArgDetails(pTBArg, widths);
				usage_info << GetArgUsage(pTBArg);
			}
		}
		if(details.GetCount() > 0)
			{usage_info << Format("\n%s\n", details);}
	}
	usage_info << GetHelpUsage();
	return usage_info;
}

String ArgProcessor::UsageInformationCondensed()
{
	String usage_info("");
	usage_info << VersionInformation() << "\nUsage Information:\n";	
	for(int i=0;i<_arg_sets.GetCount();i++)
	{
		usage_info << Format("%d. %s ", i+1, GetFileName(GetExeFilePath()));
		for(int j=0; j<_arg_sets[i]->_options.GetCount(); j++)
		{
			Ptr<TOption> pTDArg = _arg_sets[i]->_options[j];
			usage_info << GetArgUsage(pTDArg);
		}
		for(int k=0; k<_arg_sets[i]->_base_args.GetCount(); k++)
		{
			Ptr<TArg> pTBArg = _arg_sets[i]->_base_args[k];
			usage_info << GetArgUsage(pTBArg);
		}
		usage_info << "\n";//\n
	}
	
	Vector<int> widths = GetGlobalWidths();
	for(int k=0; k<_arg_list.GetCount(); k++)
	{
		usage_info << GetArgDetails(_arg_list[k], widths);
	}
	//usage_info << "\n";
	for(int k=0; k<_base_arg_list.GetCount(); k++)
	{
		usage_info << GetArgDetails(_base_arg_list[k], widths);
	}
	//usage_info << "\n";
	return usage_info;
}

Vector<int> ArgProcessor::GetArgSetWidths(Ptr<ArgSet> arg_set)
{
	Vector<int> widths;
	widths.SetCount(6);
	widths.Insert(0,0,6);
	for(int k=0; k<arg_set->_options.GetCount(); k++)
	{
		widths[0] = max(widths[4], arg_set->_options[k]->GetShortFlag().GetCount());
		widths[1] = max(widths[5], arg_set->_options[k]->GetLongFlag().GetCount());
		widths[2] = max(widths[4], arg_set->_options[k]->GetTitle().GetCount());
		widths[3] = max(widths[5], arg_set->_options[k]->GetDescription().GetCount());
	}
	for(int k=0; k<arg_set->_base_args.GetCount(); k++)
	{
		widths[4] = max(widths[4], arg_set->_base_args[k]->GetTitle().GetCount());
		widths[5] = max(widths[5], arg_set->_base_args[k]->GetDescription().GetCount());
	}
	return widths;
}

String ArgProcessor::GetArgUsage(Ptr<TOption> pTDArg)
{
	String usage_info("");
	if(!pTDArg->IsRequired()){ usage_info << " ["; }
	usage_info << Format("%s%s", _short_delim, pTDArg->GetShortFlag());
	if(pTDArg->RequiresValue())
		{ usage_info << Format(" <%s>", pTDArg->GetTitle());}
	if(pTDArg->Max() > pTDArg->Min())
		{usage_info << " ...";}
	if(!pTDArg->IsRequired()){ usage_info << "]"; }
	return usage_info;
}

String ArgProcessor::GetArgUsage(Ptr<TArg> pTBArg)
{
	String usage_info("");
	if(pTBArg->Max() <= 1)
	{
		String strbound("[%s]");
		if(pTBArg->IsRequired()) { strbound = "<%s>"; }
		usage_info << Format( Format(" %s", strbound) , pTBArg->GetTitle());
	}
	else
	{
		if(!pTBArg->IsRequired()){ usage_info << " ["; }
		
		if(pTBArg->Min() < 3)
		{
			usage_info << Format("<%s>", pTBArg->GetTitle()+"1");
			for(int imin=1; imin<pTBArg->Min(); imin++)
				usage_info << Format(" <%s>", pTBArg->GetTitle()+IntStr(imin+1));
		}
		else 
		{
			usage_info << Format("<%s> ...", pTBArg->GetTitle()+"1");
			usage_info << Format(" <%s>", pTBArg->GetTitle()+IntStr(pTBArg->Min()));
		}

		if(pTBArg->Min() < pTBArg->Max())	
		{
			if(pTBArg->Max()-pTBArg->Min() < 2)
			{
				usage_info << Format(" [%s]", pTBArg->GetTitle()+IntStr(pTBArg->Min()+1));
				for(int imax=pTBArg->Min()+1; imax<pTBArg->Max(); imax++)
					usage_info << Format(" [%s]", pTBArg->GetTitle()+IntStr(imax+1));
			}
			else
			{
				//usage_info << Format(" [%s] ...", pTBArg->GetTitle()+IntStr(pTBArg->Min()+1));
				usage_info << " ...";
				if(pTBArg->Max()<INT_MAX)
					usage_info << Format(" [%s]", pTBArg->GetTitle()+IntStr(pTBArg->Max()));
			}
		}
		
		if(!pTBArg->IsRequired()){ usage_info << "]"; }
	}
	return usage_info;
}

String ArgProcessor::GetArgDetails(Ptr<TOption> pTDArg, const Vector<int> & widths)
{
	String desc = pTDArg->GetDescription();
	//int MAX_W = 20;
	
	String details("");
	details 	<< Format("      %*<s, ", widths[0]+_short_delim.GetCount(), _short_delim+pTDArg->GetShortFlag()) \
				<< Format("%*<s :  ", widths[1]+_long_delim.GetCount(), _long_delim+pTDArg->GetLongFlag()) \
				<< Format("%*<s\n", widths[3], desc); //desc.Left(MAX_W)
	/*						
	int w = widths[3]-MAX_W;	
	desc.Remove(0, desc.GetCount() > MAX_W ? MAX_W : desc.GetCount());
	while(w/MAX_W>0)
	{
		Cout() << "HERE\n";
		details << Format( "      %*<s, %*<s :  %*<s\n", widths[0], " ", widths[1], " ", w, desc.Left(MAX_W) );
		//details << Format("      %*<s :  %*<s\n", widths[3], " ", w, desc.Left(MAX_W));
		desc.Remove(0, desc.GetCount() > MAX_W ? MAX_W : desc.GetCount() );
		w -= MAX_W;
	}
	*/
	if(pTDArg->Max()>1)
	{
		String range_desc(""), can_must("must");
		if(!pTDArg->IsRequired()){ can_must = "can"; }
		range_desc << "This option " << can_must << " be specified " << pTDArg->Min();
		if(pTDArg->Min() < pTDArg->Max())	
		{
			if(pTDArg->Max()<INT_MAX)
				range_desc << " to " << pTDArg->Max();
			else
				range_desc << " to an unlimited number of";
		}
		range_desc << " times.";
		details 	<< Format("      %*<s  ", widths[0]+_short_delim.GetCount(), "") \
					<< Format("%*<s    ", widths[1]+_long_delim.GetCount(), "") \
					<< Format("%s\n", range_desc);
	}
	return details;
}

String ArgProcessor::GetArgDetails(Ptr<TArg> pTBArg, const Vector<int> & widths)
{
	String details("");	
	String desc = pTBArg->GetDescription();
	//int MAX_W = 60;
	
	details << Format("      %*<s :  ", widths[4], pTBArg->GetTitle()) \
			<< Format("%*<s\n", widths[5], desc); // desc.Left(MAX_W)
	/*
	int w = widths[5]-MAX_W;
	desc.Remove(0, desc.GetCount() > MAX_W ? MAX_W : desc.GetCount());
	while(w/MAX_W>1)
	{
		details << Format("      %*<s :  %*<s\n", widths[4], " ", w, desc.Left(MAX_W));
		desc.Remove(0, desc.GetCount() > MAX_W ? MAX_W : desc.GetCount() );
		w -= MAX_W;
	}
	*/
	if(pTBArg->Max()>1)
	{
		String range_desc(""), can_must("must");
		if(!pTBArg->IsRequired()){ can_must = "can"; }
		range_desc << "This argument " << can_must << " be specified " << pTBArg->Min();
		if(pTBArg->Min() < pTBArg->Max())	
		{
			if(pTBArg->Max()<INT_MAX)
				range_desc << " to " << pTBArg->Max();
			else
				range_desc << " to an unlimited number of";
		}
		range_desc << " times.";
		details << Format("      %*<s    %s\n", widths[4], "", range_desc);
	}
	return details;
}


Vector<int> ArgProcessor::GetGlobalWidths()
{
	Vector<int> widths;
	widths.SetCount(6);
	widths.Insert(0,0,6);
	for(int k=0; k<_arg_list.GetCount(); k++)
	{
		widths[0] = max(widths[0], _arg_list[k]->GetShortFlag().GetCount());
		widths[1] = max(widths[1], _arg_list[k]->GetLongFlag().GetCount());
		widths[2] = max(widths[2], _arg_list[k]->GetTitle().GetCount());
		widths[3] = max(widths[3], _arg_list[k]->GetDescription().GetCount());
	}
	for(int k=0; k<_base_arg_list.GetCount(); k++)
	{
		widths[4] = max(widths[4], _base_arg_list[k]->GetTitle().GetCount());
		widths[5] = max(widths[5], _base_arg_list[k]->GetDescription().GetCount());
	}
	return widths;
}

String ArgProcessor::GetHelpUsage()
{
	String usage_info("");
	String helpsflag("h"), helplflag("help"), helpdetails("Displays help information.");
	String versionsflag("v"), versionlflag("version"), versiondetails("Displays program version information.");
	
	int	col1 = max(helpsflag.GetCount(), versionsflag.GetCount()),\
		col2 = max(helplflag.GetCount(), versionlflag.GetCount()),\
		col3 = max(helpdetails.GetCount(), versiondetails.GetCount());
		
	usage_info	<< "   Other Options:\n" \
				<< Format("      %*<s, ", col1+_short_delim.GetCount(), _short_delim+helpsflag) \
				<< Format("%*<s :  ", col2+_long_delim.GetCount(), _long_delim+helplflag) \
				<< Format("%*<s\n", col3, helpdetails);
	usage_info	<< Format("      %*<s, ", col1+_short_delim.GetCount(), _short_delim+versionsflag) \
				<< Format("%*<s :  ", col2+_long_delim.GetCount(), _long_delim+versionlflag) \
				<< Format("%*<s\n", col3, versiondetails);
	return usage_info;
}

void ArgProcessor::PrintUsageInformation(bool condensed)
{
	if(_out_stream->IsOpen())
	{
		if(condensed) { (*_out_stream) << UsageInformationCondensed(); }
		else{ (*_out_stream) << UsageInformation(); }
	}
}

String ArgProcessor::VersionInformation()
{
	return Format("%s Version %s", _program_title, _version);
}

void ArgProcessor::PrintVersionInformation()
{
	if(_out_stream->IsOpen()){
		(*_out_stream) << VersionInformation() << "\n";
	}
}

void ArgProcessor::SetArgDelim(String short_delim, String long_delim)
{
	if(short_delim.GetCount()>0)
	{
		_short_delim = short_delim;
	}
	if(long_delim.GetCount()>0)
	{
		_long_delim = long_delim;
	}
}
		
void ArgProcessor::PrintToStream(Stream & ss){ _out_stream = &ss; }

//usage_info << Format("%s Version %s\nUsage Information:\n", _program_title, _version); //\n
//usage_info << Format("%s  Version %s\nUsage Information:\n\n", _program_title, _version);

//From GetArgUsage(pTDArg)
//if(pTDArg->RequiresValue())
//	{ usage_info << Format("%s%s <%s> ", _short_delim, pTDArg->GetShortFlag(), pTDArg->GetTitle()); }
//else
//	{ usage_info << Format("%s%s ", _short_delim, pTDArg->GetShortFlag()); }
