#include <Core/Core.h>
#include <CmdLineArgProcessor/ArgProcessor.hpp>

using namespace Upp;

void Example1()
{
	const Vector<String>& cmdline = CommandLine();
	ArgProcessor ap("CmdLineArgProcessor Example", "1");
	Option<bool> numbers_option ("Numbers", "If this switch is specified, the program will expect a list of integers as arguments.", true);
	MultiArg<int> numbers ("Number", "An integer number.", true);
	MultiArg<String> strings ("String", "A string input.", true);
	
	ArgSet set1, set2;
	set1 << numbers_option << numbers;
	set2 << strings;
	ap << set1 << set2;
	ap.ProcessCmdLine(cmdline);
	
	if(ap.HasMatch())
	{
		if(set1.IsMatch())
		{
			Cout() << "set1 matches the user's input\n";
			Cout() << "Numbers:\n";
			for(int i=0;i<numbers.GetCount();i++)
			{
				Cout() << numbers[i] << "\n";
			}
			Cout() << "\n";
		}
		else if(set2.IsMatch())
		{
			Cout() << "set2 matches the user's input\n";
			Cout() << "Strings:\n";
			for(int i=0;i<strings.GetCount();i++)
			{
				Cout() << strings[i] << "\n";
			}
			Cout() << "\n";
		}
	}
	else if(ap.IsError())
	{
		ap.PrintUsageInformation();
	}	
}

// Example 2: Set verbosity switch
/* Output of PrintUsageInformation():
CmdLineArgProcessor Example Version 2
Usage Information:
1. CmdLineArgProcessorExample  [-V ]
      -V, --verbose :  Define how verbose the output should be.
                       This option can be specified 1 to 3 times.
*/

void Example2()
{
	ArgProcessor ap("CmdLineArgProcessor Example", "2");
	MultiOption<bool> verbosity ("V", "verbose", "Verbosity", "Define how verbose the output should be.", false, 1, 3);
	// This reads "define a command line option that can be specified as "-V" or "--verbose", it is not required, and can be specified 1 to 3 times."
	ArgSet set1;
	set1.Add(verbosity);
	ap.AddArgSet(set1);
	ap.ProcessCmdLine(CommandLine());
	
	if(set1.IsMatch())
	{
		switch((int)verbosity.Val())
		{
			case 3:
				Cout() << "VERY VERY VERBOSE\n";
				break;
			case 2: 
				Cout() << "Very Verbose\n";
				break;
			case 1: 
				Cout() << "Slightly verbose\n";
				break;
			case 0:
				Cout() << "I shouldn't even be outputting this, shhh.\n";
		}
	}
	
	if(ap.IsError())
	{
		Cout() << ap.GetError() << "\n";
		ap.PrintUsageInformation();
	}
}

// Example 3a: determine calculating average vs summation.
/* Output of PrintUsageInformation():
CmdLineArgProcessor Example Version 3a
Usage Information:
1. CmdLineArgProcessorExample -m <Method> <Number1> ...
      -m, --method :  Can either be 'average' to calculate the average or 'sum' to calculate the sum of the numbers.
      Number :  An number.
                This argument must be specified 1 to an unlimited number of times.
*/
void Example3a()
{
	ArgProcessor ap("CmdLineArgProcessor Example", "3a");
	Option<String> method_opt ("Method", "Can either be 'average' to calculate the average or 'sum' to calculate the sum of the numbers.", true);
	MultiArg<double> numbers ("Number", "An number.", true);	
	ArgSet set1;
	set1 << method_opt << numbers;
	ap.AddArgSet(set1);
	ap.ProcessCmdLine(CommandLine());
	
	if(!ap.IsError() && set1.IsMatch())
	{
		double answer=0;
		if(ToLower(method_opt.Val().ToString()).IsEqual("sum"))
		{
			for(int i=0;i<numbers.GetCount();i++)
			{
				answer += (double) numbers[i];
			}
			Cout() << answer << "\n";
		}
		else if(ToLower(method_opt.Val().ToString()).IsEqual("average"))
		{
			for(int i=0;i<numbers.GetCount();i++)
			{
				answer += (double) numbers[i];
			}
			answer = answer/numbers.GetCount();
			Cout() << answer << "\n";
		}
		else
		{
			Cout() << "Invalid method.\n";
			ap.PrintUsageInformation();
		}
	}
	else if(ap.IsError())
	{
		ap.PrintUsageInformation();
	}
}

// Example 3b does the same thing as 3a above, but uses switch options instead to 
/*  Output of PrintUsageInformation():
Usage Information:
1. CmdLineArgProcessorExample -s <Number1> ...
2. CmdLineArgProcessorExample -a <Number1> ...
      -s, --summation :  If specified, program will calculate the sum of the numbers.    
      -a, --average   :  If specified, program will calculate the average of the numbers.
      Number :  An number.
                This argument must be specified 1 to an unlimited number of times.
*/
void Example3b()
{
	ArgProcessor ap("CmdLineArgProcessor Example", "3b");
	Option<bool> sum_opt ("Summation", "If specified, program will calculate the sum of the numbers.", true);
	Option<bool> avg_opt ("Average", "If specified, program will calculate the average of the numbers.", true);
	MultiArg<double> numbers ("Number", "An number.", true);
	ArgSet set1_sum, set2_avg;
	set1_sum << sum_opt << numbers;
	set2_avg << avg_opt << numbers;
	ap << set1_sum << set2_avg;
	ap.ProcessCmdLine(CommandLine());
	
	if(ap.HasMatch())
	{
		double answer = 0;
		if(set1_sum.IsMatch())
		{
			for(int i=0;i<numbers.GetCount();i++)
			{
				answer += (double) numbers[i];
			}
			Cout() << answer << "\n";
		}		
		else if(set2_avg.IsMatch())
		{
			for(int i=0;i<numbers.GetCount();i++)
			{
				answer += (double) numbers[i];
			}
			answer = answer/numbers.GetCount();
			Cout() << answer << "\n";
		}
	}
	else if(ap.IsError())
	{
		Cout() << ap.GetError() << "\n";
		ap.PrintUsageInformation();
	}
}

CONSOLE_APP_MAIN
{
	// Example1();
	 Example2();
	// Example3a();
	// Example3b();
}
