#ifndef _pycpp_h
#define _pycpp_h
//---------------------------------------------------------------------------

#ifdef _DEBUG
  #undef _DEBUG
  #include <Python.h>
  #define _DEBUG
#else
  #include <Python.h>
#endif

#include <vector>
using namespace std;

#include "../recdef/recdef.h"

//---------------------------------------------------------------------------
// forward deklaracie

class PxObject;
class PxString;
class PxSequence;
class PxIter;
class PxTuple;
class PxDict;

//---------------------------------------------------------------------------
#define ACQUIRE_PX_THREAD PyGILState_STATE pyGILState = PyGILState_Ensure()
#define ACQUIRE_PX_THREAD_ND pyGILState = PyGILState_Ensure()

#define RELEASE_PX_THREAD PyGILState_Release(pyGILState)

#define BEGIN_PX_SECTION \
    { ACQUIRE_PX_THREAD; \
    try {
#define END_PX_SECTION \
    } \
    catch (PxException& e) { \
        LOG_PX_TRACEBACK; \
    } \
    catch (...) { \
        RELEASE_PX_THREAD; \
        throw; \
    } \
    RELEASE_PX_THREAD; }

//---------------------------------------------------------------------------
const bool OWNED = true;

//---------------------------------------------------------------------------
class PxException {
public:
    PxException(void);
    PxException(const PxException& obj);
    ~PxException(void);
    void restore(void);
    PxObject getType(void);
    PxObject getValue(void);
    const char* getStr(void);
    void printTraceback(void);
protected:
    PyObject* ptype;
    PyObject* pvalue;
    PyObject* ptraceback;
};

//---------------------------------------------------------------------------
class PxObject {
public:
    // konstruktor bez parametrov je nebezpecny!
    PxObject(void);
    // owned = true znamena nezvysovat pocet odkazov na objekt
    PxObject(PyObject* pyObj, bool owned = false);
    PxObject(const PxObject& obj);
    // destruktor
    ~PxObject(void);
    // zvysi/znizi pocet odkazov na objekt
    void IncRef(void) const;
    void DecRef(void);
    // operatory priradenia
    PxObject& operator=(PyObject* pyObj);
    PxObject& operator=(const PxObject& obj);
    
    // testuje moznost priradenia vzhladom na typ objektu
    bool assignable(const PxObject& obj);
    virtual bool assignablePy(PyObject* pyObj);

    // vrati prislusny PyObject* smernik
    PyObject* ptr(void) const;
    // vrati prislusny PyObject* smernik a zvysi pocet odkazov
    PyObject* ptrIncRef(void) const;
    // zneplatni odkaz na objekt
    void invalidate(void);

    //----------------------------------------------------------
    // metody kazdeho objektu typu PyObject
    bool HasAttr(const char* attr_name) const;
    PxObject GetAttr(const char* attr_name) const;
    PxObject operator()(const char* attr_name) const;
    void SetAttr(const char* attr_name, const PxObject& obj) const;
    void operator()(const char* attr_name, const PxObject& obj) const;
    bool IsEq(const PxObject& obj) const;
    PxString Str(void) const;
    bool IsInstance(const PxObject& cls) const;
    bool IsCallable(void) const;
    PxObject Call(const PxTuple& args, const PxDict& kw) const;
    PxObject Call(const PxTuple& args) const;
    PxObject Call(void) const;
    bool IsTrue(void) const;
    bool IsFalse(void) const;
    PxIter GetIter(void) const;
    
    #define PAR(I) const PxObject& i##I
    #define COM(I) SetItem(I - 1, i##I)
    #define MET(I) \
    PxObject CallArgs(EXPPAR##I(PAR));
    EXPMET(MET)
    #undef PAR
    #undef COM
    #undef MET
    
    //----------------------------------------------------------
    // mnou doplnene metody - testy na podtyp typu PyObject
    bool IsBool() const;
    bool IsInt() const;
    bool IsLong() const;
    bool IsFloat() const;
    bool IsComplex() const;
    bool IsString() const;
    bool IsUnicode() const;
    bool IsTuple() const;
    bool IsList() const;
    bool IsDict() const;

    //----------------------------------------------------------
    // mnou doplnene metody - konverzie na zakladne typy
    bool AsBool(void) const;
    long AsLong(void) const;
    unsigned long AsUnsignedLong(void) const;
    double AsDouble(void) const;
    const char* AsString(void) const;

protected:
    PyObject* p;

    // retazec typovej vynimky ako metoda
    virtual const char* getTypeExcMsg(void);
    // vyvola typovu vynimku
    void typeExc(void);
    // vyvola typovu vynimku alebo nerobi nic na zaklade podmienky
    void typeExcCond(bool cond);
    // pred priradenim testuje moznost priradenia vzhladom na typ
    void assignPy(PyObject* pyObj);
};

//---------------------------------------------------------------------------
class PxBool: public PxObject {
public:
    PxBool(PyObject* pyObj, bool owned = false):
        PxObject(pyObj, owned)
    {
        typeExcCond(!assignablePy(pyObj));
    }
    PxBool(const PxObject& obj): PxObject(obj) {
        typeExcCond(!assignable(obj));
    }
    explicit PxBool(bool value) {
        PyObject* pyObj = PyBool_FromLong(value);
        if (!pyObj)
            throw PxException();
        p = pyObj;
    }
    virtual bool assignablePy(PyObject* pyObj) {
        return pyObj && PyBool_Check(pyObj);
    }

    //----------------------------------------------------------
protected:
    virtual const char* getTypeExcMsg(void) {
        return "Must be Bool !";
    }
};

//---------------------------------------------------------------------------
class PxInt: public PxObject {
public:
    PxInt(PyObject* pyObj, bool owned = false):
        PxObject(pyObj, owned)
    {
        typeExcCond(!assignablePy(pyObj));
    }
    PxInt(const PxObject& obj): PxObject(obj) {
        typeExcCond(!assignable(obj));
    }
    explicit PxInt(long value) {
        PyObject* pyObj = PyInt_FromLong(value);
        if (!pyObj)
            throw PxException();
        p = pyObj;
    }
    virtual bool assignablePy(PyObject* pyObj) {
        return pyObj && PyInt_Check(pyObj);
    }

    //----------------------------------------------------------
    long AsLong(void) const {
        return PyInt_AsLong(p);
    }
    long AsUnsignedLong(void) const {
        long rl = PyInt_AsLong(p);
        if (rl >= 0)
            return rl;
        PyErr_SetString(PyExc_OverflowError, "Int must be >= 0 !");
        throw PxException();
    }

protected:
    virtual const char* getTypeExcMsg(void) {
        return "Must be Int !";
    }
};

//---------------------------------------------------------------------------
class PxLong: public PxObject {
public:
    PxLong(PyObject* pyObj, bool owned = false):
        PxObject(pyObj, owned)
    {
        typeExcCond(!assignablePy(pyObj));
    }
    PxLong(const PxObject& obj): PxObject(obj) {
        typeExcCond(!assignable(obj));
    }
    explicit PxLong(long value) {
        PyObject* pyObj = PyLong_FromLong(value);
        if (!pyObj)
            throw PxException();
        p = pyObj;
    }
    explicit PxLong(unsigned long value) {
        PyObject* pyObj = PyLong_FromUnsignedLong(value);
        if (!pyObj)
            throw PxException();
        p = pyObj;
    }
    virtual bool assignablePy(PyObject* pyObj) {
        return pyObj && PyLong_Check(pyObj);
    }

    //----------------------------------------------------------
    long AsLong(void) const {
        long rl = PyLong_AsLong(p);
        if (PyErr_Occurred())
            throw PxException();
        return rl;
    }
    unsigned long AsUnsignedLong(void) const {
        long rul = PyLong_AsUnsignedLong(p);
        if (PyErr_Occurred())
            throw PxException();
        return rul;
    }
    double AsDouble(void) const {
        double rd = PyLong_AsDouble(p);
        if (PyErr_Occurred())
            throw PxException();
        return rd;
    }

protected:
    virtual const char* getTypeExcMsg(void) {
        return "Must be Long !";
    }
};

//---------------------------------------------------------------------------
class PxFloat: public PxObject {
public:
    PxFloat(PyObject* pyObj, bool owned = false):
        PxObject(pyObj, owned)
    {
        typeExcCond(!assignablePy(pyObj));
    }
    PxFloat(const PxObject& obj): PxObject(obj) {
        typeExcCond(!assignable(obj));
    }
    explicit PxFloat(double value) {
        PyObject* pyObj = PyFloat_FromDouble(value);
        if (!pyObj)
            throw PxException();
        p = pyObj;
    }
    virtual bool assignablePy(PyObject* pyObj) {
        return pyObj && PyFloat_Check(pyObj);
    }

    //----------------------------------------------------------
    double AsDouble(void) const {
        return PyFloat_AsDouble(p);
    }

protected:
    virtual const char* getTypeExcMsg(void) {
        return "Must be Float !";
    }
};

//---------------------------------------------------------------------------
class PxString: public PxObject {
public:
    PxString(PyObject* pyObj, bool owned = false):
        PxObject(pyObj, owned)
    {
        typeExcCond(!assignablePy(pyObj));
    }
    PxString(const PxObject& obj): PxObject(obj) {
        typeExcCond(!assignable(obj));
    }
    explicit PxString(const char* value) {
        if (!value)
            typeExc();
        PyObject* pyObj = PyString_FromString(value);
        if (!pyObj)
            throw PxException();
        p = pyObj;
    }
    virtual bool assignablePy(PyObject* pyObj) {
        return pyObj && PyString_Check(pyObj);
    }

    //----------------------------------------------------------
    int Size(void) const {
        return PyString_Size(p);
    }
    const char* AsString(void) const {
        return PyString_AsString(p);
    }    
    
protected:
    virtual const char* getTypeExcMsg(void) {
        return "Must be PyString !";
    }
};

//---------------------------------------------------------------------------
class PxSequence: public PxObject {
public:
    // konstruktor bez parametrov je nebezpecny!
    PxSequence(void) {
    }
    PxSequence(PyObject* pyObj, bool owned = false):
        PxObject(pyObj, owned)
    {
        typeExcCond(!assignablePy(pyObj));
    }
    PxSequence(const PxObject& obj): PxObject(obj) {
        typeExcCond(!assignable(obj));
    }
    virtual bool assignablePy(PyObject* pyObj) {
        return pyObj && PySequence_Check(pyObj);
    }

    //----------------------------------------------------------
    int Size(void) const {
        return PySequence_Length(p);
    }
    PxObject Concat(const PxObject& obj) const {
        PyObject* pyObj = PySequence_Concat(p, obj.ptr());
        if (pyObj == NULL)
            throw PxException();
        return PxObject(pyObj, OWNED);
    }
    PxObject InPlaceConcat(const PxObject& obj) const {
        PyObject* pyObj = PySequence_InPlaceConcat(p, obj.ptr());
        if (pyObj == NULL)
            throw PxException();
        return PxObject(pyObj, OWNED);
    }
    PxObject GetItem(int index) const {
        PyObject* pyObj = PySequence_GetItem(p, index);
        if (pyObj == NULL)
            throw PxException();
        return PxObject(pyObj, OWNED);
    }
    PxObject operator()(int index) const {
        return GetItem(index);
    }
    void SetItem(int index, const PxObject& obj) {
        int ri = PySequence_SetItem(p, index, obj.ptr());
        if (ri == -1)
            throw PxException();
    }
    void operator()(int index, const PxObject& obj) {
        SetItem(index, obj);
    }

protected:
    virtual const char* getTypeExcMsg(void) {
        return "Must be PySequence !";
    }
};

//---------------------------------------------------------------------------
class PxIter: public PxObject {
public:
    PxIter(PyObject* pyObj, bool owned = false):
        PxObject(pyObj, owned)
    {
        typeExcCond(!assignablePy(pyObj));
    }
    PxIter(const PxObject& obj) {
        PyObject* pyObj = obj.ptr();
        if (assignablePy(pyObj)) {
            Py_IncRef(pyObj);
        }
        else {
            pyObj = PyObject_GetIter(pyObj);
            if (pyObj == NULL) {
                PyErr_Clear();
                typeExcCond(true);
            }
        }
        p = pyObj;
    }
    virtual bool assignablePy(PyObject* pyObj) {
        return pyObj && PyIter_Check(pyObj);
    }

    //----------------------------------------------------------
    bool next(void) {
        PyObject* pyObj = PyIter_Next(p);
        if (pyObj == NULL) {
            if (PyErr_Occurred())
                throw PxException();
            else
                return false;
        }
        else {
            item = PxObject(pyObj, OWNED);
            return true;
        }
    }
    PxObject get(void) {
        return item;
    }
protected:
    PxObject item;
    
    virtual const char* getTypeExcMsg(void) {
        return "Must be PyIter or iterable PyObject !";
    }
};

//---------------------------------------------------------------------------
class PxList: public PxObject {
public:
    PxList(PyObject* pyObj, bool owned = false):
        PxObject(pyObj, owned)
    {
        typeExcCond(!assignablePy(pyObj));
    }
    PxList(const PxObject& obj): PxObject(obj) {
        typeExcCond(!assignable(obj));
    }
    explicit PxList(int len) {
        p = PyList_New(len);
        if (!p)
            throw PxException();
    }
    virtual bool assignablePy(PyObject* pyObj) {
        return pyObj && PyList_Check(pyObj);
    }

    //----------------------------------------------------------
    void SetItem(int index, const PxObject& obj) {
        obj.IncRef();
        if (PyList_SetItem(p, index, obj.ptr()) == -1)
            throw PxException();
    }
    void operator()(int index, const PxObject& obj) {
        SetItem(index, obj);
    }
    void Insert(int index, const PxObject& obj) {
        if (PyList_Insert(p, index, obj.ptr()) == -1)
            throw PxException();
    }
    void Append(const PxObject& obj) {
        if (PyList_Append(p, obj.ptr()) == -1)
            throw PxException();
    }
};

#define PAR(I) const PxObject& i##I
#define MET(I) \
PxList PxListFromArgs(EXPPAR##I(PAR));
EXPMET(MET)
#undef PAR
#undef MET

//---------------------------------------------------------------------------
class PxTuple: public PxObject {
public:
    PxTuple(PyObject* pyObj, bool owned = false):
        PxObject(pyObj, owned)
    {
        typeExcCond(!assignablePy(pyObj));
    }
    PxTuple(const PxObject& obj): PxObject(obj) {
        typeExcCond(!assignable(obj));
    }
    explicit PxTuple(int len) {
        p = PyTuple_New(len);
        if (!p)
            throw PxException();
    }
    virtual bool assignablePy(PyObject* pyObj) {
        return pyObj && PyTuple_Check(pyObj);
    }
    
    //----------------------------------------------------------
    void SetItem(int index, const PxObject& obj) {
        obj.IncRef();
        if (PyTuple_SetItem(p, index, obj.ptr()) == -1)
            throw PxException();
    }
    void operator()(int index, const PxObject& obj) {
        SetItem(index, obj);
    }
};

#define PAR(I) const PxObject& i##I
#define MET(I) \
PxTuple PxTupleFromArgs(EXPPAR##I(PAR));
EXPMET(MET)
#undef PAR
#undef MET

//---------------------------------------------------------------------------
class PxDict: public PxObject {
public:
    PxDict(PyObject* pyObj, bool owned = false):
        PxObject(pyObj, owned)
    {
        typeExcCond(!assignablePy(pyObj));
    }
    PxDict(const PxObject& obj): PxObject(obj) {
        typeExcCond(!assignable(obj));
    }
    explicit PxDict(void) {
        p = PyDict_New();
        if (!p)
            throw PxException();
    }
    virtual bool assignablePy(PyObject* pyObj) {
        return pyObj && PyDict_Check(pyObj);
    }

    //----------------------------------------------------------
};

//---------------------------------------------------------------------------
class PxCObject: public PxObject {
public:
    PxCObject(PyObject* pyObj, bool owned = false):
        PxObject(pyObj, owned)
    {
        typeExcCond(!assignablePy(pyObj));
    }
    PxCObject(const PxObject& obj): PxObject(obj) {
        typeExcCond(!assignable(obj));
    }
    explicit PxCObject(void* cobj, void(*destr)(void*)) {
        if (!cobj)
            typeExc();
        PyObject* pyObj = PyCObject_FromVoidPtr(cobj, destr);
        if (!pyObj)
            throw PxException();
        p = pyObj;
    }
    explicit PxCObject(void* cobj, void* desc, void(*destr)(void*, void*)) {
        if (!cobj)
            typeExc();
        PyObject* pyObj = PyCObject_FromVoidPtrAndDesc(cobj, desc, destr);
        if (!pyObj)
            throw PxException();
        p = pyObj;
    }
    virtual bool assignablePy(PyObject* pyObj) {
        return pyObj && PyCObject_Check(pyObj);
    }

    //----------------------------------------------------------
    void* AsVoidPtr(void) const {
        return PyCObject_AsVoidPtr(p);
    }
    void* GetDesc(void) const {
        return PyCObject_GetDesc(p);
    }
    
    //----------------------------------------------------------
    // mnou doplnene metody - konverzie na typ s popisom desc
    void* getCObj(void* desc, const char* errMsg = "Desc mismatch in getCObj !") {
        if (GetDesc() != desc) {
            PyErr_SetString(PyExc_TypeError, errMsg);
            throw PxException();
        }
        else {
            return AsVoidPtr();
        }
    }
    
protected:
    virtual const char* getTypeExcMsg(void) {
        return "Must be PyCObject !";
    }   
};

//---------------------------------------------------------------------------
void PxRun_SimpleString(const char *command);
PxObject PxImport_AddModule(const char *name);

//---------------------------------------------------------------------------
template<typename T1, class T2> void fromPxSequence(
    const PxSequence& xSeq, vector<T1>& ret,
    T2 (PxObject::*getter)(void) const
)
{
    int len = xSeq.Size();
    ret.resize(len);
    for (int i = 0; i < len; i++)
        ret[i] = static_cast<T1>((xSeq.GetItem(i).*getter)());
}

template<typename T1, class T2> PxList fromVector(
    const vector<T1>& v, const int len
)
{
    PxList xL(len);
    for (int i = 0; i < len; i++) {
        T2 xValue(v[i]);
        xL.SetItem(i, xValue);
    }
    return xL;
}

template<typename T1, class T2> PxList fromVector2(
    const vector< vector<T1> >& vv, const int len
)
{
    PxList xL(len);
    for (int i = 0; i < len; i++) {
        int len2 = vv[i].size();
        PxList xL2(len2);
        for (int j = 0; j < len2; j++) {
            T2 xValue(vv[i][j]);
            xL2.SetItem(j, xValue);
        }
        xL.SetItem(i, xL2);
    }
    return xL;
}

//---------------------------------------------------------------------------
#endif
