#include "connsrv.h"

#define TFILE <connsrv/connsrv.t>
#include <Core/t.h>

//---------------------------------------------------------------------------
// ConnsTable

Id idId("idId");
Id idType("idType");
Id idDescr("idDescr");
Id idLocalPort("idLocalPort");
Id idRemotePort("idRemotePort");
Id idRoute("idRoute");
Id idState("idState");
Id idRT("idRT");
Id idEnabled("idEnabled");

ConnsTable::ConnsTable(void)
{
    dropType.Add("S");
    dropType.Add("C");
    
    dropType << THISBACK(dropTypeChanged);
    
    //----------------------------------------------------------
    arrayConns.HeaderObject().Moving(false).Absolute();
    
    arrayConns.AddColumn(idId, "ID");
    arrayConns.AddColumn(idType, t_("Type")).Edit(dropType);
    arrayConns.AddColumn(idDescr, t_("Description")).Edit(editDescr);
    arrayConns.AddColumn(idLocalPort, t_("Local port")).Edit(editLocalPort);
    arrayConns.AddColumn(idRemotePort, t_("Remote port")).Edit(editRemotePort);
    arrayConns.AddColumn(idRoute, t_("Route")).Edit(editRoute);
    arrayConns.AddColumn(idState, t_("State"));
    arrayConns.AddColumn(idRT, t_("R/T"));
    arrayConns.AddColumn(idEnabled, t_("Activate"));
    
    arrayConns.ColumnWidths("30 50 150 75 150 75 75 60 75");
    
    arrayConns.NoCursor();
    arrayConns.SetEditable(false);
    arrayConns.WhenBar = THISBACK(contextMenu);
    
    arrayConns.WhenStartEdit = THISBACK(whenStartEdit);
    arrayConns.WhenAcceptRow = THISBACK(whenAcceptRow);
    arrayConns.WhenUpdateRow = THISBACK(whenUpdateRow);
        
    //----------------------------------------------------------
    buttonResetCnts << THISBACK(buttonResetCntsClick);    
    optionEditable << THISBACK(optionEditableChanged);
    buttonSaveChanges << THISBACK(buttonSaveChangesClick);
    buttonActivate << THISBACK(buttonActivateClick);
    buttonDeactivate << THISBACK(buttonDeactivateClick);

    buttonActivate.Disable();
    buttonDeactivate.Enable();
    optionEditable.Set(false);
    optionEditable.Disable();
    buttonSaveChanges.Disable();
};

void ConnsTable::addNewItem(int id, const PxObject& xConn)
{
    if (xConn.IsInstance(xTcpServer)) {                     // TcpServer ?
        arrayConns.Add(
            id,
            xConn("typeStr").AsString(),
            xConn("descrStr").AsString(),
            Format("%d", 
                xConn("localPort").AsLong()
            ),
            "",
            xConn("connIndsToSend").Str().AsString(),
            "",
            "",
            xConn("enabled").AsBool()
        );
    }
    else
    if (xConn.IsInstance(xTcpClient)) {                     // TcpClient ?
        arrayConns.Add(
            id,
            xConn("typeStr").AsString(),
            xConn("descrStr").AsString(),
            "",
            Format("%s:%d", 
                xConn("remoteIP").AsString(), xConn("remotePort").AsLong()
            ),
            xConn("connIndsToSend").Str().AsString(),
            "",
            "",
            xConn("enabled").AsBool()
        );
    }
    Option* option = new Option();
    optionsEnabled.Add(option);
    option->WantFocus(false).Disable();
    option->WhenAction = THISBACK1(whenOptionEnabledChanged, option);
    arrayConns.SetCtrl(
        id, arrayConns.GetPos(idEnabled), option
    );
}

void ConnsTable::init(const PxObject& xConnsTable_)
{   
    BEGIN_PX_SECTION
    
    xConnsTable = xConnsTable_;
    
    int i = 0;
    for(PxIter it = xConnsTable("conns"); it.next(); i++) {
        PxObject xConn = it.get();
        addNewItem(i, xConn);
    }
    
    xConnsTable("activate").Call();
    
    END_PX_SECTION
}

void ConnsTable::tstChanged(void)
{
    BEGIN_PX_SECTION
    
    int i = 0;
    for(PxIter it = xConnsTable("conns"); it.next(); i++) {
        PxObject xConn = it.get();
        
        PxSequence xChanged = xConn("getChanged").Call();
        bool changed = xChanged(0).AsBool();
        int64 receivedCnt = xChanged(1).AsUnsignedLong();
        int64 sentCnt = xChanged(2).AsUnsignedLong();
        
        if (changed) {
            bool connected = xConn("connected").AsBool();
        
            switch (connected) {
            case false:
                arrayConns.Set(i, idState, "not conn.");
                break;
            case true:
                arrayConns.Set(i, idState, "conn.");
                break;
            }
            
            arrayConns.Set(i, 7,
                Format("%ld / %ld", receivedCnt, sentCnt)
            );
            
            if (xConn.IsInstance(xTcpServer)) {                 // TcpServer ?
                if (connected)
                    arrayConns.Set(i, idRemotePort,
                        Format("%s:%d", 
                            xConn("remoteIP").AsString(), xConn("remotePort").AsLong()
                        )
                    );
                else
                    arrayConns.Set(i, idRemotePort, "");
            }
            else
            if (xConn.IsInstance(xTcpClient)) {                 // TcpClient ?              
                if (connected)
                    arrayConns.Set(i, idLocalPort,
                        xConn("localPort").AsLong()
                    );
                else
                    arrayConns.Set(i, idLocalPort, "");
            }
        }
    }
    
    END_PX_SECTION
}

//--------------------------------------------------------------
void ConnsTable::buttonResetCntsClick(void) {
    BEGIN_PX_SECTION
    
    xConnsTable("resetCnts").Call();
    
    END_PX_SECTION
}

void ConnsTable::optionEditableChanged(void)
{
    buttonActivate.Disable();
    optionEditable.Disable();
    buttonSaveChanges.Enable();
    
    for (int i = 0; i < optionsEnabled.GetCount(); i++)
        optionsEnabled[i]->Enable();
    
    arrayConns.SetEditable(true);
    arrayConns.NoCursor(false);
    arrayConns.SetCursor(0);
}

void ConnsTable::buttonSaveChangesClick(void) {
    BEGIN_PX_SECTION
    
    xConnsTable("saveCfg").Call();
    
    for (int i = 0; i < optionsEnabled.GetCount(); i++)
        optionsEnabled[i]->Disable();
    
    arrayConns.CancelCursor();
    arrayConns.NoCursor(true);
    arrayConns.SetEditable(false);
    
    optionEditable.Set(false);
    
    buttonActivate.Enable();
    optionEditable.Enable();
    buttonSaveChanges.Disable();

    END_PX_SECTION
}

void ConnsTable::buttonActivateClick(void) {
    BEGIN_PX_SECTION
    
    xConnsTable("activate").Call();
    
    buttonActivate.Disable();
    buttonDeactivate.Enable();
    optionEditable.Disable();
    
    END_PX_SECTION
}

void ConnsTable::buttonDeactivateClick(void) {
    BEGIN_PX_SECTION
    
    xConnsTable("deactivate").Call();

    buttonActivate.Enable();
    buttonDeactivate.Disable();
    optionEditable.Enable();
    
    END_PX_SECTION
}

//--------------------------------------------------------------
void ConnsTable::whenStartEdit(void)
{
    String typeStr = arrayConns.Get(idType);
    
    buttonSaveChanges.Disable();

    if (typeStr == "S") {
        editLocalPort.Enable();
        editRemotePort.Disable();
    }
    else
    if (typeStr == "C") {       
        editLocalPort.Disable();
        editRemotePort.Enable();
    }
}

bool ConnsTable::whenAcceptRow(void)
{
    int id = arrayConns.GetCursor();
    String typeStr = arrayConns.Get(id, idType);
    
    bool accept = false;
    
    BEGIN_PX_SECTION

    PxObject xRet = xConnsTable("setCfg").CallArgs(
        PxInt(id),
        PxString(String(arrayConns.Get(idType))),
        PxString(String(arrayConns.Get(idDescr))),
        PxString(String(arrayConns.Get(idLocalPort))),
        PxString(String(arrayConns.Get(idRemotePort))),
        PxString(String(arrayConns.Get(idRoute))),
        PxBool(bool(arrayConns.Get(idEnabled)))
    );
    accept = xRet.AsBool();
    
    END_PX_SECTION

    if (accept)
        buttonSaveChanges.Enable();
    else
        buttonSaveChanges.Disable();
    
    return accept;
}

void ConnsTable::whenOptionEnabledChanged(Option *self)
{   
    int id;
    
    for (id = 0; id < optionsEnabled.GetCount(); id++)
        if (optionsEnabled[id] == self)
            break;
    
    BEGIN_PX_SECTION

    PxObject xRet = xConnsTable("toggleEnabled").CallArgs(
        PxInt(id)
    );
    
    END_PX_SECTION
    
    arrayConns.SetCursor(id);
}

void ConnsTable::whenUpdateRow(void)
{
}

void ConnsTable::dropTypeChanged(void)
{
    String typeStr = dropType.Get();
    if (typeStr == "S") {
        editLocalPort.SetData("?");
        editLocalPort.Enable();
        editRemotePort.SetData("");
        editRemotePort.Disable();
        editRoute.SetData("[]");
    }
    else
    if (typeStr == "C") {
        editLocalPort.SetData("");
        editLocalPort.Disable();
        editRemotePort.SetData("?.?.?.?:?");
        editRemotePort.Enable();
        editRoute.SetData("[]");
    }
}

//--------------------------------------------------------------
void ConnsTable::contextMenu(Bar &bar) {
    bool e = arrayConns.IsEditable();
    bool c = !arrayConns.IsEdit() && e;
    bool d = c && arrayConns.IsCursor();

    int count = arrayConns.GetCount();
    int id = arrayConns.GetCursor();
    
    bar.Add(c, t_("Add new line"), THISBACK(onAddNewItem));
    bar.Add(d, t_("Remove line"), THISBACK1(onRemoveItem, id));
    bar.Add(d, t_("Edit line"), THISBACK1(onEditItem, id));
    bar.Add(d && id > 0, 
        t_("Move up"), THISBACK1(onMoveUp, id));
    bar.Add(d && id >= 0 && id < count - 1, 
        t_("Move down"), THISBACK1(onMoveDown, id));
}

void ConnsTable::onEditItem(int id) {
    arrayConns.DoEdit();
}

void ConnsTable::onRemoveItem(int id) {
    BEGIN_PX_SECTION
    
    xConnsTable("remove").CallArgs(PxInt(id));
    
    END_PX_SECTION
    
    arrayConns.DoRemove();
    for (int i = id; i < arrayConns.GetCount(); i++)
        arrayConns.Set(i, idId, i);
    optionsEnabled.Remove(id);
}

void ConnsTable::onAddNewItem(void) {
    BEGIN_PX_SECTION
    
    xConnsTable("addNewItem").Call();
    
    END_PX_SECTION
    
    int id = arrayConns.GetCount();
    arrayConns.Add(
        id, "S", "?", "?", "", "[]", "", "", true
    );
    Option* option = new Option();
    optionsEnabled.Add(option);
    option->WantFocus(false).Enable();
    option->WhenAction = THISBACK1(whenOptionEnabledChanged, option);
    arrayConns.SetCtrl(
        id, arrayConns.GetPos(idEnabled), option
    );
    
    arrayConns.SetCursor(id);
    arrayConns.StartEdit(2);
}

void ConnsTable::onMoveUp(int id) {
    BEGIN_PX_SECTION
    
    xConnsTable("moveUp").CallArgs(PxInt(id));
    
    END_PX_SECTION
    
    arrayConns.SwapUp();
    
    int a = arrayConns.Get(id, idId);
    int b = arrayConns.Get(id - 1, idId);
    arrayConns.Set(id, idId, b);
    arrayConns.Set(id - 1, idId, a);

//  optionsEnabled.Swap(id, id - 1);
}

void ConnsTable::onMoveDown(int id) {
    BEGIN_PX_SECTION
    
    xConnsTable("moveDown").CallArgs(PxInt(id));
    
    END_PX_SECTION
    
    arrayConns.SwapDown();
    
    int a = arrayConns.Get(id, idId);
    int b = arrayConns.Get(id + 1, idId);
    arrayConns.Set(id, idId, b);
    arrayConns.Set(id + 1, idId, a);
    
//  optionsEnabled.Swap(id, id + 1);
}

//---------------------------------------------------------------------------
// Connsrv

Connsrv::Connsrv(const PxObject& xRun_, const PxObject& xConnsTable_) 
{
    CtrlLayout(*this, String("CONNSRV 1.3, ") + t_("Server of TCP/IP connections"));
    CenterScreen().Zoomable().Sizeable().SetMinSize(Size(400, 300));
    WhenClose = THISBACK(exit);

    CtrlLayout(connsTable);
    connsTable.SizePos();
    tabMain.Add(connsTable, t_("Connections"));
    
    statusMain.SetDefault(String(t_("Status: ")) + t_("OK"));
    statusMain.Set(Null);
    
    //----------------------------------------------------------
    BEGIN_PX_SECTION
    
    xRun = xRun_;
    xConnsTable = xConnsTable_; 
    
    END_PX_SECTION
    
    connsTable.init(xConnsTable);
    
    thr.Run(THISBACK(pyRun));
    timerRun();
}

bool Connsrv::Accept() {
    return true;
}

void Connsrv::exit(void)
{
    if (!PromptYesNo(t_("Really quit the application?")))
        return;
    
    Thread::ShutdownThreads();
    thr.Wait();
    Close();
}

void Connsrv::timerRun(void)
{   
    connsTable.tstChanged();
    
    tstNewPrintLog();
    
    SetTimeCallback(100, THISBACK(timerRun));
}

void Connsrv::tstNewPrintLog(void)
{
    BEGIN_PX_SECTION
    
    for (PxIter it = xStdErr("getAll").Call(); it.next();) {
        RLOG(it.get().AsString());
        statusMain.Temporary(
            String(t_("Status: ")) + t_("status\vNew event logged"), 5000
        );
    }
    
    END_PX_SECTION
}

void Connsrv::pyRun(void)
{
    int sleepsNum = 1;
    int sleepMs = 1;
    
    while (true) {
        for (int i = 0; i < sleepsNum; i++) {
            if (Thread::IsShutdownThreads())
                return;
            Thread::Sleep(sleepMs);
        }
        
        ACQUIRE_PX_THREAD;
        try {
            xRun.Call();
        }
        catch (PxException& e) {
            LOG_PX_TRACEBACK;
        }
        catch (...) {}
        RELEASE_PX_THREAD;
    }
}

//---------------------------------------------------------------------------
GUI_APP_MAIN
{
    SetLanguage(GetSystemLNG());
    ChClassicSkin();
    
    try {
        pythonInit();
    }
    catch (...) {
        RLOG("Error initializing Python");
    }
    
    {
        PxObject xRun = xNone;
        PxObject xConnsTable = xNone;
        
        ACQUIRE_PX_THREAD;
        try {
            xRun = xMain.GetAttr("run");
            xConnsTable = xMain.GetAttr("connsTable");
        }
        catch (PxException& e) {
            LOG_PX_TRACEBACK;
        }
        catch (...) {}
        RELEASE_PX_THREAD;
        
        Connsrv(xRun, xConnsTable).Run();
    }
    
    try {
        pythonFinalize();
    }
    catch (...) {
        RLOG("Error finalizing Python");
        return;
    }
}
