#include "pycpp.h"

//---------------------------------------------------------------------------
// PxException

PxException::PxException(void) {
    PyErr_Fetch(&ptype, &pvalue, &ptraceback);
}

PxException::PxException(const PxException& obj) {
    Py_IncRef(obj.ptype);
    Py_IncRef(obj.pvalue);
    Py_IncRef(obj.ptraceback);
    ptype = obj.ptype;
    pvalue = obj.pvalue;
    ptraceback = obj.ptraceback;
}

PxException::~PxException(void) {
    Py_DecRef(ptype);
    Py_DecRef(pvalue);
    Py_DecRef(ptraceback);    
}

void PxException::restore(void) {
    if (ptype != NULL) {
        PyErr_Restore(ptype, pvalue, ptraceback);
        ptype = NULL;
        pvalue = NULL;
        ptraceback = NULL;
    }
    else {
        PyErr_SetString(PyExc_RuntimeError, "PxException Error: ptype == NULL in restore()");
    }
}

PxObject PxException::getType(void) {
    if (ptype != NULL)
        return PxObject(ptype);
    else
        return PxObject(Py_None);
}

PxObject PxException::getValue(void) {
    if (pvalue != NULL)
        return PxObject(pvalue);
    else
        return PxObject(Py_None);
}

const char* PxException::getStr(void) {
    if (pvalue != NULL)
        return PxObject(pvalue).AsString();
    else
        return "PxException Error: pvalue == NULL in getStr()";
}

void PxException::printTraceback(void) {
    if (ptype != NULL) {
        PyErr_Restore(ptype, pvalue, ptraceback);
        ptype = NULL;
        pvalue = NULL;
        ptraceback = NULL;
        PyErr_Print();
    }
}

//---------------------------------------------------------------------------
// PxObject

// konstruktor bez parametrov je nebezpecny!
PxObject::PxObject(void): p(NULL) {
}
// owned = true znamena nezvysovat pocet odkazov na objekt
PxObject::PxObject(PyObject* pyObj, bool owned) {
    if (!owned)
        Py_IncRef(pyObj);
    p = pyObj;
}
PxObject::PxObject(const PxObject& obj) {
    PyObject* pyObj = obj.ptr();
    Py_IncRef(pyObj);
    p = pyObj;
}
// destruktor
PxObject::~PxObject(void) {
    Py_DecRef(p);
}
// zvysi/znizi pocet odkazov na objekt
void PxObject::IncRef(void) const {
    Py_IncRef(p);
}
void PxObject::DecRef(void) {
    Py_DecRef(p);
}
// operatory priradenia
PxObject& PxObject::operator=(PyObject* pyObj) {
    assignPy(pyObj);
    return *this;
}
PxObject& PxObject::operator=(const PxObject& obj) {
    assignPy(obj.ptr());
    return *this;
}

// testuje moznost priradenia vzhladom na typ objektu
bool PxObject::assignable(const PxObject& obj) {
    return assignablePy(obj.ptr());
}
bool PxObject::assignablePy(PyObject* pyObj) {
    return (pyObj != NULL);
}

// vrati prislusny PyObject* smernik
PyObject* PxObject::ptr(void) const {
    return p;
}
// vrati prislusny PyObject* smernik a zvysi pocet odkazov
PyObject* PxObject::ptrIncRef(void) const {
    Py_IncRef(p);
    return p;
}
// zneplatni odkaz na objekt
void PxObject::invalidate(void) {
    Py_DecRef(p);
    p = NULL;    
}

//----------------------------------------------------------
bool PxObject::HasAttr(const char* attr_name) const {
    return PyObject_HasAttrString(p, const_cast<char*>(attr_name));
}
PxObject PxObject::GetAttr(const char* attr_name) const {
    PyObject* pyObj = PyObject_GetAttrString(
        p, const_cast<char*>(attr_name)
    );
    if (pyObj == NULL)
        throw PxException();
    return PxObject(pyObj, OWNED);
}
PxObject PxObject::operator()(const char* attr_name) const {
    return GetAttr(attr_name);
}
void PxObject::SetAttr(const char* attr_name, const PxObject& obj) const {
    int ri = PyObject_SetAttrString(
        p, const_cast<char*>(attr_name), obj.ptr()
    );
    if (ri == -1)
        throw PxException();
}
void PxObject::operator()(const char* attr_name, const PxObject& obj) const {
    SetAttr(attr_name, obj);
}
bool PxObject::IsEq(const PxObject& obj) const {
    int ri = PyObject_RichCompareBool(p, obj.ptr(), Py_EQ);
    if (ri == -1)
        throw PxException();
    return ri;   
}
PxString PxObject::Str(void) const {
    PyObject* pyObj = PyObject_Str(p);
    if (pyObj == NULL)
        throw PxException();
    return PxString(pyObj, OWNED);
}
bool PxObject::IsInstance(const PxObject& cls) const {
    int ri = PyObject_IsInstance(p, cls.ptr());
    if (ri == -1)
        throw PxException();
    return ri;
}
bool PxObject::IsCallable(void) const {
    return PyCallable_Check(p);
}
PxObject PxObject::Call(const PxTuple& args, const PxDict& kw) const {
    PyObject* pyObj = PyObject_Call(p, args.ptr(), kw.ptr());
    if (pyObj == NULL)
        throw PxException();
    return PxObject(pyObj, OWNED);
}
PxObject PxObject::Call(const PxTuple& args) const {
    PyObject* pyObj = PyObject_Call(p, args.ptr(), NULL);
    if (pyObj == NULL)
        throw PxException();
    return PxObject(pyObj, OWNED);
}
PxObject PxObject::Call(void) const {
    PyObject* pyObj = PyObject_Call(p, PxTuple(0).ptr(), NULL);
    if (pyObj == NULL)
        throw PxException();
    return PxObject(pyObj, OWNED);
}
bool PxObject::IsTrue(void) const {
    return PyObject_IsTrue(p);
}
bool PxObject::IsFalse(void) const {
    return !PyObject_IsTrue(p);
}
PxIter PxObject::GetIter(void) const {
    PyObject* pyObj = PyObject_GetIter(p);
    if (pyObj == NULL)
        throw PxException();
    return PxIter(pyObj, OWNED);
}

#define PAR1(I) const PxObject& i##I
#define PAR2(I) i##I
#define COM(I) 
#define MET(I) \
PxObject PxObject::CallArgs(EXPPAR##I(PAR1)) { \
    PxTuple xArgs = PxTupleFromArgs(EXPPAR##I(PAR2)); \
    return Call(xArgs); \
}
EXPMET(MET)
#undef PAR
#undef COM
#undef MET

//----------------------------------------------------------
bool PxObject::IsBool() const {
    return PyBool_Check(p);
}
bool PxObject::IsInt() const {
    return PyInt_Check(p);
}
bool PxObject::IsLong() const {
    return PyLong_Check(p);
}
bool PxObject::IsFloat() const {
    return PyFloat_Check(p);
}
bool PxObject::IsComplex() const {
    return PyComplex_Check(p);
}
bool PxObject::IsString() const {
    return PyString_Check(p);
}
bool PxObject::IsUnicode() const {
    return PyUnicode_Check(p);
}
bool PxObject::IsTuple() const {
    return PyTuple_Check(p);
}
bool PxObject::IsList() const {
    return PyList_Check(p);
}
bool PxObject::IsDict() const {
    return PyDict_Check(p);
}

//----------------------------------------------------------
bool PxObject::AsBool(void) const {
    return IsTrue();
}
long PxObject::AsLong(void) const 
{
    if (PyInt_Check(p))
        return PxInt(p).AsLong();
    if (PyLong_Check(p))
        return PxLong(p).AsLong();
    PyErr_SetString(PyExc_TypeError, "Must be Int or Long !");
    throw PxException();
}
unsigned long PxObject::AsUnsignedLong(void) const
{
    if (PyInt_Check(p))
        return PxInt(p).AsUnsignedLong();
    if (PyLong_Check(p))
        return PxLong(p).AsUnsignedLong();
    PyErr_SetString(PyExc_TypeError, "Must be Int or Long !");
    throw PxException();
}
double PxObject::AsDouble(void) const
{
    if (PyInt_Check(p))
        return (double)PxInt(p).AsLong();
    if (PyLong_Check(p))
        return PxLong(p).AsDouble();
    if (PyFloat_Check(p))
        return PxFloat(p).AsDouble();
    PyErr_SetString(PyExc_TypeError, "Must be Int, Long or Float !");
    throw PxException();
}
const char* PxObject::AsString(void) const
{
    return PxString(p).AsString();
}

//----------------------------------------------------------

// retazec typovej vynimky ako metoda
const char* PxObject::getTypeExcMsg(void) {
    return "Must be PyObject!";
}
// vyvola typovu vynimku
void PxObject::typeExc(void) {
    PyErr_SetString(PyExc_TypeError, getTypeExcMsg());
    throw PxException();
}
// vyvola typovu vynimku alebo nerobi nic na zaklade podmienky
void PxObject::typeExcCond(bool cond) {
    if (cond)
        typeExc();
}
// pred priradenim testuje moznost priradenia vzhladom na typ
void PxObject::assignPy(PyObject* pyObj) {
    typeExcCond(
        !assignablePy(pyObj)
    );
    PyObject* pOld = p;
    Py_IncRef(pyObj);
    p = pyObj;
    Py_DecRef(pOld);
}

//---------------------------------------------------------------------------
// PxList

#define PAR(I) const PxObject& i##I
#define COM(I) obj.SetItem(I - 1, i##I)
#define MET(I) \
PxList PxListFromArgs(EXPPAR##I(PAR)) { \
    PxList obj = PxList(I); \
    EXPCOM##I(COM) \
    return obj; \
}
EXPMET(MET)
#undef PAR
#undef COM
#undef MET

//---------------------------------------------------------------------------
// PxTuple

#define PAR(I) const PxObject& i##I
#define COM(I) obj.SetItem(I - 1, i##I)
#define MET(I) \
PxTuple PxTupleFromArgs(EXPPAR##I(PAR)) { \
    PxTuple obj = PxTuple(I); \
    EXPCOM##I(COM) \
    return obj; \
}
EXPMET(MET)
#undef PAR
#undef COM
#undef MET

//---------------------------------------------------------------------------
// Pomocne funkcie

void PxRun_SimpleString(const char* command)
{
    int ri = PyRun_SimpleString(command);
    if (ri == -1)
        throw PxException();
}

PxObject PxImport_AddModule(const char* name)
{
    PyObject *pyObj = PyImport_AddModule(const_cast<char*>(name));
    if (pyObj == NULL)
        throw PxException();
    return PxObject(pyObj);
}
