#include "OpenGLCtrl.h"

int OpenGLCtrl::NbInstance = 0;
int OpenGLCtrl::ContextActivated = 0;

OpenGLCtrl::OpenGLCtrl( int depthsize, int stencilsize, bool doublebuffer )
{
	NumInstance = NbInstance;
	NbInstance++;
	
	IsInitialized    = false;
	IsMapped         = false;
	SubWindow        = 0;
	SubWindowContext = NULL;

	DepthSize        = depthsize;
	StencilSize      = stencilsize;
	DoubleBuffering  = doublebuffer;

	CurrentPos.Clear();
	
	UseGLXVisual     = true;
}

OpenGLCtrl::~OpenGLCtrl()
{
	if( IsInitialized )
		CloseGL();
}

bool OpenGLCtrl::CreateGLXWindow( Window &Window, GLXContext &WindowContext )
{
	Vector<int> visual;
	visual << GLX_RGBA << GLX_DEPTH_SIZE << DepthSize;
	
	if( StencilSize > 0 )
		visual << GLX_STENCIL_SIZE << StencilSize;
	
	if( DoubleBuffering )
		visual << GLX_DOUBLEBUFFER;
	
	visual << None;

	// Try to find a visual
	XVisualInfo *visualInfo = NULL;
	visualInfo = glXChooseVisual( (Display*)Xdisplay, DefaultScreen(Xdisplay), visual );
	
	if( visualInfo == NULL )
	{
		PromptOK(DeQtf("OpenGLCtrl::CreateGLXWindow : Impossible to find a visual."));
		return false;
	}

	// Create an OpenGL rendering context
	WindowContext = glXCreateContext((Display *)Xdisplay, 
	                                  visualInfo,
	                                  NULL,      // No sharing of display lists
	                                  GL_TRUE ); // Direct rendering if possible

	if( WindowContext == NULL )
	{
		PromptOK(DeQtf("OpenGLCtrl::CreateGLXWindow : GLX context creation error."));
		return false;
	}

	if( !glXIsDirect( (Display*)Xdisplay, WindowContext ) )
	{
		PromptOK(DeQtf("OpenGLCtrl::OpenGL : DRI off."));
		return false;
	}

	// Create an X colormap since we're not using the default visual
	Colormap colorMap;
	colorMap = XCreateColormap( Xdisplay,
										 GetTopWindow()->GetWindow(),
										 visualInfo->visual,
										 AllocNone );

	XSetWindowAttributes winAttr;
	winAttr.colormap     = colorMap;
	winAttr.border_pixel = 0;
	winAttr.event_mask   = ExposureMask;
	winAttr.save_under   = XFalse;
	
	// Create an X window with the selected visual
	Rect r = RectInTopWindow();
	Window = XCreateWindow( Xdisplay,                        // display
							GetTopWindow()->GetWindow(),		      // parent
							r.left, r.top, r.Width(), r.Height(),  // x, y, width, height
							0,                                     // border_width
							visualInfo->depth,                     // depth
							InputOutput,                           // class
							visualInfo->visual,                    // visual
							CWBorderPixel | CWColormap | 
							CWSaveUnder   | CWEventMask,           // value mask
							&winAttr );                            // attributes
	
	// Add subwindow to upp list of Xwindows
	int i = Xwindow().Find(None);
	if(i >= 0) Xwindow().SetKey(i, Window );
	XWindow& cw = i >= 0 ? Xwindow()[i] : Xwindow().Add(Window);
	cw.ctrl    = this;
	cw.exposed = true;
	cw.owner   = GetParent();
	cw.xic     = NULL;
	
	if( Window == 0 )
	{
		PromptOK(DeQtf("OpenGLCtrl::CreateGLXWindow : XCreate window error."));
		return false;
	}
	
	return true;
}

void OpenGLCtrl::EventProc(XWindow& w, XEvent *event)
{
	// Flush 'Expose' events
	while( XCheckWindowEvent( Xdisplay, SubWindow, ExposureMask, event )) {};
	
	if( IsMapped )
		OpenGLPaint();
	
	Ctrl::EventProc(w, event );
}

void OpenGLCtrl::OpenGLResize()
{
	if( !IsInitialized )
		return;
	
	int w = GetSize().cx,
	    h = GetSize().cy;
	
	ActivateContext();
	
	// Call user function
	ResizeGL( w, h );	
}

void OpenGLCtrl::OpenGLPaint()
{
	if( IsMapped && IsInitialized )
	{
		ActivateContext();
		
		// Call user function
		PaintGL();
		
		if( DoubleBuffering )
			glXSwapBuffers( (Display*)Xdisplay, SubWindow ); // Buffer swap does implicit glFlush		
		else
			glFlush();
	}
}

void OpenGLCtrl::ActivateContext()
{
	if( NbInstance > 1 && ContextActivated != NumInstance && IsInitialized )
	{
		glXMakeCurrent( (Display*)Xdisplay, SubWindow, SubWindowContext );
		ContextActivated = NumInstance;
	}
}

Rect OpenGLCtrl::RectInTopWindow() const
{
	return GetScreenView() - GetTopCtrl()->GetScreenRect().TopLeft();
}

void OpenGLCtrl::MoveSubWindow()
{
	Rect r = RectInTopWindow();
	int w = GetSize().cx,
		 h = GetSize().cy;
	
	// TODO: use only XMove when needed	 
	XMoveResizeWindow( Xdisplay, SubWindow, r.TopLeft().x, r.TopLeft().y, w, h );
}

void OpenGLCtrl::State(int reason)
{
	if( IsInitialized )
	{
		switch( reason )
		{
			case SHOW: 
			{
				if( IsShown() && !IsMapped )
					MapWindow();

				if( !IsShown() && IsMapped )
					UnMapWindow();
			}; break;
			
			case LAYOUTPOS:
			{
				MoveSubWindow();
				OpenGLResize();
			}; break;
			
			default:
				break;
		}
	}
	else
		if( GetTopWindow() && GetTopWindow()->GetWindow() )
			OpenGL();
}

void OpenGLCtrl::Paint(Draw& draw)
{
	if( IsInitialized )
	{
		Point p = GetRect().TopLeft();
		if( CurrentPos != p )
		{
			CurrentPos = p;
			MoveSubWindow();
		}
	}
}

void OpenGLCtrl::MapWindow()
{
	if( !IsMapped )
	{
		XMapWindow( Xdisplay, SubWindow );
		IsMapped = true;
	}
}

void OpenGLCtrl::UnMapWindow()
{
	if( IsMapped )
	{
		XUnmapWindow( Xdisplay, SubWindow );		
		IsMapped = false;
	}
}

void OpenGLCtrl::OpenGL()
{
	if( IsInitialized )
		return;
	
	int errorBase;
	int eventBase;
	
	// Make sure OpenGL's GLX extensions supported
	if( !glXQueryExtension( (Display*)Xdisplay, &errorBase, &eventBase ) )
	{
		PromptOK(DeQtf("OpenGLCtrl::OpenGL : GLX not supported."));
		return;
	}
	
	// Create glxwindow
	if( CreateGLXWindow( SubWindow, SubWindowContext ) )
	{
		// Activate the created glxwindow
		glXMakeCurrent( (Display*)Xdisplay, SubWindow, SubWindowContext );
		
		IsInitialized = true;
		MapWindow();
	
		// Call user init function
		InitGL();
		ResizeGL( GetSize().cx, GetSize().cy );
		OpenGLPaint();		
	}
}

void OpenGLCtrl::CloseGL()
{
	if( IsInitialized )
	{
		// Call user init function
		DoneGL();
		
		IsInitialized = false;

		UnMapWindow();
		XDestroyWindow( Xdisplay, SubWindow );

		int i = Xwindow().Find(SubWindow);
		if(i >= 0) {
			Xwindow().SetKey(i, None);
			Xwindow()[i].ctrl = NULL;
		}
		
		glXMakeCurrent( (Display*)Xdisplay, None, NULL );
		glXDestroyContext( (Display *)Xdisplay, SubWindowContext );		
	}
}

void OpenGLCtrl::PostResizeGL()
{
	if( !IsInitialized )
		return;
	
	OpenGLResize();
}

void OpenGLCtrl::PostPaintGL()
{
	if( !IsInitialized )
		return;
	
	OpenGLPaint();
}

void OpenGLCtrl::InitGL(){}
void OpenGLCtrl::DoneGL(){}
void OpenGLCtrl::ResizeGL( int w, int h ){}
void OpenGLCtrl::PaintGL(){}
