/***************************************************************************
                          ezcommon  -  description
                             -------------------
    begin                : Dec. 26, 2004
    copyright            : (C) 2005 by Allen
    email                : bon_ami_@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   Explicit Distribution Limitation                                      *
 *   This rule overrides others below.                                     *
 *   This program may not be modified or used by, or, if possible,         *
 *   redistributed to people described as below,                           *
 *   1.Japanese who hold hostility against Chinese.                        *
 *   2.or, those who discriminate against people based solely on race,     *
 *     gender or sexual orientation.                                       *
 *                                                                         *
 ***************************************************************************/
/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/*
 *	source code of EZ Project common functionalities
 */

#ifdef WIN32
#define HAVE_CONIO_H
#define HAVE__GETCH
#else
#include "config.h"
#endif

extern "C" {
#ifdef HAVE_TERMIOS_H
#include <termios.h>
#else
#ifdef HAVE_CONIO_H
#include <conio.h>
#endif
#endif
#include <stdio.h>
}
#include <iostream>
#include <string>
#include <map>

#include "ezcommon.h"
#include "tinyxml.h"

namespace ezproject {

/*	ezoi's private implementation	*/
class ezoip
{
private:
	std::istream *is;	//input file stream
	std::ostream *os;	//output file stream
	std::ostream *es;	//error-output file stream
#ifdef HAVE_TERMIOS_H
	struct termios *tm;	//terminal attributes for standard input
#endif
	std::string mod_nm;	//module name for all output

#ifdef HAVE_TERMIOS_H
	erttp st(void);	//get old and set new standard Input attributes
	void rst(void);	//restore old standard Input attributes
#endif

public:
	ezoip(void) : is(NULL), os(NULL), es(NULL)
#ifdef HAVE_TERMIOS_H
		, tm(NULL)
#endif
		{};	//use standard I/O
	ezoip(std::istream *StreamIn, std::ostream *StreamOut,
		std::ostream *StreamError)
		: is(StreamIn), os(StreamOut), es(StreamError)
#ifdef HAVE_TERMIOS_H
		, tm(NULL)
#endif
		{};	//use file(s) as I/O
	~ezoip(void) {
#ifdef HAVE_TERMIOS_H
	       free(tm);
#endif
	};

	void mod(const std::string &ModuleName);
	//set prefix for all output
	void dsp(const ezoi::eoilvl level, const std::string &content);
	//output something
	void ind(const std::string &Indication);
	//output something almost in raw
	void inpt(const std::string &Indication, std::string &Result);
	//input something
	void inpt(const std::string &Indication, char &Result);
	//input something
};


/*	ezoip	*/
void ezoip::mod(const std::string &ModuleName)
{
	mod_nm = ModuleName;
}

#ifdef HAVE_TERMIOS_H
erttp ezoip::st(void)
{
	erttp ret = RTTP_CRER;

	if ((NULL == is) && (NULL == tm))
	{
		if (NULL != (tm = (struct termios *)
			malloc(sizeof(struct termios))))
		{
			struct termios t;

			if (0 == tcgetattr(STDIN_FILENO, &t))
			{
				if (NULL != memcpy(tm, &t, sizeof(t)))
				{
					ret = RTTP_RGHT;
					if (0 == (tm->c_lflag & ICANON))
					{
						tm->c_lflag &= ~ICANON;
						tcsetattr(STDIN_FILENO,
							TCSANOW, tm);
					}
					else
					{
						free(tm); tm = NULL;
					}
				}
			}
		}
	}
	else
		ret = RTTP_NOER;
	if (RTTP_CRER == ret)
	{
		free(tm);
		tm = NULL;
	}

	return ret;
}

void ezoip::rst(void)
{
	if ((tm != NULL) && (NULL == is))
		tcsetattr(STDIN_FILENO, 0, tm);
}
#endif	/* HAVE_TERMIOS_H */

void ezoip::dsp(const ezoi::eoilvl lvl, const std::string &cnt)
{
	switch (lvl)
	{
		case ezoi::OILVL_FATAL:
			((NULL == es) ? std::cerr : *es)
				<< mod_nm << " FATAL: "
				<< cnt << "!" << std::endl;
			break;
		case ezoi::OILVL_WARN:
			((NULL == es) ? std::cerr : *es)
				<< mod_nm << " Warning! "
				<< cnt << std::endl;
			break;
		case ezoi::OILVL_INFO:
			((NULL == os) ? std::cout : *os)
				<< mod_nm << " Info: "
				<< cnt << std::endl;
			break;
		case ezoi::OILVL_NA:
			((NULL == os) ? std::cout : *os)
				<< mod_nm << ' ' << cnt;
			break;
	}
}

void ezoip::ind(const std::string &ind)
{
	if (NULL == os)
		std::cout << mod_nm << ' '
			<< ind << std::flush;
	else
		dsp(ezoi::OILVL_NA, ind);
}

void ezoip::inpt(const std::string &indi, std::string &ret)
{
	ind(indi);
	getline((NULL == is) ? std::cin : *is, ret);
}

void ezoip::inpt(const std::string &indi, char &ret)
{
	std::ostream *osb = os;

	os = NULL;	//show it on screen
	ind(indi);
	os = osb;	//restore input

#ifdef HAVE_TERMIOS_H
	st();
	ret = getchar();
	if ('\n' != ret)	//finish the line
		std::cout << std::endl;
	rst();
#else
#ifdef HAVE__GETCH
	ret = _getch();
	if ('\n' != ret)	//finish the line
		std::cout << std::endl;
#else
#ifdef HAVE_GETCH
	ret = getch();
	if ('\n' != ret)	//finish the line
		std::cout << std::endl;
#else
	ret = getchar();
	/* trash whatever not the first */
	if (ret != '\n')
		while (getchar() != '\n')
			;
#endif	/* _getch */
#endif	/* getch */
#endif	/* termios */
}


/*	ezoi	*/
/*	-structors	*/
ezoi::ezoi(void)
{
	m_p = new ezoip;
}

ezoi::ezoi(std::istream *si, std::ostream *so,
		std::ostream *se)
{
	m_p = new ezoip(si, so, se);
}

ezoi::~ezoi(void)
{
	delete m_p;
}

/*	other interfaces	*/
void ezoi::mod(const std::string &ModuleName)
{
	m_p->mod(ModuleName);
}

void ezoi::dsp(const ezoi::eoilvl lvl, const std::string &cnt)
{
	m_p->dsp(lvl, cnt);
}

void ezoi::mem(void)
{
	m_p->dsp(ezoi::OILVL_FATAL, "error in allocating memory");
}

void ezoi::ind(const std::string &ind)
{
	m_p->ind(ind);
}

void ezoi::inpt(const std::string &ind, std::string &rst)
{
	m_p->inpt(ind, rst);
}

void ezoi::inpt(const std::string &ind, char &rst)
{
	m_p->inpt(ind, rst);
}


/*	ezcfg's private implementation	*/
class ezcfgp
{
private:
	ezcfgp(void);	//we just don't need this
	TiXmlDocument *doc;
	TiXmlNode *node;
	typedef std::map<std::string, TiXmlNode * /*, less<string>*/ > position_type;
	position_type positions;
	typedef position_type::value_type position_value_type;
	TiXmlAttribute *attribute;

public:
	ezcfgp(const std::string &FileName);
	~ezcfgp();

	erttp next();	// return value:
	//RTTP_CRER: no more element; RTTP_MNER: one level up;
	//RTTP_NOER: one level down; RTTP_RGHT: one element ahead
	//RTTP_NA: no element at all
	void home();	//goto the initial place
	enum eeletp getType();
	void getName(std::string &name);
	void getAttribute(std::string &name, std::string &value,
		bool fromBeginning);
	void getText(std::string &text);
	void setPosition(std::string &position);
	bool getPosition(std::string &position);
};

/*	ezcfgp	*/
/*	-structors	*/
ezcfgp::ezcfgp(const std::string &FileName)
{
	doc = new TiXmlDocument(FileName.c_str());
	if (!doc || !doc->LoadFile())
	{
		delete doc;
		doc = NULL;
	}
	node = NULL;
	attribute = NULL;
}

ezcfgp::~ezcfgp()
{
	delete doc;
}

/*	other interfaces	*/
/*
return value:	RTTP_CRER: no more element; RTTP_MNER: one level up;
		RTTP_NOER: one level down; RTTP_RGHT: one element ahead
*/
erttp ezcfgp::next()
{
	if (!doc)
		return RTTP_CRER;
	attribute = NULL;
	if (!node)
	{
		node = doc->FirstChild();
		if (!node)
			return RTTP_CRER;
		else
		{	//check for text
			if (TiXmlNode::TEXT == node->Type())
			{
				erttp nextResult = next();
				if (RTTP_NOER == nextResult)
					return RTTP_RGHT;
				else
					return nextResult;
			}
			else
				return RTTP_RGHT;
		}
	}
	else
	{
		if (node->NoChildren()
			|| ((TiXmlNode::TEXT == node->FirstChild()->Type())
				&& !node->FirstChild()->NextSibling()))
		{	/* no children, or only children is text */
			if (!node->NextSibling())
			{	//go upper. it must have a parent
				node = node->Parent()->NextSibling();
				if (!node)
					return RTTP_CRER;
				else
				{	//check for text
					if ((TiXmlNode::TEXT == node->Type())
						&& (RTTP_CRER == next()))
						return RTTP_CRER;
					else
						return RTTP_MNER;
				}
			}
			else
			{
				node = node->NextSibling();
				/*	check for text	*/
				if (TiXmlNode::TEXT == node->Type())
					return next();
				else
					return RTTP_RGHT;
			}
		}
		else
		{
			node = node->FirstChild();
			return RTTP_NOER;
		}
	}
}

void ezcfgp::home()
{
	if (doc)
		node = NULL;
}

eeletp ezcfgp::getType()
{
	eeletp type = ELETP_NA;
	if (node)
		switch (node->Type())
		{
			case TiXmlNode::ELEMENT:
				type = ELETP_SOLID;
				break;
			case TiXmlNode::COMMENT:
				type = ELETP_COMMENT;
				break;
			case TiXmlNode::DOCUMENT:
			case TiXmlNode::UNKNOWN:
			case TiXmlNode::TEXT:
			case TiXmlNode::DECLARATION:
			default:
				break;
		}
	else if (!doc)
		type = ELETP_NOFILE;

	return type;
}

void ezcfgp::getName(std::string &name)
{
	if (node)
	{
		TiXmlElement *ele = node->ToElement();
		if (ele)
		{
			const char *namep = ele->Value();

			if (namep)
			{
				name = namep;
				return;
			}
		}
	}
	name.clear();
}

void ezcfgp::getAttribute(std::string &name, std::string &value,
	bool fromBeginning)
{
	if (node)
	{
		TiXmlElement *ele = node->ToElement();
		if (ele)
		{
			if (!attribute || fromBeginning)
				attribute = ele->FirstAttribute();
			else
				attribute = attribute->Next();
			if (attribute)
			{
				const char *namep = attribute->Name();

				if (namep)
				{
					if (name.empty() || (namep == name))
					{
						name = namep;

						const char *valuep
							= attribute->Value();

						if (valuep)
							value = attribute->Value();
						else
							value.clear();
					}
					else
						getAttribute(name, value, false);
				}
				else
					getAttribute(name, value, false);
				return;
			}
		}
	}
	name.clear();
}

void ezcfgp::getText(std::string &text)
{
	if (node)
	{
		TiXmlElement *ele = node->ToElement();
		if (ele)
		{
			const char *textp = ele->GetText();

			if (textp)
			{
				text = textp;
				return;
			}
		}
	}
	text.clear();
}

void ezcfgp::setPosition(std::string &position)
{
	if (node)
	{
		positions.insert(position_value_type(position, node));
	}
	else
		positions.erase(position);
}

bool ezcfgp::getPosition(std::string &position)
{
	position_type::const_iterator it = positions.find(position);

	if (it != positions.end())
	{
		node = it->second;
		return true;
	}
	else
		return false;
}

/*	ezcfg	*/
/*	-structors	*/
ezcfg::ezcfg(const std::string &FileName)
{
	m_p = new ezcfgp(FileName);
}

ezcfg::~ezcfg(void)
{
	delete m_p;
}

/*	other interfaces	*/
erttp ezcfg::next()
{
	if (m_p)
		return m_p->next();
	else
		return RTTP_CRER;
}

void ezcfg::home()
{
	if (m_p)
		m_p->home();
}

bool ezcfg::isSolid()
{
	if (m_p)
	{
		switch (m_p->getType())
		{
			case ELETP_SOLID:
				return true;
			default:
				break;
		}
	}
	return false;
}

bool ezcfg::isLoaded()
{
	if (m_p)
	{
		switch (m_p->getType())
		{
			case ELETP_NOFILE:
				return false;
			default:
				return true;
		}
	}
	return false;
}

void ezcfg::getName(std::string &name)
{
	if (m_p)
		m_p->getName(name);
	else
		name.clear();
}

void ezcfg::getAttribute(std::string &name, std::string &value,
       bool fromBeginning)
{
	if (m_p)
		m_p->getAttribute(name, value, fromBeginning);
	else
		name.clear();
}

void ezcfg::getText(std::string &name)
{
	if (m_p)
		m_p->getText(name);
	else
		name.clear();
}

void ezcfg::setPosition(std::string &position)
{
	if (m_p)
		m_p->setPosition(position);
}

bool ezcfg::getPosition(std::string &position)
{
	if (m_p)
		return m_p->getPosition(position);
	else
		return false;
}

}	/*	namespace ezproject	*/
