/*
Project:			ColorDialog
File:				ColorDialog.cpp
File version:		V1.0.0
Date:				2006-06-29
Ultimate++ version:	uppsrc606-dev3
Code:				Werner Wenzel (public domain)
Comments:			Werner Wenzel (public domain)

This application demonstrates the use of a ColorButton, a ColorPusher,
a ColorDialog, and a custom combination of a Button and a ColorPopUp.
*/

// standard include
#include <CtrlLib/CtrlLib.h>

// individualize placeholder used by iml_header.h and iml_source.h
#define IMAGECLASS ColorDialogImg
// designate file where image resides
#define IMAGEFILE <ColorDialog/ColorDialog.iml>
// include iml_header.h and iml_source.h and make them process the image
#include <Draw/iml.h>

// start with black rectangle
const Color startcolor = Black;

void SynchronizeColors(Color newcolor);

// rectangle - "output" and right-clickable (activating ColorDialog), too -
// 1st way to change rectangle color
class MyStaticRect : public StaticRect
{
private:
	// right-click calls ColorDialog
	virtual void RightDown(Point pos, dword flags)
	{
		// OK or Cancel?
		bool ok;
		// get current color, disable transparency -
		// OK: returns selected color, Cancel: returns current color
		::Color newcolor = RunDlgSelectColor(GetColor(), true, "Select a color", &ok);
		// process returned color only if user pressed "OK"
		if (ok)
		{
			// update rectangle and color controls with selected color
			SynchronizeColors(newcolor);
		}
	}
	
public:
	// get current color (StaticRect::color is protected)
	::Color GetColor() const
	{
		return color;
	}
	
	MyStaticRect()
	{
		// show rectangle even if rectangle color is same as window color
		AddFrame(FieldFrame());
	}
};

// global to be accessible from everywhere
MyStaticRect mystaticrect;

// toolbar button -
// 2nd way to change rectangle color
class MyColorButton : public ColorButton
{
public:
	// callback target if color has been selected
	void SetColor()
	{
		// get selected color and update rectangle and color controls
		SynchronizeColors(GetData());
	}
	
	MyColorButton()
	{
		// disable transparency
		NotNull();
		// tool button image (drawn in current color)
		ColorImage(ColorDialogImg::MyColorButtonImage());
		// tell what I do
		Tip("Change color");
		// if color has been selected
		WhenAction = callback(this, &MyColorButton::SetColor);
	}
};
	
// global to be accessible from everywhere
MyColorButton mycolorbutton;

// specialized pushbutton with optical and textual color description -
// 3d way to change rectangle color
class MyColorPusher : public ColorPusher
{
public:
	// callback target if color is selected
	void SetColor()
	{
		// get selected color and update rectangle and color controls
		SynchronizeColors(GetData());
	}
	
	MyColorPusher()
	{
		// disable transparency
		NotNull();
		// add textual color description
		WithText();
		// if color has been selected
		WhenAction = callback(this, &MyColorPusher::SetColor);
	}
};

// global to be accessible from everywhere
MyColorPusher mycolorpusher;

// standard pushbutton with describing label -
// 4th way to change rectangle color
// (just to show how it can be done - ColorPusher is far better)
class MyButtonColorPusher : public Button
{
private:
	// make ColorPopUp pop up at button object (owner)
	ColorPopUp cpu;
	
	// callback target if color has been selected
	void SetColor()
	{
		// get selected color and update rectangle and color controls
		SynchronizeColors(cpu.Get());
	}
	
public:
	// callback target if button was pressed
	void Click()
	{	
		/*
		disable pop-up effect:
		static const dword NO_EFFECT_SLIDE = 0xFFFFFFFD;
		static const dword NO_EFFECT_FADE = 0xFFFFFFFB;
		cpu.SetFlags(GetFlags() & NO_EFFECT_SLIDE & NO_EFFECT_FADE);
		*/
		// if color has been selected
		cpu.WhenSelect = callback(this, &MyButtonColorPusher::SetColor);
		// activate ColorPopUp
		cpu.PopUp(this);
	}
	
	MyButtonColorPusher()
	{
		// tell what I do
		SetLabel("Change color");
		// if button was pressed
		WhenAction = callback(this, &MyButtonColorPusher::Click);
	}
};

// global to be accessible from everywhere
MyButtonColorPusher mybuttoncolorpusher;

// update rectangle and color controls
void SynchronizeColors(Color newcolor)
{
	// set new rectangle color
	mystaticrect.Color(newcolor);
	// adjust display
	mystaticrect.Refresh();
	// set new toolbar button color
	mycolorbutton.SetData(newcolor);
	// adjust display
	mycolorbutton.Refresh();
	// set new ColorPusher color
	mycolorpusher.SetData(newcolor);
	// adjust display
	mycolorpusher.Refresh();
}

// the application itself
class App : public TopWindow
{
	// to place toolbar button
	ToolBar toolbar;
	// tell what to do with rectangle
	Label mystaticrectlabel;
	// tell what ColorPusher is
	Label mycolorpusherlabel;
	// tell what button is
	Label mybuttoncolorpusherlabel;
	
	// callback target when toolbar is set up
	void PopulateToolBar(Bar& bar)
	{
		// add ColorButton to toolbar
		bar.Add(mycolorbutton);
	}
	
public:
	App()
	{
		// initialize toolbar
		toolbar.Set(callback(this, &App::PopulateToolBar));
		// add toolbar as frame
		AddFrame(toolbar);
		// fence off toolbar
		AddFrame(TopSeparatorFrame());
		
		// (distance from left edge in pixels, width in pixels)
		mystaticrectlabel.LeftPos(25,200);
		// (distance from top edge in pixels, height in pixels)
		mystaticrectlabel.TopPos(25, 25);
		// add text to label
		mystaticrectlabel.SetText("Right-click the rectangle");
		// add label
		Add(mystaticrectlabel);

		// (distance from left edge in pixels, width in pixels)
		mystaticrect.LeftPos(25,286);
		// (distance from top edge in pixels, height in pixels)
		mystaticrect.TopPos(50, 286);
		// add rectangle
		Add(mystaticrect);
		
		// (distance from right edge in pixels, width in pixels)
		mycolorpusherlabel.RightPos(25,200);
		// (distance from top edge in pixels, height in pixels)
		mycolorpusherlabel.TopPos(25, 25);
		// add text to label
		mycolorpusherlabel.SetText("Ultimate++ ColorPusher");
		// add label
		Add(mycolorpusherlabel);
		
		// (distance from right edge in pixels, width in pixels)
		mycolorpusher.RightPos(25,200);
		// (distance from top edge in pixels, height in pixels)
		mycolorpusher.TopPos(50, 50);
		// add ColorPusher
		Add(mycolorpusher);

		// (distance from right edge in pixels, width in pixels)
		mybuttoncolorpusherlabel.RightPos(25, 200);
		// (distance from bottom edge in pixels, height in pixels)
		mybuttoncolorpusherlabel.BottomPos(85, 25);
		// add text to label
		mybuttoncolorpusherlabel.SetText("Custom ColorPusher");
		// add label
		Add(mybuttoncolorpusherlabel);
		
		// (distance from right edge in pixels, width in pixels)
		mybuttoncolorpusher.RightPos(25, 200);
		// (distance from bottom edge in pixels, height in pixels)
		mybuttoncolorpusher.BottomPos(33, 50);
		// add button
		Add(mybuttoncolorpusher);
		
		// set application title
		Title("ColorDialog");
	}
};

GUI_APP_MAIN
{
	// instantiate application
	App myapp;
	// set up window (left, top, width, height) -
	// left and top both 0: "center window"
	myapp.SetRect(0, 0, 600, 400);
	// set rectangle and controls to same initial color
	SynchronizeColors(startcolor);
	// start application
	myapp.Run();
}
