///////////////////////////////////////////////////////////////////
// GUtil: graphics utilities.

#include <utility/misc.jinc>

#define LLOG(x) // RLOG(x)

#define LNGMODULE Lang

package utility;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import java.net.*;
import java.applet.*;

public class GUtil
{
	public final static String stdFontFace = "SansSerif";
	public static final int STDFONTHT = 12;
	public final static Font stdFont = new Font(stdFontFace, Font.PLAIN, STDFONTHT);

	public final static int IMAGE_END = ImageObserver.ALLBITS | ImageObserver.ERROR | ImageObserver.ABORT;

	public final static void drawRect(Graphics g, int x, int y, int w, int h, int width)
	{
		if(w <= 0 || h <= 0 || width <= 0)
			return;
		int two = 2 * width;
		if(w <= two || h <= two)
			g.fillRect(x, y, w, h);
		else
		{
			g.fillRect(x, y, w, width);
			g.fillRect(x, y + h - width, w, width);
			g.fillRect(x, y + width, width, h - two);
			g.fillRect(x + w - width, y + width, width, h - two);
		}
	}

	public final static void drawRect(Graphics g, Rectangle r, int width)
	{
		drawRect(g, r.x, r.y, r.width, r.height, width);
	}

	public final static void drawRect(Graphics g, int x, int y, int w, int h, int width, Color c)
	{
		g.setColor(c);
		drawRect(g, x, y, w, h, width);
	}

	public final static void drawRect(Graphics g, Rectangle r, int width, Color c)
	{
		g.setColor(c);
		drawRect(g, r.x, r.y, r.width, r.height, width);
	}

	public final static void drawRect(Graphics g, Rectangle small, Rectangle big)
	{
		small = small.intersection(big);
		if(small.isEmpty())
		{
			fillRect(g, big);
			return;
		}
		Point sr = Util.rightBottom(small);
		Point br = Util.rightBottom(big);
		if(small.y > big.y)
			g.fillRect(big.x, big.y, big.width, small.y - big.y);
		if(sr.y < br.y)
			g.fillRect(big.x, sr.y, big.width, br.y - sr.y);
		if(small.x > big.x)
			g.fillRect(big.x, small.y, small.x - big.x, small.height);
		if(sr.x < br.x)
			g.fillRect(sr.x, small.y, br.x - sr.x, small.height);
	}

	public final static void drawRect(Graphics g, Rectangle rc, int width, Color lt, Color rb)
	{
		drawRect(g, rc.x, rc.y, rc.width, rc.height, width, lt, rb);
	}

	public final static void drawRect(Graphics g, int x, int y, int w, int h,
		int width, Color lt, Color rb)
	{
		int two = 2 * width;
		if(w <= two || h <= two)
			return;
		g.setColor(lt);
		g.fillRect(x, y + width, width, h - two);
		g.fillRect(x, y, w, width);
		g.setColor(rb);
		g.fillRect(x + w - width, y + width, width, h - two);
		g.fillRect(x, y + h - width, w, width);
	}

	public final static void drawRect(Graphics g, Rectangle small, Dimension size)
	{
		drawRect(g, small, new Rectangle(size));
	}

	public final static void fillRect(Graphics g, Rectangle rc)
	{
		if(!rc.isEmpty())
			g.fillRect(rc.x, rc.y, rc.width, rc.height);
	}

	public final static void fillRect(Graphics g, Rectangle rc, Color co)
	{
		g.setColor(co);
		g.fillRect(rc.x, rc.y, rc.width, rc.height);
	}

	public final static void fillRect(Graphics g, int x, int y, int w, int h, Color co)
	{
		g.setColor(co);
		g.fillRect(x, y, w, h);
	}

	public final static void fillRectMinusRect(Graphics g, Rectangle out, Rectangle in)
	{
		if(out.isEmpty())
			return;
		if(in.isEmpty() || !in.intersects(out))
		{
			fillRect(g, out);
			return;
		}
		int or = out.x + out.width, ob = out.y + out.height;
		int ir = in.x + in.width, ib = in.y + in.height;
		int ht = out.y, hb = ob;
		if(in.y > out.y)
			g.fillRect(out.x, out.y, out.width, (ht = in.y) - out.y);
		if(ib < ob)
			g.fillRect(out.x, ib, out.width, ob - (hb = ib));
		if(ht >= hb)
			return;
		if(in.x > out.x)
			g.fillRect(out.x, ht, in.x - out.x, hb - ht);
		if(ir < or)
			g.fillRect(ir, ht, or - ir, hb - ht);
	}

	public final static void fillRectMinusRect(Graphics g, Rectangle out, Rectangle in, Color co)
	{
		g.setColor(co);
		fillRectMinusRect(g, out, in);
	}

	public final static void fillRectMinusFrame(Graphics g, Rectangle a, Rectangle frm, int width)
	{
		if(!a.intersects(frm))
			fillRect(g, a);
		else
		{
			fillRectMinusRect(g, a, frm);
			if(frm.width > 2 * width && frm.height > 2 * width)
				fillRect(g, Util.inflated(frm, -width).intersection(a));
		}
	}

	public final static void drawRectMinusFrame(Graphics g, Rectangle a, Rectangle b, int width)
	{
		if(a.width <= 2 * width || a.height <= 2 * width)
			fillRectMinusFrame(g, a, b, width);
		else
		{
			int il = a.x + width, it = a.y + width;
			int ir = a.x + a.width - width, ib = a.y + a.height - width;
			fillRectMinusFrame(g, new Rectangle(a.x, a.y, a.width, width), b, width);
			fillRectMinusFrame(g, new Rectangle(a.x, ib, a.width, width), b, width);
			fillRectMinusFrame(g, new Rectangle(a.x, it, width, ib - it), b, width);
			fillRectMinusFrame(g, new Rectangle(ir, it, width, ib - it), b, width);
		}
	}

	public final static void drawAnimatedRect(Graphics g, Rectangle old_rc, Rectangle new_rc, int width)
	{
//		LLOG("drawAnimatedRect " + old_rc.toString() + " -> " + new_rc.toString());
		if(old_rc != null && new_rc != null && old_rc.intersects(new_rc))
		{
			drawRectMinusFrame(g, new_rc, old_rc, width);
			drawRectMinusFrame(g, old_rc, new_rc, width);
			return;
		}
		if(new_rc != null) drawRect(g, new_rc, width);
		if(old_rc != null) drawRect(g, old_rc, width);
	}

	public final static void drawImage(Graphics g, Image i, Point at, ImageObserver o)
	{
		g.drawImage(i, at.x, at.y, o);
	}

	public final static void drawImage(Graphics g, Image i, Point at, Dimension size, ImageObserver o)
	{
		g.drawImage(i, at.x, at.y, size.width, size.height, o);
	}

	public final static void drawImage(Graphics g, Image i, Rectangle rc, ImageObserver o)
	{
		drawImage(g, i, rc, null, o);
	}

	public final static void drawImage(Graphics g, Image i, Point at, Color bg, ImageObserver o)
	{
		g.drawImage(i, at.x, at.y, bg, o);
	}

	public final static void drawImage(Graphics g, Image i, Point at, Dimension size, Color bg, ImageObserver o)
	{
		g.drawImage(i, at.x, at.y, size.width, size.height, bg, o);
	}

	public final static void drawImage(Graphics g, Image i, Rectangle rc, Color bg, ImageObserver o)
	{
//    	LLOG("drawImage to " + rc);
		int unit = Util.max(GUtil.applet.getSize().width, 100);
		if(rc.width <= unit && rc.height <= unit)
		{
			if(bg != null)
				g.drawImage(i, rc.x, rc.y, rc.width, rc.height, bg, o);
			else
				g.drawImage(i, rc.x, rc.y, rc.width, rc.height, o);
			return;
		}

//        LLOG("big image to " + rc);
		int width = i.getWidth(o), height = i.getHeight(o);
		if(width <= 0 || height <= 0)
			return;

		Dimension divisions = Util.add(Util.div(rc.getSize(), unit), 1);
		Rectangle clip = Util.minmax(rc.getSize(), Util.moved(g.getClipBounds(), -rc.x, -rc.y));
		if(clip.isEmpty())
			return;

//        LLOG("divisions = " + divisions);
//        LLOG("clip = " + clip);
		Rectangle div_dest = Util.scale(clip, divisions, rc.getSize());
//        LLOG("div_dest = " + div_dest);
		for(int ix = div_dest.x + div_dest.width; --ix >= div_dest.x;)
			for(int iy = div_dest.y + div_dest.height; --iy >= div_dest.y;)
			{
				int sx1 = Util.scale(ix, width, divisions.width);
				int sy1 = Util.scale(iy, height, divisions.height);
				int sx2 = Util.scale(ix + 1, width, divisions.width);
				int sy2 = Util.scale(iy + 1, height, divisions.height);

				int dx1 = rc.x + Util.scale(ix, rc.width, divisions.width);
				int dy1 = rc.y + Util.scale(iy, rc.height, divisions.height);
				int dx2 = rc.x + Util.scale(ix + 1, rc.width, divisions.width);
				int dy2 = rc.y + Util.scale(iy + 1, rc.height, divisions.height);

				if(bg != null)
					g.drawImage(i, dx1, dy1, dx2, dy2, sx1, sy1, sx2, sy2, bg, o);
				else
					g.drawImage(i, dx1, dy1, dx2, dy2, sx1, sy1, sx2, sy2, o);
			}
//        LLOG("done");
	}

	public final static Image getCodeImage(String name)
	{
		try
		{
//			if(name.charAt(0) == '!')
//				return getImage(name.substring(1) + ".gif");
			return helper.getCodeImage(name);
		}
		catch(NullPointerException e)
		{
			return null;
		}
	}

	private static final byte error_gif[] =
	{
		+0x47, +0x49, +0x46, +0x38, +0x37, +0x61, +0x0e, +0x00, +0x10, +0x00, -0x5e, +0x00, +0x00, -0x01, -0x01, -0x01,
		-0x80, -0x80, -0x80, +0x00, +0x00, +0x00, -0x40, -0x40, -0x40, -0x01, +0x00, +0x00, +0x00, +0x00, +0x00, +0x00,
		+0x00, +0x00, +0x00, +0x00, +0x00, +0x21, -0x07, +0x04, +0x01, +0x00, +0x00, +0x05, +0x00, +0x2c, +0x00, +0x00,
		+0x00, +0x00, +0x0e, +0x00, +0x10, +0x00, +0x00, +0x03, +0x30, +0x18, -0x46, -0x24, -0x10, +0x30, -0x7e, +0x21,
		-0x7e, -0x44, -0x2c, -0x22, -0x68, +0x37, -0x51, +0x10, +0x41, +0x3c, -0x1e, -0x2d, -0x7b, -0x1e, +0x68, -0x7e,
		+0x51, +0x09, -0x63, +0x64, -0x06, -0x4e, -0x80, -0x15, -0x3e, +0x1e, -0x42, -0x17, +0x18, -0x63, +0x57, -0x7d,
		-0x60, +0x70, +0x18, +0x14, +0x18, -0x71, -0x38, +0x64, +0x02, +0x00, +0x3b,
	};

 	private static Image error_image = null;

	public final static Image getErrorImage()
	{
		if(error_image != null)
			return error_image;
		error_image = Toolkit.getDefaultToolkit().createImage(error_gif);
		return error_image;
	}

	public final static Image getImage(String name)
	{
		try
		{
			LLOG("GUtil.getImage(" + name + ")");
			return applet.getImage(applet.getDocumentBase(), name);
		}
		catch(NullPointerException e)
		{
			return null;
		}
	}

	public final static void drawTextBg(Graphics g, Point pt, String text, Color color, Color bgcolor)
	{ drawTextBg(g, pt.x, pt.y, text, color, bgcolor); }

	public final static void drawTextBg(Graphics g, int x, int y, String text, Color color, Color bgcolor)
	{
		g.setColor(bgcolor == null ? getContrastColor(color) : bgcolor);
		g.drawString(text, x - 1, y);
		g.drawString(text, x + 1, y);
		g.drawString(text, x, y - 1);
		g.drawString(text, x, y + 1);
		g.setColor(color);
		g.drawString(text, x, y);
	}

	public final static URL getBaseURL(String name)
	{
		try
		{
			return new URL(applet.getDocumentBase(), name);
		}
		catch(MalformedURLException e)
		{
			return null;
		}
	}

	public final static void showDocument(String url)
	{
		if(!standalone)
			applet.getAppletContext().showDocument(getBaseURL(url));
		else {
			LLOG("showDocument: " + url);
		}
	}

	public final static void showDocument(String url, String target)
	{
		if(!standalone) {
			System.out.println("showDocument(URL = " + url + ", target = " + target + ")");
			applet.getAppletContext().showDocument(getBaseURL(url), target);
		}
		else {
			LLOG("showDocument@" + target + ": " + url);
		}
	}

	public final static URL getCodeURL(String name)
	{
		try
		{
			return new URL(applet.getCodeBase(), name);
		}
		catch(MalformedURLException e)
		{
			return null;
		}
	}

	public final static FontMetrics getFontMetrics(Font font)
	{
		Graphics g = applet.getGraphics();
		FontMetrics metrics = (font == null ? g.getFontMetrics() : g.getFontMetrics(font));
		g.dispose();
		return metrics;
	}

	public final static Dimension getTextSize(FontMetrics metrics, String text)
	{
		return new Dimension(text == null ? 0 : metrics.stringWidth(text), metrics.getHeight());
	}

	public final static int getTextWidth(Font font, String text)
	{
		if(text == null)
			return 0;
		FontMetrics metrics = getFontMetrics(font);
		return metrics.stringWidth(text);
	}

	public final static Dimension getTextSize(Font font, String text)
	{
		return (text == null ? new Dimension(0, 0) : getTextSize(getFontMetrics(font), text));
	}

	public final static synchronized Dimension getFontSize()
	{
		if(fontUnit == null) {
			Graphics g = applet.getGraphics();
			FontMetrics metrics = g.getFontMetrics();
			fontUnit = new Dimension(metrics.charWidth('X'), metrics.getHeight());
			g.dispose();
		}
		return fontUnit;
	}

	public final static int getFontWidth()
	{
		return getFontSize().width;
	}

	public final static int getFontHeight()
	{
		return getFontSize().height;
	}

	public final static Font getFont(String fontspec, Font dflt)
	{
		int len;
		if(fontspec == null || (len = fontspec.length()) == 0)
			return dflt;
		String name = dflt.getName();
		int flags = Font.PLAIN;
		int height = dflt.getSize();
		int i = 0;
		if(Util.isdigit(fontspec.charAt(0))) {
			height = Util.atoi(fontspec, 0);
			while(++i < len && Util.isdigit(fontspec.charAt(i)))
				;
		}
		for(; i < len; i++) {
			char c = fontspec.charAt(i);
			if(c == '/')
				flags |= Font.ITALIC;
			else if(c == '*')
				flags |= Font.BOLD;
			else if(c != ' ')
				break;
		}
		if(i < len)
			name = fontspec.substring(i);
		return new Font(name, flags, height);
	}
	
	public final static Font getFont(String fontspec)
	{
		return getFont(fontspec, stdFont);
	}

	public final static Dimension getButtonSize()
	{
		Dimension dim = getFontSize();
		return new Dimension(dim.width * 10, dim.height + 8);
	}

	public final static void setDefaultInfo(String s)
	{
		if(info_adapter != null)
			info_adapter.setDefault(s);
	}

	public final static void setDefaultInfo(String s, boolean enabled)
	{
		setDefaultInfo(parseInfo(s, enabled));
	}

	public final static void setLeftInfo(String s, int progress)
	{
		LLOG("GUtil.setLeftInfo (" + s + ", " + progress + ")");
		if(info_adapter != null)
			info_adapter.setLeft(s, progress);
	}

	public final static void setLeftInfo(String s) { setLeftInfo(s, -1); }

	public final static void setLeftInfo(String s, boolean enabled)
	{
		setLeftInfo(parseInfo(s, enabled));
	}

	public final static void setRightInfo(String s)
	{
		if(info_adapter != null)
			info_adapter.setRight(s);
	}

	private static String parseInfo(String s, boolean enabled)
	{
		if(s == null)
			return null;
		String result = "";
		int pos = 0;
		char dir = '\b';
		while(pos < s.length())
		{
			int start = pos;
			char c = '\0';
			while(pos < s.length()
				&& (c = s.charAt(pos)) != '\b' && c != '\t' && c != '\f')
				pos++;
			if((dir != '\t' || enabled) && (dir != '\f' || !enabled))
				result = result + s.substring(start, pos);
			if(pos < s.length())
			{
				pos++;
				dir = c;
			}
		}
		return result;
	}

	public final static double getScreenDPI()
	{
		int dpi = Toolkit.getDefaultToolkit().getScreenResolution();
		return dpi > 0 ? dpi : 96;
	}

	public final static double getPixelsPerMeter()
	{
		return getScreenDPI() * 39.37; // 39.37 = inches / meter
	}

	public final static int getValue(int r, int g, int b)
	{
		return Math.max(r, Math.max(g, b));
	}

	public final static int getSaturation(int r, int g, int b)
	{
		int v = getValue(r, g, b);
		if(v == 0)
			return 0;
		int delta = v - Math.min(r, Math.min(g, b));
		if(delta == 0)
			return 0;
		return Util.scale(delta, 255, v);
	}

	public final static int getHue(int r, int g, int b)
	{
		int v = getValue(r, g, b);
		int delta = 6 * (v - Math.min(r, Math.min(g, b)));
		if(delta == 0)
			return 0;
		if(g == v)
			return Util.scale(b - r, 85, delta) + 85;
		else if(b == v)
			return Util.scale(r - g, 85, delta) + 171;
		else
			return Util.scale(g - b, 85, delta) & 0xFF;
	}

	public final static int getHue(Color c)        { return c == null ? 0 : getHue(c.getRed(), c.getGreen(), c.getBlue()); }
	public final static int getSaturation(Color c) { return c == null ? 0 : getSaturation(c.getRed(), c.getGreen(), c.getBlue()); }
	public final static int getValue(Color c)      { return c == null ? 0 : getValue(c.getRed(), c.getGreen(), c.getBlue()); }

	//////////////////////////////////////////////////////////////////////

	public final static Color hsvToRgb(int h, int s, int v)
	{
		if(s == 0)
		{
			return new Color(v, v, v);
		}
		int rem = (h *= 6) & 0xFF;
		int p = Util.scale(v, 255 - s, 256);
		int q = Util.scale(v, 255 - Util.scale(s, rem, 256), 256);
		int t = Util.scale(v, 255 - Util.scale(s, 255 - rem, 256), 256);
		switch(h >> 8)
		{
		default: // error
		case 6:
		case 0: return new Color(v, t, p);
		case 1: return new Color(q, v, p);
		case 2: return new Color(p, v, t);
		case 3: return new Color(p, q, v);
		case 4: return new Color(t, p, v);
		case 5: return new Color(v, p, q);
		}
	}

	public final static Color getContrastColor(Color c)
	{
		if(c == null) return Color.white;
		int value = c.getRed() * 30 + c.getGreen() * 59 + c.getBlue() * 11;
		return (value >= 12800 ? Color.black : Color.white);
	}

	//////////////////////////////////////////////////////////////////

	public final static Frame getFrame(Component comp)
	{
		for(; comp != null; comp = comp.getParent())
			if(comp instanceof Frame)
				return (Frame)comp;
		return new Frame();
//		Frame[] list = Frame.getFrames();
//		LLOG("Application frames: " + list.size());
//		owner = GUtil.applet;
	}

	public final static void runPopup(PopupMenu popup, Component owner, Component top, int x, int y)
	{
		LLOG("GUtil.runPopup, x = " + x + ", y = " + y);
		if(popup.getParent() == null)
			top.add(popup);
		for(Component c = owner; c != null && c != top; c = c.getParent())
		{
			Rectangle rc = c.getBounds();
			x += rc.x;
			y += rc.y;
			LLOG("Component = " + c + ", rect = " + rc + ", offset = " + x + ", " + y);
			if(c instanceof Container)
			{
				Container co = (Container)c;
				Insets in = co.getInsets();
//				x += in.left;
//				y += in.top;
			}
		}
		LLOG("Raw coords: " + x + ", " + y);
		sync(top);
		popup.show(top, x, y);
	}

	private static void syncRaw(Component c)
	{
		Graphics g = c.getGraphics();
		c.paintAll(g);
		g.dispose();
		if(c instanceof Container)
		{
			Container cont = (Container)c;
			for(int i = 0; i < cont.getComponentCount(); i++)
				syncRaw(cont.getComponent(i));
		}
	}

	public final static void sync(Component c)
	{
		while(c.getParent() != null)
			c = c.getParent();
		syncRaw(c);
	}

	public final static void setKeyListenerDeep(Component component, KeyListener listener)
	{
		component.removeKeyListener(listener);
		component.addKeyListener(listener);
		if(component instanceof Container)
		{
			Container cont = (Container)component;
			Component comp[] = cont.getComponents();
			for(int i = 0; i < comp.length; i++)
				setKeyListenerDeep(comp[i], listener);
		}
	}

	private static Dimension fontUnit;

	public static Color         whiteGray = new Color(0xE0E0E0);
	public static Applet        applet;
	public static AppHelper     helper;
	public static InfoAdapter   info_adapter;
	public final static int     dpi = Toolkit.getDefaultToolkit().getScreenResolution();
	public final static double  dppt = dpi / 72.0;
	public final static boolean standalone = false;
}
