//////////////////////////////////////////////////////////////////
// Plotter: graphics context with transform matrix.

#include <utility/misc.jinc>

package utility;

#define LLOG(x) // RLOG(x)

import java.awt.*;
import java.util.*;
import utility.*;

public class Plotter
{
	public Plotter(Graphics gc, FPoint scale, FPoint delta, Color xor_bg)
	{
		this.gc = gc;
		this.scale = scale;
		this.delta = delta;
		this.xor_bg = xor_bg;
//		LLOG("Plotter scale: " + this.scale + ", delta: " + this.delta);
		this.avg_scale = (Math.abs(scale.x) + Math.abs(scale.y)) / 2;
		this.log_dpi = dpi / avg_scale;
		this.clip = gc.getClipBounds();
		this.logclip = PtoL(this.clip);
	}
	public Plotter(Graphics gc)              { this(gc, new FPoint(1, 1), new FPoint(0, 0), null); }

	public  FPoint     getScale()            { return scale; }
	public  FPoint     getDelta()            { return delta; }

	public  double     DotL(double dot)      { return dot / 600.0 * log_dpi; }
	public  double     PointL(double pt)     { return pt / 72.0 * log_dpi; }

	public  double     PtoL(double d)        { return d / avg_scale; }
	public  FPoint     PtoL(FPoint pt)       { return pt == null ? null : pt.minus(delta).over(scale); }
	public  FPoint     PtoL(Point pt)        { return pt == null ? null : new FPoint(pt).minus(delta).over(scale); }
	public  FRect      PtoL(FRect rc)        { return rc == null ? null : rc.minus(delta).over(scale).sort(); }
	public  FRect      PtoL(Rectangle rc)    { return rc == null ? null : new FRect(rc).minus(delta).over(scale).sort(); }

	public  double     LtoF(double d)        { return d * avg_scale; }
	public  int        LtoP(double d)        { return (int)(d * avg_scale); }
	public  FPoint     LtoF(FPoint pt)       { return pt == null ? null : pt.times(scale).plus(delta); }
	public  Point      LtoP(FPoint pt)       { return pt == null ? null : pt.times(scale).plus(delta).asPoint(); }
	public  FRect      LtoF(FRect rc)        { return rc == null ? null : rc.times(scale).plus(delta).sort(); }
	public  Rectangle  LtoP(FRect rc)        { return rc == null ? null : rc.times(scale).plus(delta).sort().asRect(); }

	public  boolean    isLClip(FRect rc)     { return rc.intersects(logclip); }
	public  boolean    isPClip(Rectangle rc) { return rc.intersects(clip); }
	public  boolean    isLClip(FPoint pt, double size)
	{
		return pt.x + size < logclip.right  && pt.x - size > logclip.left
		&&     pt.y + size < logclip.bottom && pt.y - size > logclip.top;
	}
	public  boolean    isLClip(FPoint pt)    { return logclip.contains(pt); }

	public  void       setColor(Color c)
	{
		if(c != null)
			if(xor_bg != null) {
				gc.setColor(xor_bg);
				gc.setXORMode(c);
			}
			else
			{
				gc.setPaintMode();
				gc.setColor(c);
			}
	}

	public  void       drawLine(FPoint a, FPoint b, Color c, double width)
	{
		if(a == null || b == null)
			return;
		int ow = LtoP(width);
		Point oa = LtoP(a), ob = LtoP(b);
		setColor(c);
		if(ow <= 1)
		{
//			LLOG("Simple line " + oa + " -> " + ob);
			gc.drawLine(oa.x, oa.y, ob.x, ob.y);
		}
		else if(oa.equals(ob))
		{
			int hf = ow / 2;
			gc.fillOval(oa.x - hf, oa.y - hf, ow, ow);
		}
		else
		{
			int xpoints[] = new int[6], ypoints[] = new int[6];
			FPoint fforw = new FPoint(ob.x - oa.x, ob.y - oa.y).length(ow * 0.5 + 1);
			Point forw = fforw.asPoint();
			Point back = new Point(forw.x - (int)Math.round(2 * fforw.x), forw.y - (int)Math.round(2 * fforw.y));
//			LLOG("forw = " + forw + ", sc = " + sc);
			xpoints[0] = oa.x - forw.y; ypoints[0] = oa.y + forw.x;
			xpoints[1] = oa.x - forw.x; ypoints[1] = oa.y - forw.y;
			xpoints[2] = oa.x - back.y; ypoints[2] = oa.y + back.x;
			xpoints[3] = ob.x - back.y; ypoints[3] = ob.y + back.x;
			xpoints[4] = ob.x + forw.x; ypoints[4] = ob.y + forw.y;
			xpoints[5] = ob.x - forw.y; ypoints[5] = ob.y + forw.x;
			gc.fillPolygon(xpoints, ypoints, 6);
		}
	}

	public void        drawArrow(FPoint a, FPoint b, Color c, double width, double length, double angle, boolean bi_arrow)
	{
		drawLine(a, b, c, width);
		double d = a.dist(b);
		length = Math.min(length, d / 2);
		if(length * avg_scale >= 1)
		{
			FPoint f = b.minus(a).length(length);
			FPoint l = f.rotated(+angle);
			FPoint r = f.rotated(-angle);
			drawLine(b, b.minus(l), c, width);
			drawLine(b, b.minus(r), c, width);
			if(bi_arrow)
			{
				drawLine(a, a.plus(l), c, width);
				drawLine(a, a.plus(r), c, width);
			}
		}
	}

	public void        drawArrow(FPoint a, FPoint b, Color c, double width, double length, double angle)   { drawArrow(a, b, c, width, length, angle, false); }
	public void        drawBiArrow(FPoint a, FPoint b, Color c, double width, double length, double angle) { drawArrow(a, b, c, width, length, angle, true); }
	public void        drawArrow(FPoint a, FPoint b, Color c, double width, double length)                 { drawArrow(a, b, c, width, length, 0.3); }
	public void        drawBiArrow(FPoint a, FPoint b, Color c, double width, double length)               { drawArrow(a, b, c, width, length, 0.3); }

	public void        drawCross(FPoint p, double cx, double cy, Color c, double width)
	{
		drawLine(new FPoint(p.x - cx, p.y - cy), new FPoint(p.x + cx, p.y + cy), c, width);
		drawLine(new FPoint(p.x - cx, p.y + cy), new FPoint(p.x + cx, p.y - cy), c, width);
	}

	public void        drawCross(FPoint centre, FPoint size, Color c, double width) { drawCross(centre, size.x, size.y, c, width); }
	public void        drawCross(FPoint centre, double size, Color c, double width) { drawCross(centre, size, size, c, width); }

	public void        drawRect(FRect rc, double width)
	{
		if(rc != null)
			GUtil.drawRect(gc, LtoP(rc.inflated(width / 2)), Util.max(1, LtoP(width)));
	}

	public void        drawRect(FRect rc, Color outline, double outline_width, Color fill)
	{
		if(rc == null || (outline == null && fill == null))
			return;
		Rectangle or = LtoP(rc.inflated(outline_width / 2));
		int ow = Util.max(1, LtoP(outline_width));
		Rectangle in = new Rectangle(or.x + ow, or.y + ow, or.width - 2 * ow, or.height - 2 * ow);
		if(fill != null)
		{
			setColor(fill);
			GUtil.fillRect(gc, outline == null ? or : in);
		}
		if(outline != null)
		{
			setColor(outline);
			GUtil.drawRect(gc, or, ow);
		}
	}

	public void        drawRect(FRect rc, Color fill) { drawRect(rc, null, 0, fill); }

	public void        drawEllipse(FRect rc, double width)
	{
		FPoint size = rc.getSize();
		if(width * avg_scale <= 1.5)
		{
			Rectangle r = LtoP(rc);
			gc.drawOval(r.x, r.y, r.width, r.height);
		}
		else if(size.x <= 2 * width || size.y <= 2 * width)
		{
			Rectangle r = LtoP(rc);
			gc.fillOval(r.x, r.y, r.width, r.height);
		}
		else
		{
			int count = Util.minmax((int)Math.sqrt(2.5 * size.max() * avg_scale), 4, 100);
			FPoint c = LtoF(rc.center()), r = size.plus(width).times(Math.abs(scale.x / 2), Math.abs(scale.y / 2));
			FPoint i = r.minus(Math.max(width * avg_scale, 2));
			int limit = 2 * count + 1;
			int xarray[] = new int[limit + 1], yarray[] = new int[limit + 1];
			xarray[0] = xarray[count] = (int)(c.x + r.x);
			yarray[0] = yarray[count] = (int)c.y;
			xarray[count + 1] = xarray[limit] = (int)(c.x + i.x);
			yarray[count + 1] = yarray[limit] = (int)c.y;
			for(int t = 1; t < count; t++)
			{
				FPoint unit = FPoint.Polar(t * 2 * Math.PI / count);
				Point pt = unit.times(r).plus(c).asPoint();
				xarray[t] = pt.x;
				yarray[t] = pt.y;
				pt = unit.times(i).plus(c).asPoint();
				xarray[limit - t] = pt.x;
				yarray[limit - t] = pt.y;
			}
			gc.fillPolygon(xarray, yarray, limit + 1);
		}
	}

	public void        drawEllipse(FRect rc, Color outline, double outline_width, Color fill)
	{
		if(fill != null)
		{
			setColor(fill);
			Rectangle or = LtoP(rc);
			gc.fillOval(or.x, or.y, or.width, or.height);
		}
		if(outline != null)
		{
			setColor(outline);
			drawEllipse(rc, outline_width);
		}
	}

    public void drawEllipse(FRect rc, Color fill) { drawEllipse(rc, null, 0, fill); }

	private static class RawPolygon
	{
		int xlist[], ylist[];
	};

	private RawPolygon getRawPolygon(Vector points)
	{
		if(points == null || points.size() <= 1)
			return null;
		RawPolygon rp = new RawPolygon();
		int n = points.size();
		rp.xlist = new int[n];
		rp.ylist = new int[n];
		for(int i = 0; i < n; i++)
		{
			Point pt = LtoP((FPoint)points.elementAt(i));
			rp.xlist[i] = pt.x;
			rp.ylist[i] = pt.y;
		}
		return rp;
	}

	public void        drawPolyline(Vector points, Color color, double width, boolean closed)
	{
		if(points == null || points.size() <= 1)
			return;
		setColor(color);
		int n = points.size();
		if(width * avg_scale <= 1.5)
		{ // simple line
			RawPolygon rp = getRawPolygon(points);
			gc.drawPolyline(rp.xlist, rp.ylist, n);
			if(n >= 3 && closed)
				gc.drawLine(rp.xlist[n - 1], rp.ylist[n - 1], rp.xlist[0], rp.ylist[0]);
		}
		else
		{
			FPoint last = (FPoint)points.elementAt(0);
			for(int i = 1; i < n; i++)
			{
				FPoint next = (FPoint)points.elementAt(i);
				drawLine(last, next, null, width);
				last = next;
			}
			if(n >= 3 && closed)
				drawLine(last, (FPoint)points.elementAt(0), null, width);
		}
	}

	public void drawPolygon(Vector points, Color outline, double outline_width, Color fill)
	{
		if(fill == null && outline == null)
			return;
		boolean thin_edge = outline != null && (outline_width * avg_scale <= 1.5);
		if(fill != null || thin_edge)
		{
			RawPolygon rp = getRawPolygon(points);
			int n = points.size();
			if(fill != null)
			{
				setColor(fill);
				gc.fillPolygon(rp.xlist, rp.ylist, n);
			}
			if(thin_edge)
			{
				setColor(outline);
				gc.drawPolyline(rp.xlist, rp.ylist, n);
				if(n >= 3)
					gc.drawLine(rp.xlist[n - 1], rp.ylist[n - 1], rp.xlist[0], rp.ylist[0]);
			}
		}
		if(outline != null && !thin_edge)
		{
			setColor(outline);
			drawPolyline(points, null, outline_width, true);
		}
	}

	public void drawPolygon(Vector points, Color fill)
	{
		drawPolygon(points, null, 0, fill);
	}

	public  Graphics   gc;
	public  FPoint     scale;
	public  FPoint     delta;
	public  FRect      logclip;
	public  Rectangle  clip;
	public  double     avg_scale;
	public  static int dpi = Toolkit.getDefaultToolkit().getScreenResolution();
	public  double     log_dpi;
	public  Color      xor_bg;
};
