//////////////////////////////////////////////////////////////////////
// DlgColor: color editor.

#ifndef flagNODLGCOLOR

#include <utility/misc.jinc>

#define LLOG(x) // RLOG(x)

package utility;

import java.awt.*;
import java.awt.event.*;

public class DlgColor extends ModalDialog
{
	private abstract static class Palette extends Component
	{
		private static final Color palette[] =
		{
			Color.black, Color.darkGray, Color.gray, Color.lightGray, Color.white,
			Color.red, Color.green, Color.blue,
			Color.cyan, Color.magenta, Color.yellow,
			Color.orange, Color.pink,
		};

		private static final int OGAP = 4;
		private static final int GAP = 4;
		private static final int XCOUNT = 3;
		private static final int YCOUNT = 5;

		public Palette()
		{
			addMouseListener(new MouseAdapter()
			{
				public void mousePressed (MouseEvent e) { requestFocus(); setRaw(e.getPoint()); }
			});

			addMouseMotionListener(new MouseMotionAdapter()
			{
				public void mouseDragged (MouseEvent e) { setRaw(e.getPoint()); }
			});
		}

		public void setRaw(Point pt)
		{
			Dimension dim = getSize();
			int x = Util.minmax(Util.scale(pt.x - OGAP + GAP / 2, XCOUNT, dim.width  - 2 * OGAP + GAP),  0, XCOUNT - 1);
			int y = Util.minmax(Util.scale(pt.y - OGAP + GAP / 2, YCOUNT, dim.height - 2 * OGAP + GAP), 0, YCOUNT - 1);
			int l = x * YCOUNT + y;
			if(l >= 0 && l < palette.length)
				setColor(palette[l]);
		}

		public void paint(Graphics g)
		{
			Dimension dim = getSize();
			int wgap = dim.width + GAP - 2 * OGAP, hgap = dim.height + GAP - 2 * OGAP;
			GUtil.fillRect(g, 0, 0, dim.width, dim.height, SystemColor.control);
			for(int y = 0; y < YCOUNT; y++)
			{
				int y1 = OGAP + Util.scale(hgap, y, YCOUNT);
				int y2 = OGAP + Util.scale(hgap, y + 1, YCOUNT) - GAP;
				for(int x = 0; x < XCOUNT; x++)
				{
					int x1 = OGAP + Util.scale(wgap, x, XCOUNT);
					int x2 = OGAP + Util.scale(wgap, x + 1, XCOUNT) - GAP;
					int ix = x * YCOUNT + y;
					if(ix < palette.length)
					{
						if(palette[ix] != null)
						{
							GUtil.drawRect(g, x1, y1, x2 - x1, y2 - y1, 1, Color.white, Color.black);
							GUtil.fillRect(g, x1 + 1, y1 + 1, x2 - x1 - 2, y2 - y1 - 2, palette[ix]);
						}
						else
						{
							GUtil.drawRect(g, x1, y1, x2 - x1, y2 - y1, 1, Color.black);
							g.setColor(Color.black);
							g.drawLine(x1 + 1, y1 + 1, x2 - 2, y2 - 2);
							g.drawLine(x1 + 1, y2 - 2, x2 - 2, y1 + 1);
						}
					}
				}
			}
		}

		protected abstract void setColor(Color c);
	};

	private static abstract class Slider extends Component
	{
		public Slider()
		{
			LLOG("DlgColor.Slider ctr");
			addMouseListener(new MouseAdapter()
			{
				public void mousePressed (MouseEvent e) { requestFocus(); setRaw(e.getPoint().y); }
			});
			addMouseMotionListener(new MouseMotionAdapter()
			{
				public void mouseDragged (MouseEvent e)       { setRaw(e.getPoint().y); }
			});
		}

		static final int OGAP = 4;
		static final int LEFT = 5;

		public void setRaw(int y)
		{
			LLOG("setRaw(" + y + ")");
			synchronized(this) { lock++; }
			int ht = getSize().height;
			y = Util.minmax(Util.scale(ht - OGAP - y, 255, Math.max(ht - 2 * OGAP, 1)), 0, 255);
			LLOG("setRaw converted(" + y + ")");
			setColor(pos = y);
			repaint();
			synchronized(this) { lock--; }
		}

/*		public void update(Graphics g)
		{
			g.setColor(Color.pink);
			g.fillRect(0, 0, 500, 500);
			paint(g);
		}
*/
		public void paint(Graphics g)
		{
			Dimension size = getSize();
			LLOG("DlgColor.Slider.paint, size = " + size);
			int half = (LEFT + size.width) >> 1;
			int steps = 2;
			int yred = size.height - 2 * OGAP;
			while(4 * steps <= size.height && steps < 256)
				steps <<= 1;
			LLOG("steps = " + steps);
			GUtil.drawRect(g, LEFT, OGAP, size.width - LEFT, yred, 1, Color.black);
			GUtil.fillRect(g, half, OGAP, 1, yred, Color.black);
			GUtil.fillRect(g, 0, 0, LEFT, size.height, this.getBackground());
			int y0 = size.height - OGAP - 1;
			for(int i = 0; i < steps; i++)
			{
				int y1 = size.height - OGAP - 1 - Util.scale(i, yred - 2, steps - 1);
				int v = Util.scale(i, 255, steps - 1);
				LLOG("paint: i = " + i + ", v = " + v + ", color = " + getColor(v) + ", pure = " + getPureColor(v));
				GUtil.fillRect(g, LEFT + 1, y1, half - LEFT - 1, y0 - y1, getColor(v));
				GUtil.fillRect(g, half + 1, y1, size.width - half - 2, y0 - y1, getPureColor(v));
				y0 = y1;
			}
			if(pos >= 0)
			{
				y0 = size.height - OGAP - Util.scale(pos, yred, 255);
				Color fg = this.getForeground();
				GUtil.fillRect(g, 0, y0 - 3, 1, 7, fg);
				GUtil.fillRect(g, 1, y0 - 2, 1, 5, fg);
				GUtil.fillRect(g, 2, y0 - 1, 1, 3, fg);
				GUtil.fillRect(g, 3, y0, 1, 1, fg);
			}
		}

		public void setPos(int pos)
		{
			synchronized(this)
			{
				if(lock == 0)
				{
					this.pos = pos;
					repaint();
				}
			}
		}

		abstract protected Color getColor(int level);
		abstract protected Color getPureColor(int level);
		abstract protected void  setColor(int level);

		private int          pos = 0;
		private int          lock = 0;
	};

	public DlgColor(Component owner, boolean not_null, String title)
	{
		super(owner, title == null ? t_("Select color") : title);
		this.not_null = not_null;
		fadd(0, 0, 80, 100, palette = new Palette()
		{
			protected void setColor(Color c) { DlgColor.this.setColor(c); }
		});
		fadd(100,  0, 20, 14, "R:");
		fadd(120,  0, 40, 14, fr = new TextField());
		fr.addTextListener(new TextListener()
		{
			public void textValueChanged(TextEvent e)
			{
				int r;
				if(getFocusOwner() == fr && (r = tryGet(fr)) >= 0)
					setRGB(r, g0, b0, TM_SR | TM_HSV);
			}
		});
		fadd(100, 15, 60, 85, sr = new Slider()
		{
			protected Color getColor(int level)
			{ return new Color(level, g0, b0); }
			protected Color getPureColor(int level)
			{ return new Color(level, 0, 0); }
			protected void setColor(int level)
			{ DlgColor.this.setRGB(level, g0, b0, TM_FR | TM_HSV); }
		});
		fadd(170,  0, 20, 14, "G:");
		fadd(190,  0, 40, 14, fg = new TextField());
		fg.addTextListener(new TextListener()
		{
			public void textValueChanged(TextEvent e)
			{
				int g;
				if(getFocusOwner() == fg && (g = tryGet(fg)) >= 0)
					setRGB(r0, g, b0, TM_SG | TM_HSV);
			}
		});
		fadd(170, 15, 60, 85, sg = new Slider()
		{
			protected Color getColor(int level)
			{ return new Color(r0, level, b0); }
			protected Color getPureColor(int level)
			{ return new Color(0, level, 0); }
			protected void setColor(int level)
			{ DlgColor.this.setRGB(r0, level, b0, TM_FG | TM_HSV); }
		});
		fadd(240,  0, 20, 14, "B:");
		fadd(260,  0, 40, 14, fb = new TextField());
		fb.addTextListener(new TextListener()
		{
			public void textValueChanged(TextEvent e)
			{
				int b;
				if(getFocusOwner() == fb && (b = tryGet(fb)) >= 0)
					setRGB(r0, g0, b, TM_SB | TM_HSV);
			}
		});
		fadd(240, 15, 60, 85, sb = new Slider()
		{
			protected Color getColor(int level)
			{ return new Color(r0, g0, level); }
			protected Color getPureColor(int level)
			{ return new Color(0, 0, level); }
			protected void setColor(int level)
			{ DlgColor.this.setRGB(r0, g0, level, TM_FB | TM_HSV); }
		});
		fadd(320,  0, 20, 14, "H:");
		fadd(340,  0, 40, 14, fh = new TextField());
		fh.addTextListener(new TextListener()
		{
			public void textValueChanged(TextEvent e)
			{
				int h;
				if(getFocusOwner() == fh && (h = tryGet(fh)) >= 0)
					setHSV(h, s0, v0, TM_SH | TM_RGB);
			}
		});
		fadd(320, 15, 60, 85, sh = new Slider()
		{
			protected Color getColor(int level)
			{ return GUtil.hsvToRgb(level, s0, v0); }
			protected Color getPureColor(int level)
			{ return GUtil.hsvToRgb(level, 255, 255); }
			protected void setColor(int level)
			{ DlgColor.this.setHSV(level, s0, v0, TM_FH | TM_RGB); }
		});
		fadd(390,  0, 20, 14, "S:");
		fadd(410,  0, 40, 14, fs = new TextField());
		fs.addTextListener(new TextListener()
		{
			public void textValueChanged(TextEvent e)
			{
				int s;
				if(getFocusOwner() == fs && (s = tryGet(fs)) >= 0)
					setHSV(h0, s, v0, TM_SS | TM_RGB);
			}
		});
		fadd(390, 15, 60, 85, ss = new Slider()
		{
			protected Color getColor(int level)
			{ return GUtil.hsvToRgb(h0, level, v0); }
			protected Color getPureColor(int level)
			{ return GUtil.hsvToRgb(170, level, 255); }
			protected void setColor(int level)
			{ DlgColor.this.setHSV(h0, level, v0, TM_FS | TM_RGB); }
		});
		fadd(460,  0, 20, 14, "V:");
		fadd(480,  0, 40, 14, fv = new TextField());
		fv.addTextListener(new TextListener()
		{
			public void textValueChanged(TextEvent e)
			{
				int v;
				if(getFocusOwner() == fv && (v = tryGet(fv)) >= 0)
					setHSV(h0, s0, v, TM_FV | TM_RGB);
			}
		});
		fadd(460, 15, 60, 85, sv = new Slider()
		{
			protected Color getColor(int level)
			{ return GUtil.hsvToRgb(h0, s0, level); }
			protected Color getPureColor(int level)
			{ return GUtil.hsvToRgb(0, 0, level); }
			protected void setColor(int level)
			{ DlgColor.this.setHSV(h0, s0, level, TM_FV | TM_RGB); }
		});

		fadd(0, 110, 60, 16, t_("Sample:"));
		fadd(60, 110, 100, 16, sample = new ColorCtrl());
		sample.setEditable(false);

		if(!not_null)
			fadd(220, 110, 90, 16, new ActionButton(t_("Empty"))
			{ public void doAction() { setColor(null); endModalLoop(-1); } });
		fadd(330, 110, 90, 16, new ActionButton(t_("OK"))     { public void doAction() { if(accept()) endModalLoop(1); } });
		fadd(430, 110, 90, 16, new ActionButton(t_("Cancel")) { public void doAction() { endModalLoop(0); } });
	}

	public DlgColor(Component owner, boolean not_null)
	{ this(owner, not_null, null); }

	private void setColor(int text_mask)
	{
		color0 = (color == null ? Color.black : color);
		r0 = color0.getRed();
		g0 = color0.getGreen();
		b0 = color0.getBlue();
		h0 = GUtil.getHue(color0);
		s0 = GUtil.getSaturation(color0);
		v0 = GUtil.getValue(color0);
		if((text_mask & TM_SR) != 0) sr.setPos(r0);
		if((text_mask & TM_SG) != 0) sg.setPos(g0);
		if((text_mask & TM_SB) != 0) sb.setPos(b0);
		if((text_mask & TM_SH) != 0) sh.setPos(h0);
		if((text_mask & TM_SS) != 0) ss.setPos(s0);
		if((text_mask & TM_SV) != 0) sv.setPos(v0);
		if((text_mask & TM_FR) != 0) fr.setText("" + r0);
		if((text_mask & TM_FG) != 0) fg.setText("" + g0);
		if((text_mask & TM_FB) != 0) fb.setText("" + b0);
		if((text_mask & TM_FH) != 0) fh.setText("" + h0);
		if((text_mask & TM_FS) != 0) fs.setText("" + s0);
		if((text_mask & TM_FV) != 0) fv.setText("" + v0);
		sample.setColor(color);
		repaint();
	}

	private void setHSV(int h, int s, int v, int text_mask)
	{
		color = GUtil.hsvToRgb(h, s, v);
		setColor(text_mask);
	}

	private void setRGB(int r, int g, int b, int text_mask)
	{
		color = new Color(r, g, b);
		setColor(text_mask);
	}

	public void  setColor(Color c)
	{
		color = c;
		setColor(TM_ALL);
	}

	private int  tryGet(TextField tf)
	{
		try
		{
			return new Pump(true).pump(0, tf, true, 0, 255);
		}
		catch(Pump.PumpException e)
		{
			e.show();
			tf.requestFocus();
			return -1;
		}
	}

	public void  update(Graphics g)
	{
		paint(g);
	}

	public Color getColor()        { return color; }

	private static final int TM_FR   = 0x0001;
	private static final int TM_FG   = 0x0002;
	private static final int TM_FB   = 0x0004;
	private static final int TM_FH   = 0x0008;
	private static final int TM_FS   = 0x0010;
	private static final int TM_FV   = 0x0040;
	private static final int TM_FRGB = TM_FR | TM_FG | TM_FB;
	private static final int TM_FHSV = TM_FH | TM_FS | TM_FV;
	private static final int TM_FALL = TM_FRGB | TM_FHSV;

	private static final int TM_SR   = 0x0080;
	private static final int TM_SG   = 0x0100;
	private static final int TM_SB   = 0x0200;
	private static final int TM_SH   = 0x0400;
	private static final int TM_SS   = 0x0800;
	private static final int TM_SV   = 0x1000;
	private static final int TM_SRGB = TM_SR | TM_SG | TM_SB;
	private static final int TM_SHSV = TM_SH | TM_SS | TM_SV;
	private static final int TM_SALL = TM_SRGB | TM_SHSV;

	private static final int TM_RGB  = TM_FRGB | TM_SRGB;
	private static final int TM_HSV  = TM_FHSV | TM_SHSV;
	private static final int TM_ALL  = TM_SALL | TM_FALL;

	private boolean          not_null;
	private Color            color, color0;
	private int              r0, g0, b0, h0, s0, v0;
	private Palette          palette;
	private Slider           sr, sg, sb, sh, ss, sv;
	private TextField        fr, fg, fb, fh, fs, fv;
	private ColorCtrl        sample;
}

#else

class DlgColor {}

#endif//flagNODLGCOLOR
