#include <utility/misc.jinc>

package utility;

#define LLOG(x) // RLOG(x)

import java.awt.*;
import java.awt.event.*;

public class DragDropAdapter extends DragDropTarget
{
	//////////////////////////////////////////////////////////////////
	// interface

	public DragDropAdapter(Component component)
	{
		LLOG("DragDropAdapter(" + component + ")");
		owner = component;
		owner.addMouseListener(new MouseAdapter()
		{
			public void mousePressed(MouseEvent e)
			{
				LLOG("mousePressed: " + e.getPoint());
				target.isShift = e.isShiftDown();
				target.isCtrl  = e.isControlDown();
				boolean right = (e.getModifiers() & MouseEvent.BUTTON3_MASK) != 0;
				boolean drop_right = false;
				synchronized(DragDropAdapter.this) {
					undraw();
					if(start != null && dragging && right)
						drop_right = true;
				}
				owner.requestFocus();
				if(drop_right) {
					Point pt = target.doDropRight(start, e.getPoint());
					synchronized(DragDropAdapter.this) {
						start = end = pt;
						dragging = true;
						draw();
					}
				}
				else if(right)
					target.doClickRight(e.getPoint());
				else {
					boolean pushed = target.isDown = target.doPush(e.getPoint());
					synchronized(DragDropAdapter.this) {
						start = end = (pushed ? e.getPoint() : null);
						dragging = false;
					}
				}
			}

			public void mouseReleased(MouseEvent e)
			{
				LLOG("mouseReleased: " + e.getPoint());
				target.isShift = e.isShiftDown();
				target.isCtrl  = e.isControlDown();
				Point _start = start, _end = end;
				boolean _drag = dragging;
				synchronized(DragDropAdapter.this)
				{
//					LOG("mouseReleased; modifiers = " + e.getModifiers());
					if((e.getModifiers() & MouseEvent.BUTTON3_MASK) != 0)
					{
//						LOG("end of doDropRight");
						return; // end right inter-click
					}
					undraw();
					start = null;
					dragging = false;
				}
				if(_start != null)
					if(_drag)
						target.doDrop(_start, _end);
					else
						target.doClick(_start);
				target.isDown = false;
				setCursor();
			}

			//////////////////////////////////////////////////////////////////
		});

		owner.addMouseMotionListener(new MouseMotionAdapter()
		{
			public void mouseMoved(MouseEvent e)
			{
				LLOG("mouseMoved: " + e.getPoint());
				lastPos = e.getPoint();
				target.isShift = e.isShiftDown();
				target.isCtrl  = e.isControlDown();
                target.doMove(lastPos);
				setCursor();
//				LOG("WM_MOVE " + e.getPoint());
			}

			public void mouseDragged(MouseEvent e)
			{
				LLOG("mouseDragged: " + e.getPoint());
				lastPos = e.getPoint();
				target.isShift = e.isShiftDown();
				target.isCtrl  = e.isControlDown();
				synchronized(DragDropAdapter.this)
				{
//	            	LOG("mouseDragged");
					Point pt = e.getPoint();
					if(!dragging && start != null)
					{ // check distance
						if(dragging = Util.dist2(start, pt) >= TOLERANCE2)
							draw(pt);
					}
					else if(!pt.equals(end))
						draw(pt);
					setCursor();
				}
			}

					public void mouseExited(MouseEvent e)
			{
				LLOG("mouseExited: " + e.getPoint());
				cancel();
			}

			//////////////////////////////////////////////////////////////////
		});

		owner.addKeyListener(new KeyAdapter()
		{
			public void keyPressed(KeyEvent e)
			{
				if(e.getKeyCode() == KeyEvent.VK_ESCAPE)
					cancel();
			}
		});
	}

	public synchronized void cancel()
	{
		undraw();
		target.isDown = false;
		dragging = false;
		if(start != null)
		{
			Point old = start;
			start = null;
			target.doCancel();
			setCursor();
		}
	}

	public synchronized void reset()
	{
		drawn = false;
		start = null;
		dragging = false;
	}

	public DragDropTarget getTarget()   { return target; }
	public boolean        isDrawn()     { return drawn; }
	public boolean        isDragging()  { return dragging; }

	public synchronized void setTarget(DragDropTarget target)
	{
		cancel();
		this.target = (target != null ? target : this);
			setCursor();
	}

	public void setCursor()
	{
		if(lastPos != null)
		{
			Cursor cursor = target.getCursor(lastPos);
			owner.setCursor(cursor != null ? cursor : Cursor.getDefaultCursor());
		}
	}

	public synchronized void undraw()
	{
		if(drawn && start != null)
			draw(null);
	}

	public synchronized void draw(Point _end)
	{
		if(start != null)
		{
			target.doDrag(start, drawn ? end : null, _end);
			end = _end;
			drawn = (end != null);
		}
	}

	public synchronized void draw() { draw(end); }

	public static final int TOLERANCE2 = 20; // abs2(distance)

	private DragDropTarget target = this;
	private boolean drawn = false, dragging = false;
	private Point lastPos, start, end;
	private Component owner;
}
