///////////////////////////////////////////////////////////////////
// ScrollView: automated child view scrollbars.

#include <utility/misc.jinc>

#define LLOG(x) // RLOG(x)

package utility;

import java.awt.*;
import java.awt.event.*;

public class ScrollView extends Panel
{
	// alignments
	public static final byte LEFT   = 0;
	public static final byte CENTER = 1;
	public static final byte RIGHT  = 2;

	public static final byte TOP    = 0;
	public static final byte BOTTOM = 2;

	CATCH_EVENTS()

	public ScrollView()
	{
		super(null);
		setBackground(Color.white);
		this.addKeyListener(new KeyAdapter()
		{
			public void keyPressed(KeyEvent e)
			{
				if(isHorzScroll())
					sb_horz.doKey(e);
				if(isVertScroll())
					sb_vert.doKey(e);
			}
		});
		addMouseListener(new MouseAdapter()
		{
			public void mousePressed(MouseEvent e)
			{
				if(wants_focus) {
					LLOG("ScrollView.requestFocus");
					requestFocus();
				}
			}
		});

		sb_horz = new ScrollBar(true);
		sb_horz.setBackground(sb_background);
		sb_horz.setForeground(sb_foreground);
		add(sb_horz);
	   	sb_horz.addScrollListener(new ScrollAdapter()
		{ public void doScroll(boolean user) { ScrollView.this.doScroll(user); } });

		sb_vert = new ScrollBar(false);
		sb_vert.setBackground(sb_background);
		sb_vert.setForeground(sb_foreground);
		add(sb_vert);
		sb_vert.addScrollListener(new ScrollAdapter()
		{ public void doScroll(boolean user) { ScrollView.this.doScroll(user); } });

		setScrollSize(0, 0);
	}

	public final boolean isOpen()                   { return getParent() != null; }
	public final boolean isBackDraw()               { return backdraw; }
	public final void    setBackDraw(boolean _back) { backdraw = _back; }
	public final void    setBackDraw()              { setBackDraw(true); }
	public final boolean isAutoHide()               { return auto_hide; }
	public final void    setAutoHide(boolean ah)    { auto_hide = ah; }

	public final void setScrollColor(Color fg, Color bg)
	{
		LLOG("scroll foreground = " + fg + ", back = " + bg);
		sb_foreground = fg;
		sb_background = bg;
		if(sb_horz != null) { sb_horz.setBackground(bg); sb_horz.setForeground(fg); }
		if(sb_vert != null) { sb_vert.setBackground(bg); sb_vert.setForeground(fg); }
	}

	public final byte getHorzAlignment() { return align_horz; }
	public final void setHorzAlignment(byte _alignment)
	{
		if(align_horz != _alignment)
		{
			align_horz = _alignment;
			repaint();
		}
	}

	public final byte getVertAlignment() { return align_vert; }
	public final void setVertAlignment(byte _alignment)
	{
		if(align_vert != _alignment)
		{
			align_vert = _alignment;
			repaint();
		}
	}

	public final void setAlignment(byte _horz, byte _vert)
	{ setHorzAlignment(_horz); setVertAlignment(_vert); }

	public final boolean isHorzScroll() { return is_horz; }
	public final boolean isHorzScrollActive() { return is_horz && sb_horz.isVisible(); }
	public final void    setHorzScroll(boolean _active)
	{
		if(is_horz != _active)
		{
			is_horz = _active;
			if(isOpen())
			{
//				repaint_layout();
				invalidate();
				validate();
//				doLayout();
			}
		}
	}

	public final boolean isVertScroll() { return is_vert; }
	public final boolean isVertScrollActive() { return is_vert && sb_vert.isVisible(); }
	public final void    setVertScroll(boolean _active)
	{
		if(is_vert != _active)
		{
			is_vert = _active;
			if(isOpen())
			{
//				repaint_layout();
				invalidate();
				validate();
//				doLayout();
			}
		}
	}

	public Dimension getMinimumSize()
	{
		Insets insets = getInsets();
		return new Dimension(
			3 * border_size.width + insets.left + insets.right,
			3 * border_size.height + insets.top + insets.bottom);
	}

	public Dimension getPreferredSize() { return preferred_size; }
	public void setPreferredSize(Dimension _size) { preferred_size = _size; }

	public Dimension getContentSize() { return content_size; }
	public Dimension getViewportSize()
	{
		Insets insets = getInsets();
		Dimension dim = getSize();
		return Util.sub(dim, insets.left + insets.right, insets.top + insets.bottom);
	}

	public Dimension getChildSize()
	{
//		LLOG("ScrollView.getChildSize");
//		DUMP(content_size);
//		DUMP(border_size);
		return Util.sub(content_size, Util.shl(border_size, 1));
	}

	public Dimension getMaxFullSize(boolean with_sb)
	{
		Dimension size = Util.sub(getViewportSize(), Util.shl(border_size, 1));
		if(with_sb)
		{
			if(is_horz) size.height -= bar_size.height;
			if(is_vert) size.width -= bar_size.width;
		}
		return size;
	}

	public Dimension getMaxFullSize() { return getMaxFullSize(!auto_hide); }

	public Dimension getScrollSize() { return scroll_size; }
	public void setScrollSize(int w, int h) { setScrollSize(new Dimension(w, h)); }
	public void setScrollSize(Dimension _size)
	{
//		if(!scroll_size.equals(_size))
		{
			scroll_size = _size;
			LLOG("setScrollSize: " + scroll_size + ", open = " + isOpen());
			if(isOpen())
			{
//				repaint_layout();
//				invalidate();
//				validate();
				LLOG("calling doLayout");
				doLayout();
				LLOG("//doLayout");
			}
		}
	}

	public Dimension getBorderSize() { return border_size; }
	public void setBorderSize(int w, int h) { setBorderSize(new Dimension(w, h)); }
	public void setBorderSize(Dimension _size)
	{
		if(!border_size.equals(_size))
		{
			border_size = _size;
			if(isOpen())
			{
//				repaint_layout();
				invalidate();
				validate();
//				doLayout();
			}
		}
	}

	public final Color getBorderColor() { return border_color; }
	public final void setBorderColor(Color _color)
	{
		border_color = _color;
		repaint();
	}

	public synchronized Point getScrollPosition()
	{
		Point pos = new Point(0, 0);
		if(is_horz && sb_horz.isVisible())
			pos.x = sb_horz.getValue();
		if(is_vert && sb_vert.isVisible())
			pos.y = sb_vert.getValue();
		return pos;
	}

	public Point getCenterPosition()
	{
		return clientToChild(Util.asPoint(Util.half(getContentSize())));
	}

	public void setScrollPosition(Point pos, boolean user) { setScrollPosition(pos.x, pos.y, user); }
	public void setScrollPosition(int x, int y, boolean user)
	{
		if(is_horz && sb_horz.isVisible())
			sb_horz.setValue(x);
		if(is_vert && sb_vert.isVisible())
			sb_vert.setValue(y);
		doScroll(user);
	}

	public Point getChildOffset()            { return old_offset; }

	public Dimension getPageIncrement() { return page; }
	public void setPageIncrement(Dimension _page)
	{
		if(!page.equals(_page))
		{
			page = _page;
			if(isOpen())
			{
				repaint_layout();
//				invalidate();
//				validate();
//				doLayout();
			}
		}
	}

	public Dimension getLineIncrement() { return line; }
	public void setLineIncrement(Dimension _line)
	{
		if(!line.equals(_line))
		{
			line = _line;
			if(isOpen())
				repaint_layout();
		}
	}

	public boolean isWantFocus() { return wants_focus; }
	public void setWantFocus(boolean _want_focus)
	{
		wants_focus = _want_focus;
	}

	public void centerInto(Point p)
	{
		LLOG("ScrollView.centerInto " + p + ", scroll size = " + getScrollSize());
		setScrollPosition(Util.add(getScrollPosition(), Util.sub(p, getCenterPosition())), false); //Util.add(Util.sub(p, Util.half(getContentSize())), border_size));
		LLOG("//ScrollView.centerInto " + p + ", center pos = " + getCenterPosition());
	}

	public void scrollInto(Point p) { scrollInto(new Rectangle(p.x, p.y, 1, 1)); }
	public void scrollInto(Rectangle rc)
	{
		Point co = getChildOffset();
		rc = Util.moved(rc, co);
		Point pos = new Point(getScrollPosition());
		if(rc.x < 0)
   		 	pos.x += rc.x;
		else if(rc.x + rc.width > content_size.width)
   			pos.x += rc.x + rc.width - content_size.width;
	   	if(rc.y < 0)
			pos.y += rc.y;
		else if(rc.y + rc.height > content_size.height)
			pos.y += rc.y + rc.height - content_size.height;
		setScrollPosition(pos, false);
	}

	public Point clientToChild(Point pt)
	{
		if(pt == null)
			return null;
		Point co = getChildOffset();
		return new Point(pt.x - co.x, pt.y - co.y);
	}

	public Rectangle clientToChild(Rectangle r)
	{
		if(r == null)
			return null;
		Point pt = getChildOffset();
		return new Rectangle(r.x - pt.x, r.y - pt.y, r.width, r.height);
	}

	public Point childToClient(Point pt)
	{
		if(pt == null)
			return null;
		Point co = getChildOffset();
		return new Point(pt.x + co.x, pt.y + co.y);
	}

	public Rectangle childToClient(Rectangle r)
	{
		if(r == null)
			return null;
		Point pt = getChildOffset();
		return new Rectangle(r.x + pt.x, r.y + pt.y, r.width, r.height);
	}

	public Rectangle getClientArea()
	{
		return new Rectangle(getInsets().left, getInsets().top,
			content_size.width, content_size.height);
	}

	public Graphics getChildGraphics()
	{
		Graphics g = getGraphics();
		Insets insets = getInsets();
		Dimension vport = getContentSize();
		Point pt = getChildOffset();
		Rectangle rc = Util.minmax(new Rectangle(pt, scroll_size),
			new Rectangle(insets.left, insets.top, vport.width, vport.height));
		g.clipRect(rc.x, rc.y, rc.width, rc.height);
		g.translate(pt.x, pt.y);
		return g;
	}

	public void doScroll(boolean user)
	{
		if(calc_layout || !enable_scroll)
			return;
//		LLOG("ScrollView.doScroll");
		if(wants_focus) {
			LLOG("ScrollView.requestFocus");
			requestFocus();
		}
		Insets insets = getInsets();
		Point pos = calcChildOffset();
		Dimension amount = Util.sub(pos, getChildOffset());
		if(amount.width == 0 && amount.height == 0)
			return;

		synchronized(this)
		{
			if(scrolling)
			{
				old_offset = pos;
				repaint();
				return;
			}
			scrolling = true;
		}

        LLOG("doScroll pos = " + pos);
		old_offset = pos;
		if(Math.abs(2 * amount.width) >= content_size.width
		|| Math.abs(2 * amount.height) >= content_size.height
		|| backdraw)
		{
			repaint(insets.left, insets.top, content_size.width, content_size.height);
			scrolling = false;
			return;
		}
		Graphics g = getGraphics();
		g.clipRect(insets.left, insets.top, content_size.width, content_size.height);
//		LLOG("scroll amount = " + amount.toString());
		g.copyArea(insets.left, insets.top, content_size.width, content_size.height,
			amount.width, amount.height);
		if(amount.width > 0)
		{
			Graphics gm = g.create();
			gm.clipRect(insets.left, insets.top, amount.width, content_size.height);
			paint(gm);
			gm.dispose();
		}
		else if(amount.width < 0)
		{
			Graphics gm = g.create();
			gm.clipRect(insets.left + content_size.width + amount.width, insets.top, -amount.width, content_size.height);
			paint(gm);
			gm.dispose();
		}
		if(amount.height > 0)
		{
			Graphics gm = g.create();
			gm.clipRect(insets.left, insets.top, content_size.width, amount.height);
			paint(gm);
			gm.dispose();
		}
		else if(amount.height < 0)
		{
			Graphics gm = g.create();
			gm.clipRect(insets.left, insets.top + content_size.height + amount.height, content_size.width, -amount.height);
			paint(gm);
			gm.dispose();
		}
		g.dispose();
		scrolling = false;
//		LLOG("/ScrollView.doScroll");
	}

	public void repaintChild(Rectangle r) { repaintChild(r.x, r.y, r.width, r.height); }
	public void repaintChild(int x, int y, int w, int h)
	{
		Point co = getChildOffset();
		repaint(x + co.x, y + co.y, w, h);
	}

	public void paintChild(Graphics g) {}

	//////////////////////////////////////////////////////////////////////

	public synchronized void repaint_layout()
	{
		dirty_layout = true;
		LLOG("Splitter.repaint_layout -> dirty_layout = " + dirty_layout + ", this = " + this);
		repaint();
	}

//	private int paint_index = 0;

	private boolean painting = false;

	public synchronized void paint(Graphics g)
	{
		LLOG("ScrollView.paint");
		synchronized(this)
		{
			if(painting)
				return;
			painting = true;
		}
		if(dirty_layout)
		{
			LLOG("ScrollView.paint -> layout");
			dirty_layout = false;
			invalidate();
			validate();
//			doLayout();
			LLOG("//ScrollView.paint -> layout");
		}
		old_offset = getChildOffset();
		LLOG("ScrollView.paint - getClipBounds");
		Rectangle rc = g.getClipBounds();
		if(rc == null)
			rc = new Rectangle(new Point(0, 0), getSize());
//		LLOG("ScrollView.paint - insets");
		Rectangle crc = getClientArea();
//		LLOG("ScrollView.paint - getChildOffset");
		Point co = getChildOffset();
//		LLOG("ScrollView.paint - co / scroll_size, co = " + co + ", scroll_size = " + scroll_size);
		Rectangle ch = new Rectangle(co, scroll_size);
		LLOG("ScrollView.paint - sb tests");
		if(is_horz && is_vert && sb_horz.isVisible() && sb_vert.isVisible())
		{
			Dimension strut = Util.sub(getViewportSize(), content_size);
			if(strut.width > 0 && strut.height > 0
			&& (rc.x + rc.width > crc.x + crc.width || rc.y + rc.height > crc.y + crc.height))
			{
				Insets insets = getInsets();
				GUtil.fillRect(g, insets.left + content_size.width, insets.top + content_size.height,
					strut.width, strut.height, sb_foreground);
			}
		}

		LLOG("ScrollView.paint - intersect tests, ch = " + ch + ", crc = " + crc);
		if(!Util.contains(ch, crc))
		{
//        	LLOG("ch:  " + ch.toString());
//            LLOG("crc: " + crc.toString());
			g.setColor(border_color);
			GUtil.drawRect(g, ch, crc);
		}
		rc = Util.minmax(ch, Util.minmax(crc, rc));
		if(rc.isEmpty())
		{
			painting = false;
			return;
		}

		Graphics gd = g;
		Image backimage = null;
		LLOG("ScrollView.paint - backdraw");
		if(backdraw)
		{
			gd = (backimage = createImage(rc.width, rc.height)).getGraphics();
			gd.translate(-rc.x, -rc.y);
			LLOG("ScrollView.paint: backimage translation = " + rc);
		}

		gd.clipRect(rc.x, rc.y, rc.width, rc.height);

		LLOG("ScrollView.paint - paintChild");
//		Color colors[] = { Color.red, Color.green, Color.blue, Color.yellow, Color.magenta };
//		if(++paint_index >= colors.length)
//        	paint_index = 0;
		gd.setColor(getBackground());
		gd.fillRect(rc.x, rc.y, rc.width, rc.height);
		gd.setColor(getForeground());
		gd.translate(co.x, co.y);
		LLOG("gd translated " + co);
		paintChild(gd);

		LLOG("ScrollView.paint - backdraw flush");
		if(backdraw)
		{
			gd.dispose();
			g.drawImage(backimage, rc.x, rc.y, null);
			backimage.flush();
		}

		painting = false;
		LLOG("//ScrollView.paint");
	}

	public void update(Graphics g) { paint(g); }

	public boolean isFocusTraversable() { return true; }

//	private static int dump_count = 3;

	public synchronized void doLayout()
	{
		LLOG("ScrollView.doLayout (raw)");
		synchronized(this)
		{
			if(calc_layout)
				return;
			calc_layout = true;
		}

		Dimension size = getSize();
//		DUMP(size);
		Insets insets = getInsets();
		LLOG("ScrollView.doLayout, size = " + size + ", insets = " + insets + " @ " + this);
//		DUMP(insets);

		content_size = Util.sub(size,
			insets.left + insets.right, insets.top + insets.bottom);
		LLOG("ScrollView.doLayout, content_size = " + content_size);
//		DUMP(content_size);

		boolean h = false, v = false;
		if(is_horz && bar_size.height == 0)
			bar_size.height = sb_horz.getPreferredSize().height;
		if(is_vert && bar_size.width == 0)
			bar_size.width = sb_vert.getPreferredSize().width;
		Dimension outer_size = Util.add(scroll_size, Util.shl(border_size, 1));
/*
//        if(content_size.width >= 300 && this instanceof webmap.ThemeListCtrl)
//          LLOG("Too wide!");
		LLOG("Data size: " + outer_size);
		LLOG("View size: " + content_size);
		LLOG("is_horz " + is_horz + ", is_vert " + is_vert);
//*/
		if(!auto_hide || is_horz && outer_size.width > content_size.width)
		{
			h = true;
			content_size.height -= bar_size.height;
		}
		if(!auto_hide || is_vert && outer_size.height > content_size.height)
		{
			v = true;
			content_size.width -= bar_size.width;
		}
		if(!h && is_horz && outer_size.width > content_size.width)
		{
			h = true;
			content_size.height -= bar_size.height;
		}
		content_size = Util.max(content_size, 1);

		LLOG("ScrollView.doLayout, final content_size = " + content_size);

/*
		LLOG("Data size: " + outer_size);
		LLOG("View size: " + content_size);
		LLOG("horz " + (h ? "on" : "off"));
		LLOG("vert " + (v ? "on" : "off"));
//*/
		if(h)
		{
			sb_horz.setBounds(insets.left, size.height - insets.bottom
				- bar_size.height, content_size.width, bar_size.height);
//			LLOG("hbar bounds: " + sb_horz.getBounds());
			sb_horz.setMinimum(0);
			sb_horz.setMaximum(outer_size.width);
			sb_horz.setVisibleAmount(content_size.width);
			LLOG("hbar <- max = " + outer_size.width + ", visible " + content_size.width
				+ ", set " + sb_horz.getVisibleAmount());
			sb_horz.setBlockIncrement(page.width != 0 ? page.width : content_size.width);
			sb_horz.setUnitIncrement(line.width != 0 ? line.width : content_size.width >> 4);
			sb_horz.invalidate();
			sb_horz.validate();
			LLOG("hbar max " + sb_horz.getMaximum() + ", visible " + sb_horz.getVisibleAmount()
				+ ", self " + this);
//			sb_horz.doLayout();
		}
//        LLOG("horz bounds = " + sb_horz.getBounds());
//        LLOG("horz visible " + (sb_horz.isVisible() ? "yes" : "no"));
		if(v)
		{
			sb_vert.setBounds(size.width - insets.right
				- bar_size.width, insets.top, bar_size.width, content_size.height);
			sb_vert.setMaximum(outer_size.height);
			sb_vert.setVisibleAmount(content_size.height);
			sb_vert.setBlockIncrement(page.height != 0 ? page.height : content_size.height);
			sb_vert.setUnitIncrement(line.height != 0 ? line.height : content_size.height >> 4);
			sb_vert.invalidate();
			sb_vert.validate();
			LLOG("vbar max " + sb_vert.getMaximum() + ", visible " + sb_vert.getVisibleAmount()
				+ ", self " + this);
//			sb_vert.doLayout();
		}
		if(h != sb_horz.isVisible())
		{
//			LLOG("sb_horz -> setVisible");
			sb_horz.setVisible(h);
//			LLOG("/sb_horz -> setVisible");
//            sb_horz.doLayout();
		}
		if(v != sb_vert.isVisible())
		{
//			LLOG("sb_vert -> setVisible");
			sb_vert.setVisible(v);
//			LLOG("/sb_vert -> setVisible");
//            sb_vert.doLayout();
		}
		old_offset = calcChildOffset();
		repaint();

//        Util.LLOG("vscroll = " + sb_vert.getBounds().toString());

		dirty_layout = false;
		calc_layout = false;
		enable_scroll = true;
		LLOG("/ScrollView.doLayout");
	}

	private Point calcChildOffset()
	{
	   	Point pos = getScrollPosition();
		Dimension port = getContentSize();

		if(is_horz && scroll_size.width > port.width)
			pos.x = border_size.width - pos.x;
		else
			switch(align_horz)
			{
			case LEFT:  pos.x = border_size.width; break;
			case RIGHT: pos.x = port.width - border_size.width - scroll_size.width; break;
			default:    pos.x = ((port.width - scroll_size.width) >> 1); break;
			}

		if(is_vert && scroll_size.height > port.height)
			pos.y = border_size.height - pos.y;
		else
			switch(align_vert)
			{
			case TOP:    pos.y = border_size.height; break;
			case BOTTOM: pos.y = port.height - border_size.height - scroll_size.height; break;
			default:     pos.y = ((port.height - scroll_size.height) >> 1); break;
			}

		return pos;
	}

	private Dimension bar_size = new Dimension(0, 0);
	private Dimension line = new Dimension(0, 0), page = new Dimension(0, 0);
	private Dimension preferred_size = new Dimension(400, 300);
	private byte align_horz = CENTER, align_vert = CENTER;
	private boolean wants_focus = false;
	private boolean is_horz = true, is_vert = true, auto_hide = true;
	private ScrollBar sb_horz, sb_vert;
	private Color sb_background = SystemColor.controlShadow, sb_foreground = SystemColor.control;
	private Dimension content_size = new Dimension(1, 1);
	private Dimension scroll_size = new Dimension(1000, 1000);
	private Dimension border_size = new Dimension(5, 5);
	private Color border_color = Color.gray;
	private Point old_offset = new Point(0, 0);
	private boolean backdraw = false, scrolling = false, calc_layout = false, enable_scroll = false;
	private boolean dirty_layout = true;
}
