///////////////////////////////////////////////////////////////////
// Util: miscellaneous functions.

#include <utility/misc.jinc>

#if defined(_DEBUG) && defined(flagDOLOG)
	#define XXXDOLOGXXX
#endif

#define LLOG(x) // RLOG(x)

package utility;

import java.awt.*;
import java.net.*;
import java.io.*;

public class Util
{
	public static final int ALIGN_LEFT   = 0;
	public static final int ALIGN_CENTER = 1;
	public static final int ALIGN_RIGHT  = 2;
	public static final int ALIGN_FULL   = 3;

	public static boolean isNull(int i)    { return i == INT_NULL; }
	public static boolean isNull(double d) { return d == DBL_NULL; }
	public static boolean isNull(String s) { return s == null || s.length() == 0; }
	public static String  nvl(String a, String b) { return isNull(a) ? b : a; }

	public static String leftStr(String s, int start, int len)
	{
		if(s == null || s.length() <= start)
			return "";
		return s.substring(start, Math.min(start + len, s.length()));
	}

	public static String formatIntHex(int i, int digits)
	{
		char temp[] = new char[digits];
		for(int e = digits; --e >= 0; i >>= 4)
			temp[e] = (char)((i & 15) <= 9 ? '0' + (i & 15) : 'A' + ((i & 15) - 10));
		return new String(temp);
	}

	public static int atoi(String s)            { return atoi(s, 0, INT_NULL); }
	public static int atoi(String s, int start) { return atoi(s, start, INT_NULL); }
	public static int atoi(String s, int start, int dflt)
	{
		boolean negative = false;
		if(s == null || s.length() <= start)
			return dflt;
		if(s.charAt(start) == '+' || s.charAt(start) == '-')
			negative = (s.charAt(start++) == '-');
		if(start >= s.length())
			return dflt;
		char dig = s.charAt(start++);
		if(dig < '0' || dig > '9')
			return dflt;
		int value = dig - '0';
		while(start < s.length() && (dig = s.charAt(start++)) >= '0' && dig <= '9')
			value = 10 * value + dig - '0';
		return negative ? -value : value;
	}

	public static int ctoi(char c)
	{
		if(c >= '0' && c <= '9')
			return c - '0';
		if(c >= 'A' && c <= 'Z')
			return c - 'A' + 10;
		if(c >= 'a' && c <= 'z')
			return c - 'a' + 10;
		return 255;
	}

	public static boolean isdigit(char c)  { return c >= '0' && c <= '9'; }
	public static boolean isxdigit(char c) { return ctoi(c) < 16; }
	public static boolean isalpha(char c)  { return c >= 'A' && c <= 'Z' || c >= 'a' && c <= 'z'; }
	public static boolean isalnum(char c)  { return isalpha(c) || isdigit(c); }

	public static int atohex(String s, int start)
	{
		int l;
		if(s == null || start >= (l = s.length()) || !isxdigit(s.charAt(start)))
			return INT_NULL;
		int out = 0;
		while(start < l && isxdigit(s.charAt(start)))
			out = out * 16 + ctoi(s.charAt(start++));
		return out;
	}

	public static String ctohex(Color c)
	{
		return formatIntHex(c.getRed(), 2) + formatIntHex(c.getGreen(), 2) + formatIntHex(c.getBlue(), 2);
	}

	public static double atof(String s) { return atof(s, 0, DBL_NULL); }
	public static double atof(String s, int start) { return atof(s.substring(start)); }
	public static double atof(String s, int start, double dflt)
	{
		return atofp(s, start, dflt).value;
	}

	public static class ValPos {
		public double value;
		public int    endpos;
	};
	public static ValPos atofp(String s, int start, double dflt)
	{
		ValPos vp = new ValPos();
		vp.value = dflt;
		vp.endpos = start;

//		LOG("atof: " + leftStr(s, start, 20));
		if(s == null || s.length() <= start)
			return vp;
		int i = start, l = s.length();
		if(s.charAt(i) == '+' || s.charAt(i) == '-')
			i++;
		if(i >= s.length() || !isdigit(s.charAt(i)))
			return vp;
		while(i < l && isdigit(s.charAt(i)))
			i++;
		if(i < l && s.charAt(i) == '.')
			while(++i < l && isdigit(s.charAt(i)))
				;
		if(i < l && (s.charAt(i) == 'e' || s.charAt(i) == 'E'))
		{
			if(++i >= l)
				return vp;
			char c = s.charAt(i);
			if(c == '+' || c == '-')
			{
				if(++i >= l)
					return vp;
				c = s.charAt(i);
			}
			if(!isdigit(c))
				return vp;
			while(++i < l && isdigit(s.charAt(i)))
				;
		}
		vp.endpos = i;
		if(start == 0 && i == l)
			vp.value = Double.valueOf(s).doubleValue();
		else
			vp.value = Double.valueOf(s.substring(start, i)).doubleValue();
		return vp;
	}

	public static double atoffrac(String s) { return atoffrac(s, 0, DBL_NULL); }
	public static double atoffrac(String s, int start) { return atoffrac(s.substring(start)); }
	public static double atoffrac(String snom, int start, double dflt)
	{
		String sden = "1";
		int icol = start;
		while(icol < snom.length() && snom.charAt(icol) != ':')
			icol++;
		if(icol < snom.length()) {
			sden = snom.substring(icol + 1);
			snom = snom.substring(start, icol);
		}
		LLOG("snom = " + snom + ", sden = " + sden);
		double n = atof(snom);
		double d = atof(sden);
		if(n != Util.DBL_NULL && d != Util.DBL_NULL && d != 0.0)
			n /= d;
		return n;
	}

	public static String asCString(String s)
	{
		if(s == null)
			return null;
		StringBuffer sb = new StringBuffer();
		sb.append('\"');
		for(int i = 0, n = s.length(); i < n; i++) {
			char c = s.charAt(i);
			if(c >= ' ' && c <= 127)
			{
				if(c == '\\' || c == '\"' || c == '\'')
					sb.append('\\');
				sb.append(c);
			}
			else
				switch(c)
				{
				case '\b': sb.append("\\b"); break;
				case '\f': sb.append("\\f"); break;
				case '\n': sb.append("\\n"); break;
				case '\r': sb.append("\\r"); break;
				default:
					sb.append("\\x");
					sb.append("0123456789ABCDEF".charAt((c >> 4) & 15));
					sb.append("0123456789ABCDEF".charAt((c >> 0) & 15));
					break;
				}
		}
		sb.append('\"');
		return sb.toString();
	}

	public static Dimension asSize(Point a)      { return new Dimension(a.x, a.y); }
	public static Dimension asSize(int a)        { return new Dimension(a, a); }
	public static Point     asPoint(Dimension a) { return new Point(a.width, a.height); }

	public static int minmax(int x, int l, int h)
	{
		if(x > h) x = h;
		if(x < l) x = l;
		return x;
	}

	public static int rminmax(double x, int l, int h)
	{
		if(x > h) x = h;
		if(x < l) x = l;
		return (int)Math.round(x);
	}

	public static double fminmax(double x, double l, double h)
	{
		if(x > h) x = h;
		if(x < l) x = l;
		return x;
	}

	public static final int fround(double d)
	{
		long l = Math.round(d);
		return l >=  0x7FFFFFFFL ?  0x7FFFFFFF
		:      l <= -0x80000000L ? -0x80000000
		: (int)l;
	}

	public static double hypot(double x, double y) { return Math.sqrt(x * x + y * y); }

	public static Point minmax(Point p, Point l, Point h)
	{ return new Point(minmax(p.x, l.x, h.x), minmax(p.y, l.y, h.y)); }

	public static Dimension minmax(Dimension p, Dimension l, Dimension h)
	{ return new Dimension(minmax(p.width, l.width, h.width), minmax(p.height, l.height, h.height)); }

	public static Dimension minmax(Dimension p, int l, int h)
	{ return new Dimension(minmax(p.width, l, h), minmax(p.height, l, h)); }

	public static Rectangle minmax(Rectangle inside, Rectangle rc)
	{
		if(!rc.intersects(inside))
			return new Rectangle(inside.x, inside.y, 0, 0);
		int x = max(rc.x, inside.x);
		int y = max(rc.y, inside.y);
		int w = min(rc.x + rc.width, inside.x + inside.width) - x;
		int h = min(rc.y + rc.height, inside.y + inside.height) - y;
		return new Rectangle(x, y, w, h);
	}

	public static Rectangle minmax(Dimension size, Rectangle rc)
	{
		return minmax(new Rectangle(size), rc);
	}

	public static int scale(int x, int m, int d)
	{
		return (int)(x * (long)m / d);
	}

	public static int scale(int x, double y)
	{
		return (int)Math.round(x * y);
	}

	public static Point scale(Point x, Dimension m, Dimension d)
	{ return new Point(scale(x.x, m.width, d.width), scale(x.y, m.height, d.height)); }

	public static Point scale(Point x, int m, int d)
	{ return new Point(scale(x.x, m, d), scale(x.y, m, d)); }

	public static Point scale(Point x, double t)
	{ return new Point(scale(x.x, t), scale(x.y, t)); }

	public static Dimension scale(Dimension x, Dimension m, Dimension d)
	{ return new Dimension(scale(x.width, m.width, d.width), scale(x.height, m.height, d.height)); }

	public static Dimension scale(Dimension x, int m, int d)
	{ return new Dimension(scale(x.width, m, d), scale(x.height, m, d)); }

	public static Dimension scale(Dimension x, double t)
	{ return new Dimension(scale(x.width, t), scale(x.height, t)); }

	public static Rectangle scale(Rectangle rc, Dimension m, Dimension d)
	{
		Point lt = scaleFloor(rc.getLocation(), m, d);
		Point rb = scaleCeil(add(rc.getLocation(), rc.getSize()), m, d);
		return new Rectangle(lt, sub(rb, lt));
	}

	public static int scaleCeil(int x, int m, int d)
	{
		return (int)((x * (long)m + (x >= 0 ? d - 1 : 0)) / d);
	}

	public static Point scaleCeil(Point x, Dimension m, Dimension d)
	{ return new Point(scaleCeil(x.x, m.width, d.width), scaleCeil(x.y, m.height, d.height)); }

	public static Point scaleCeil(Point x, int m, int d)
	{ return new Point(scaleCeil(x.x, m, d), scaleCeil(x.y, m, d)); }

	public static int scaleFloor(int x, int m, int d)
	{
		return (int)((x * (long)m + (x >= 0 ? 0 : 1 - d)) / d);
	}

	public static Point scaleFloor(Point x, Dimension m, Dimension d)
	{ return new Point(scaleFloor(x.x, m.width, d.width), scaleFloor(x.y, m.height, d.height)); }

	public static Point scaleFloor(Point x, int m, int d)
	{ return new Point(scaleFloor(x.x, m, d), scaleFloor(x.y, m, d)); }

	public static Dimension ratioSize(Dimension a, Dimension b)
	{
		int m, d;
		if(vector(a, b) >= 0)
		{
			m = b.height;
			d = a.height;
		}
		else
		{
			m = b.width;
			d = a.width;
		}
		return d != 0 ? scale(a, m, d) : a;
	}

	public static Rectangle div(Rectangle rc, Dimension d)
	{
		return scale(rc, new Dimension(1, 1), d);
	}

	public static int min(int a, int b)
	{
		return a <= b ? a : b;
	}

	public static Point min(Point a, Point b)
	{
		return new Point(a.x < b.x ? a.x : b.x, a.y < b.y ? a.y : b.y);
	}

	public static Dimension min(Dimension a, Dimension b)
	{
		return new Dimension(Math.min(a.width, b.width), Math.min(a.height, b.height));
	}

	public static Dimension min(Dimension a, int b)
	{
		return new Dimension(Math.min(a.width, b), Math.min(a.height, b));
	}

	public static int max(int a, int b)
	{
		return a >= b ? a : b;
	}

	public static Point max(Point a, Point b)
	{
		return new Point(a.x > b.x ? a.x : b.x, a.y > b.y ? a.y : b.y);
	}

	public static Dimension max(Dimension a, Dimension b)
	{
		return new Dimension(Math.max(a.width, b.width), Math.max(a.height, b.height));
	}

	public static Dimension max(Dimension a, int b)
	{
		return new Dimension(Math.max(a.width, b), Math.max(a.height, b));
	}

	public static Point add(Point a, Dimension b)
	{
		return new Point(a.x + b.width, a.y + b.height);
	}

	public static Point add(Point a, Point b)
	{
		return new Point(a.x + b.x, a.y + b.y);
	}

	public static Point mid(Point a, Point b)
	{
		return new Point((a.x + b.x) >> 1, (a.y + b.y) >> 1);
	}

	public static Dimension halfDist(Point a, Point b)
	{
		return new Dimension((b.x - a.x) >> 1, (b.y - a.y) >> 1);
	}

	public static Point add(Point a, int x, int y)
	{
		return new Point(a.x + x, a.y + y);
	}

	public static Point add(Point a, int d)
	{
		return new Point(a.x + d, a.y + d);
	}

	public static Point inc(Point a)
	{
		return new Point(a.x + 1, a.y + 1);
	}

	public static Point sub(Point a, Dimension b)
	{
		return new Point(a.x - b.width, a.y - b.height);
	}

	public static Dimension sub(Point a, Point b)
	{
		return new Dimension(a.x - b.x, a.y - b.y);
	}

	public static Point sub(Point a, int amount)
	{
		return new Point(a.x - amount, a.y - amount);
	}

	public static Point dec(Point a)
	{
		return new Point(a.x - 1, a.y - 1);
	}

	public static Dimension add(Dimension a, Dimension b)
	{
		return new Dimension(a.width + b.width, a.height + b.height);
	}

	public static Dimension add(Dimension a, int w, int h)
	{
		return new Dimension(a.width + w, a.height + h);
	}

	public static Dimension add(Dimension a, int i)
	{
		return new Dimension(a.width + i, a.height + i);
	}

	public static Dimension inc(Dimension a)
	{
		return new Dimension(a.width + 1, a.height + 1);
	}

	public static Dimension mid(Dimension a, Dimension b)
	{
		return new Dimension((a.width + b.width) >> 1, (a.height + b.height) >> 1);
	}

	public static Dimension halfDist(Dimension a, Dimension b)
	{
		return new Dimension((b.width - a.width) >> 1, (b.height - a.height) >> 1);
	}

	public static Dimension sub(Dimension a, Dimension b)
	{
		return new Dimension(a.width - b.width, a.height - b.height);
	}

	public static Dimension sub(Dimension a, int w, int h)
	{
		return new Dimension(a.width - w, a.height - h);
	}

	public static Dimension sub(Dimension a, int i)
	{
		return new Dimension(a.width - i, a.height - i);
	}

	public static Dimension dec(Dimension a)
	{
		return new Dimension(a.width - 1, a.height - 1);
	}

	public static Dimension neg(Dimension a)
	{
		return new Dimension(-a.width, -a.height);
	}

	public static Dimension mul(Dimension a, Dimension b)
	{
		return new Dimension(a.width * b.width, a.height * b.height);
	}

	public static Dimension mul(Dimension a, int w, int h)
	{
		return new Dimension(a.width * w, a.height * h);
	}

	public static Dimension mul(Dimension a, int i)
	{
		return new Dimension(a.width * i, a.height * i);
	}

	public static Dimension shl(Dimension a, int i)
	{
		return new Dimension(a.width << i, a.height << i);
	}

	public static Rectangle mul(Rectangle rc, Dimension m)
	{
		return new Rectangle(rc.x * m.width, rc.y * m.height,
			rc.width * m.width, rc.height * m.height);
	}

	public static Rectangle mul(Rectangle rc, int m)
	{
		return new Rectangle(rc.x * m, rc.y * m, rc.width * m, rc.height * m);
	}

	public static Dimension div(Dimension a, Dimension b)
	{
		return new Dimension(a.width / b.width, a.height / b.height);
	}

	public static Dimension div(Dimension a, int w, int h)
	{
		return new Dimension(a.width / w, a.height / h);
	}

	public static Dimension div(Dimension a, int i)
	{
		return new Dimension(a.width / i, a.height / i);
	}

	public static Dimension shr(Dimension a, int i)
	{
		return new Dimension(a.width >> i, a.height >> i);
	}

	public static Dimension half(Dimension a)
	{
		return new Dimension(a.width >> 1, a.height >> 1);
	}

	public static long scalar(Dimension a, Dimension b)
	{
		long s1 = a.width * (long)b.width, s2 = a.height * (long)b.height;
		return s1 + s2;
	}

	public static long vector(Dimension a, Dimension b)
	{
		long v1 = a.width * (long)b.height, v2 = a.height * (long)b.width;
		return v1 - v2;
	}

	public static long abs2(Dimension a)
	{
		long xx = a.width * (long)a.width, yy = a.height * (long)a.height;
		return xx + yy;
	}

	public static double abs(Dimension a)
	{
		return Math.sqrt(a.width * (long)a.width + a.height * (long)a.height);
	}

	public static long dist2(Point a, Point b)
	{
		return abs2(sub(a, b));
	}

	public static double dist(Point a, Point b)
	{
		return abs(sub(a, b));
	}

	public static long dist2(Dimension a, Dimension b)
	{
		return abs2(sub(a, b));
	}

	public static double dist(Dimension a, Dimension b)
	{
		return abs(sub(a, b));
	}

	public static Rectangle sort(Point a, Point b)
	{
		return new Rectangle(
			Math.min(a.x, b.x), Math.min(a.y, b.y),
			Math.abs(b.x - a.x) + 1, Math.abs(b.y - a.y) + 1);
	}

	public static void move(Rectangle r, Dimension move)
	{
		r.x += move.width;
		r.y += move.height;
	}

	public static void move(Rectangle r, Point move)
	{
		r.x += move.x;
		r.y += move.y;
	}

	public static void move(Rectangle r, int x, int y)
	{
		r.x += x;
		r.y += y;
	}

	public static void move(Rectangle r, int amount)
	{
		r.x += amount;
		r.y += amount;
	}

	public static void inflate(Rectangle r, Dimension amount)
	{
		r.x -= amount.width;
		r.y -= amount.height;
		r.width += 2 * amount.width;
		r.height += 2 * amount.height;
	}

	public static void inflate(Rectangle r, int amount)
	{
		r.x -= amount;
		r.y -= amount;
		r.width += 2 * amount;
		r.height += 2 * amount;
	}

	public static Rectangle moved(Rectangle r, Dimension move)
	{
		return new Rectangle(r.x + move.width, r.y + move.height, r.width, r.height);
	}

	public static Rectangle moved(Rectangle r, Point move)
	{
		return new Rectangle(r.x + move.x, r.y + move.y, r.width, r.height);
	}

	public static Rectangle moved(Rectangle r, int x, int y)
	{
		return new Rectangle(r.x + x, r.y + y, r.width, r.height);
	}

	public static Rectangle moved(Rectangle r, int amount)
	{
		return new Rectangle(r.x + amount, r.y + amount, r.width, r.height);
	}

	public static Rectangle inflated(Rectangle r, Dimension amount)
	{
		return new Rectangle(r.x - amount.width, r.y - amount.height,
			r.width + 2 * amount.width, r.height + 2 * amount.height);
	}

	public static Rectangle inflated(Rectangle r, int amount)
	{
		return new Rectangle(r.x - amount, r.y - amount,
			r.width + 2 * amount, r.height + 2 * amount);
	}

	public static Point mid(Rectangle r)
	{
		return new Point(r.x + (r.width >> 1), r.y + (r.height >> 1));
	}

	public static Point rightBottom(Rectangle r)
	{
		return new Point(r.x + r.width, r.y + r.height);
	}

	public static Rectangle center(Dimension dim, Rectangle inside)
	{
		return new Rectangle(add(inside.getLocation(),
			half(sub(inside.getSize(), dim))), dim);
	}

	public static Rectangle center(Rectangle r, Rectangle inside)
	{
		return new Rectangle(add(inside.getLocation(),
			half(sub(inside.getSize(), r.getSize()))), r.getSize());
	}

	public static Rectangle center(Rectangle r, Dimension size)
	{
		return new Rectangle(asPoint(half(sub(size, r.getSize()))), r.getSize());
	}

	public static Rectangle center(Rectangle r, Point around)
	{
		return new Rectangle(sub(around, half(r.getSize())), r.getSize());
	}

	public static boolean contains(Rectangle a, Rectangle b)
	{
		return b.x >= a.x && b.y >= a.y
		&& b.x + b.width <= a.x + a.width && b.y + b.height <= a.y + a.height;
	}

	public static Rectangle sortRect(Point a, Point b)
	{
		if(a == null || b == null)
			return null;
		return new Rectangle(Util.min(a.x, b.x), Util.min(a.y, b.y),
			Math.abs(b.x - a.x) + 1, Math.abs(b.y - a.y) + 1);
	}

	public static double log10(double x)
	{
		return Math.log(x) / LOG10;
	}

	//////////////////////////////////////////////////////////////////////

	public static String asString(double x) { return asString(x, 3, true); }
	public static String asString(double x, int places) { return asString(x, places, true); }
	public static String asString(double x, int places, boolean nls)
	{
		StringBuffer result = new StringBuffer();
		if(x < 0)
		{
			result.append('-');
			x = -x;
		}
		double low = Math.pow(10, places);
		if(x * low < 0.5)
			return "0";
		int exp = 0;
		if(x >= 1e10)
		{ // too big, use exponential notation
			exp = (int)Math.floor(log10(x));
			x /= Math.pow(10, exp);
			if(x >= 10) { x /= 10; exp++; }
			else if(x < 1) { x *= 10; exp--; }
		}
		String raw = String.valueOf(Math.round(x * low));
		int end = raw.length(), comma = end - places;
		for(int rcount = Math.max(comma, 0); end > rcount && raw.charAt(end - 1) == '0'; end--)
			;
		if(comma <= 0)
			result.append('0');
		else if(!nls)
			result.append(raw.substring(0, comma));
		else
		{
			int first = (comma + 2) % 3 + 1;
			if(first >= comma)
				result.append(raw.substring(0, comma));
			else
			{
				for(result.append(raw.substring(0, first)); first < comma; first += 3)
				{
					result.append(' ');
					result.append(raw.substring(first, first + 3));
				}
			}
		}
		if(end > comma)
		{
			result.append(nls ? ',' : '.');
			for(; comma < 0; comma++)
				result.append('0');
			result.append(raw.substring(comma, end));
		}

		if(exp != 0)
		{
			result.append('e');
			result.append(String.valueOf(exp));
		}

		return result.toString();
	}

	public static String asDegree(double deg, int digits)
	{
		StringBuffer result = new StringBuffer();
		if(deg < 0) {
			result.append('-');
			deg = -deg;
		}
		double lo = Math.floor(deg);
		result.append(asString(lo, 0));
		result.append("\u00B0");
		lo = Math.floor(deg = (deg - lo) * 60);
		result.append(asString(lo, 0));
		result.append('\'');
		deg = (deg - lo) * 60;
		result.append(asString(deg, digits));
		result.append('\"');
		return result.toString();
	}

	private static final String hex = "0123456789ABCDEF";

	public static String binHexEncode(byte c)
	{
		char temp[] = new char[2];
		temp[0] = hex.charAt((c >> 4) & 15);
		temp[1] = hex.charAt(c & 15);
		return new String(temp);
	}

	public static String urlEncode(String string)
	{
		if(string == null || string.length() == 0)
			return "";
		byte src[] = null;
		try { src = string.getBytes(codepage); }
		catch(UnsupportedEncodingException e) {}
		StringBuffer buffer = new StringBuffer();
		for(int i = 0; i < src.length; i++)
		{
			byte c = src[i];
			if(c >= '0' && c <= '9' || c >= 'A' && c <= 'Z'
			|| c >= 'a' && c <= 'z')
				buffer.append((char)c);
			else if(c == ' ')
				buffer.append('+');
			else
			{
				buffer.append('%');
				buffer.append(hex.charAt((c >> 4) & 15));
				buffer.append(hex.charAt((c >> 0) & 15));
			}
		}
		return buffer.toString();
	}

	public static String urlDecode(String string)
	{
		if(string == null || string.length() == 0)
			return "";
		StringBuffer out = new StringBuffer();
		for(int i = 0; i < string.length(); i++) {
			char c = string.charAt(i);
			if(c == '+')
				out.append(' ');
			else if(c == '%' && i + 3 <= string.length()) {
				char c1 = string.charAt(i + 1);
				int d1 = ctoi(string.charAt(i + 1));
				int d2 = ctoi(string.charAt(i + 2));
				if((c1 == 'u' || c1 == 'U') && i + 6 <= string.length()) {
					int d3 = ctoi(string.charAt(i + 3));
					int d4 = ctoi(string.charAt(i + 4));
					int d5 = ctoi(string.charAt(i + 5));
					if(d2 < 16 && d3 < 16 && d4 < 16 && d5 < 16) {
						out.append((char)((d2 << 12) | (d3 << 8) | (d4 << 4) | d5));
						i += 5;
					}
					else
						out.append('%');
				}
				else if(d1 < 16 && d2 < 16) {
					out.append((char)((d1 << 4) | d2));
					i += 2;
				}
				else
					out.append('%');
			}
			else
				out.append(c);
		}
		return out.toString();
	}

	public static String patchEoln(String s)
	{
		LLOG("patchEoln " + s);
		if(s == null || s.length() == 0)
			return s;
		int len = s.length();
		char bytes[] = new char[len];
		for(int i = 0; i < len; i++)
			bytes[i] = (s.charAt(i) == (char)0x7F ? (char)0x0A : s.charAt(i));
		String out = new String(bytes);
		LLOG("//patchEoln -> " + out);
		return out;
	}

#ifdef DOLOG
	public synchronized static void print_stack()
	{
		(new Throwable()).printStackTrace(log());
	}
#endif

#ifdef DOLOG
	public synchronized static PrintWriter log()
	{
		if(log_file == null)
			try
			{
				log_file = new PrintWriter(new FileWriter("e:\\temp\\java.log"), true);
				log_file.println("Current time: " + new java.util.Date());
			}
			catch(IOException e)
			{
				return new PrintWriter(System.out);
			}
		return log_file;
	}
/*
#else
	public synchronized static PrintStream log()
	{
		return System.out;
	}
*/
#endif

	//////////////////////////////////////////////////////////////////////

	public synchronized static void print(String s)
	{
#ifdef DOLOG
		log().print(s);
#endif
		System.out.print(s);
	}

	//////////////////////////////////////////////////////////////////////

	public synchronized static void println(String s)
	{
		if(start_ticks == 0)
			start_ticks = System.currentTimeMillis();
#ifdef DOLOG
		log().println("[" + (System.currentTimeMillis() - start_ticks) + " @ "
			+ Thread.currentThread().getName() + "] " + s);
#endif
		System.out.println(s);
	}

	//////////////////////////////////////////////////////////////////////

	public static byte[] copy(byte src[], int length, int reserve)
	{
		byte out[] = new byte[reserve];
		for(int i = 0; i < length; i++)
			out[i] = src[i];
		return out;
	}

	//////////////////////////////////////////////////////////////////////

	public static byte[] ascii85Decode(String string, int offset)
	{
		byte out[] = new byte[max((string.length() - offset) * 4 / 5 + 10, 100)], buffer[] = new byte[5];
		int full = 0;
		int pos = 0;
		while(offset < string.length())
		{
			if(full >= out.length - 8)
				out = copy(out, full, 2 * out.length);
			char c = string.charAt(offset++);
			if(c >= 33 && c < 33 + 85)
			{
				buffer[pos++] = (byte)(c - 33);
				if(pos >= 5)
				{
					int v = buffer[4] + 85 * (buffer[3] + 85 * (buffer[2] + 85 * (buffer[1] + 85 * buffer[0])));
					out[full + 0] = (byte)(v >> 24);
					out[full + 1] = (byte)(v >> 16);
					out[full + 2] = (byte)(v >> 8);
					out[full + 3] = (byte)v;
					full += 4;
					pos = 0;
				}
			}
			else if(c == 'z')
			{
				out[full + 0] = out[full + 1] = out[full + 2] = out[full + 3] = 0;
				full += 4;
				pos = 0;
			}
		}
		switch(pos)
		{
		case 2:
			{
				int v = 84 + 84 * 85 + 84 * 85 * 85 + 85 * 85 * 85 * (buffer[1] + 85 * buffer[0]);
				out[full++] = (byte)(v >> 24);
			}
			break;

		case 3:
			{
				int v = 84 + 84 * 85 + 85 * 85 * (buffer[2] + 85 * (buffer[1] + 85 * buffer[0]));
				out[full++] = (byte)(v >> 24);
				out[full++] = (byte)(v >> 16);
			}
			break;

		case 4:
			{
				int v = 84 + 85 * (buffer[3] + 85 * (buffer[2] + 85 * (buffer[1] + 85 * buffer[0])));
				out[full++] = (byte)(v >> 24);
				out[full++] = (byte)(v >> 16);
				out[full++] = (byte)(v >> 8);
			}
			break;
		}
		return copy(out, full, full);
	}

	public static synchronized long getID() { return ++uuid; }

	public static final int INT_NULL = -0x7FFFFFFF;
	public static final double DBL_NULL = -1e200;

	public static final int LANG_EN = 0;
	public static final int LANG_CS = 1;
	public static final int LANG_DE = 2;

	public static String codepage = "";
//	public static boolean quitting = false;
	public static String language = "EN-US";
	public static int langcode = LANG_EN;
	public static final double LOG10 = Math.log(10);
	public static PrintWriter log_file;
	public static long uuid = System.currentTimeMillis();
	public static long start_ticks = 0;
//  	public static ThreadGroup app_group = new ThreadGroup("app")
//	{ public void checkAccess() {} }
}
