///////////////////////////////////////////////////////////////////
// FPoint: floating-point point.

#include <utility/misc.jinc>

package utility;

import java.awt.*;

public final class FPoint implements Cloneable
{
	public double x = 0, y = 0;

	public FPoint() {}
	public FPoint(double x, double y) { this.x = x; this.y = y; }
	public FPoint(Point pt)           { this.x = pt.x; this.y = pt.y; }
	public FPoint(Dimension dim)      { this.x = dim.width; this.y = dim.height; }

	public Object clone() { try { return super.clone(); } catch(CloneNotSupportedException e) { return null; } }

	public final static FPoint Polar(double a)
	{ return new FPoint(Math.cos(a), Math.sin(a)); }

	public final Point     asPoint() { return new Point(Util.fround(x), Util.fround(y)); }
	public final Dimension asSize()  { return new Dimension(Util.fround(x), Util.fround(y)); }

	public final String    toString()
	{ return "[" + String.valueOf(x) + ", " + String.valueOf(y) + "]"; }

	public final boolean   equals(FPoint p) { return x == p.x && y == p.y; }

    public final void      add(FPoint p)             { x += p.x; y += p.y; }
    public final void      add(double dx, double dy) { x += dx;  y += dy; }
    public final void      add(double d)             { x += d;   y += d; }

    public final void      sub(FPoint p)             { x -= p.x; y -= p.y; }
    public final void      sub(double dx, double dy) { x -= dx;  y -= dy; }
    public final void      sub(double d)             { x -= d;   y -= d; }

    public final void      mul(FPoint p)             { x *= p.x; y *= p.y; }
    public final void      mul(double dx, double dy) { x *= dx;  y *= dy; }
    public final void      mul(double d)             { x *= d;   y *= d; }

    public final void      div(FPoint p)             { x /= p.x; y /= p.y; }
    public final void      div(double dx, double dy) { x /= dx;  y /= dy; }
    public final void      div(double d)             { x /= d;   y /= d; }

    public final FPoint    plus(FPoint p)              { return new FPoint(x + p.x, y + p.y); }
    public final FPoint    plus(double dx, double dy)  { return new FPoint(x + dx,  y + dy); }
    public final FPoint    plus(double d)              { return new FPoint(x + d,   y + d); }

    public final FPoint    minus(FPoint p)             { return new FPoint(x - p.x, y - p.y); }
    public final FPoint    minus(double dx, double dy) { return new FPoint(x - dx,  y - dy); }
    public final FPoint    minus(double d)             { return new FPoint(x - d,   y - d); }

    public final FPoint    times(FPoint p)             { return new FPoint(x * p.x, y * p.y); }
    public final FPoint    times(double dx, double dy) { return new FPoint(x * dx,  y * dy); }
    public final FPoint    times(double d)             { return new FPoint(x * d,   y * d); }

    public final FPoint    over(FPoint p)              { return new FPoint(x / p.x, y / p.y); }
    public final FPoint    over(double dx, double dy)  { return new FPoint(x / dx,  y / dy); }
    public final FPoint    over(double d)              { return new FPoint(x / d,   y / d); }

    public final FPoint    min(FPoint p)              { return new FPoint(x < p.x ? x : p.x, y < p.y ? y : p.y); }
    public final FPoint    min(double dx, double dy)  { return new FPoint(x < dx  ? x : dx,  y < dy  ? y : dy); }
    public final FPoint    min(double d)              { return new FPoint(x < d   ? x : d,   y < d   ? y : d); }
	public final double    min()                      { return Math.min(x, y); }
	public final double    absMin()                   { return Math.min(Math.abs(x), Math.abs(y)); }

	public final FPoint    max(FPoint p)              { return new FPoint(x > p.x ? x : p.x, y > p.y ? y : p.y); }
	public final FPoint    max(double dx, double dy)  { return new FPoint(x > dx  ? x : dx,  y > dy  ? y : dy); }
	public final FPoint    max(double d)              { return new FPoint(x > d   ? x : d,   y > d   ? y : d); }
	public final double    max()                      { return Math.max(x, y); }
	public final double    absMax()                   { return Math.max(Math.abs(x), Math.abs(y)); }

	public final FPoint    mid(FPoint p)              { return new FPoint((x + p.x) / 2, (y + p.y) / 2); }
    public final double    abs2()                     { return x * x + y * y; }
    public final double    abs()                      { return Math.sqrt(abs2()); }
    public final double    dist2(FPoint p)            { return minus(p).abs2(); }
	public final double    dist(FPoint p)             { return minus(p).abs(); }
	public final double    qdist(FPoint p)            { return Math.max(Math.abs(p.x - x), Math.abs(p.y - y)); }
	public final double    rdist(FPoint p)            { return Math.abs(p.x - x) + Math.abs(p.y - y); }

	public final FPoint    left()                     { return new FPoint(-y, x); }
	public final FPoint    minus()                    { return new FPoint(-x, -y); }
	public final FPoint    right()                    { return new FPoint(y, -x); }

	public final double    vector(FPoint p)           { return x * p.y - y * p.x; }
	public final double    scalar(FPoint p)           { return x * p.x + y * p.y; }

	public final double    dist(FPoint a, FPoint b)
	{
		FPoint ab = b.minus(a), ax = this.minus(a);
		double ab2 = ab.abs2();
		double abx = ab.scalar(ax);
		if(abx <= 0 || ab2 == 0)
			return this.dist(a);
		if(abx >= ab2)
			return this.dist(b);
		double asb = ax.vector(ab);
		return Math.sqrt(asb * asb / ab2);
	}

	public final FPoint    rotated(double rad)
	{
		double s = Math.sin(rad), c = Math.cos(rad);
		return new FPoint(x * c - y * s, x * s + y * c);
	}

	public final FPoint    rotated(FPoint centre, double rad)
	{
		return minus(centre).rotated(rad).plus(centre);
	}

	public final FPoint    scale(FPoint m, FPoint d)  { return new FPoint(x * m.x / d.x, x * m.y / d.y); }
	public final FPoint    scale(double m, double d)  { return new FPoint(x * m / d, x * m / d); }

	public final FPoint    unit()                     { return over(abs()); }
	public final FPoint    length(double l)           { return times(l / abs()); }
	public final double    bearing()
	{
		if(y == 0)
			return (x >= 0 ? 0 : Math.PI);
		if(x == 0)
			return (y >= 0 ? Math.PI * 0.5 : Math.PI * 1.5);
		double b = Math.atan2(y, x);
		if(b < 0)
			b += 2 * Math.PI;
		return b;
	}
}
