///////////////////////////////////////////////////////////////////
// FPoint: floating-point rectangle.

#include <utility/misc.jinc>

package utility;

import java.awt.*;

public class FRect implements Cloneable
{
	public double left = 0, top = 0, right = 0, bottom = 0;

	public FRect() {}
	public FRect(double left, double top, double right, double bottom)
	{ this.left = left; this.top = top; this.right = right; this.bottom = bottom; }
	public FRect(Rectangle rc) { this(rc.x, rc.y, rc.x + rc.width, rc.y + rc.height); }

	public FRect(Point a)  { left = right = a.x; top = bottom = a.y; }
	public FRect(FPoint a) { left = right = a.x; top = bottom = a.y; }

	public FRect(FPoint a, FPoint b)
	{
		left   = (a.x < b.x ? a.x : b.x);
		top    = (a.y < b.y ? a.y : b.y);
		right  = (a.x > b.x ? a.x : b.x);
		bottom = (a.y > b.y ? a.y : b.y);
	}

	public FRect(Point a, Point b) { this(new FPoint(a), new FPoint(b)); }

	public Object clone() { try { return super.clone(); } catch(CloneNotSupportedException e) { return null; } }

	public final static FRect Empty()           { return new FRect(0, 0, -1, -1); }
	public final static FRect Size(FPoint p)    { return new FRect(0, 0, p.x, p.y); }
	public final static FRect Size(Dimension p) { return new FRect(0, 0, p.width, p.height); }

	public final static FRect Sort(FPoint p, FPoint q)
	{
		return new FRect(Math.min(p.x, q.x), Math.min(p.y, q.y),
			Math.max(p.x, q.x), Math.max(p.y, q.y));
	}

	public final static FRect Sort(Point p, Point q)
	{
		return new FRect(Math.min(p.x, q.x), Math.min(p.y, q.y),
			Math.max(p.x, q.x), Math.max(p.y, q.y));
	}

	public final boolean isEmpty() { return right < left || bottom < top; }

	public final Rectangle asRect()
	{ return new Rectangle(leftTop().asPoint(), getSize().asSize()); }

	public final String    toString()
	{
		return "{" + leftTop().toString() + " - " + rightBottom().toString()
		+ " = " + getSize().toString() + "}";
	}

	public final FRect sortHorz()               { if(left > right) { double t = left; left = right; right = t; } return this; }
	public final FRect sortVert()               { if(top > bottom) { double t = top; top = bottom; bottom = t; } return this; }
	public final FRect sort()                   { return sortHorz().sortVert(); }

	public final void add(FPoint pt)            { left += pt.x; top += pt.y; right += pt.x; bottom += pt.y; }
	public final void add(double dx, double dy) { left += dx;   top += dy;   right += dx;   bottom += dy; }
	public final void add(double d)             { left += d;    top += d;    right += d;    bottom += d; }

	public final void sub(FPoint pt)            { left -= pt.x; top -= pt.y; right -= pt.x; bottom -= pt.y; }
	public final void sub(double dx, double dy) { left -= dx;   top -= dy;   right -= dx;   bottom -= dy; }
	public final void sub(double d)             { left -= d;    top -= d;    right -= d;    bottom -= d; }

	public final void mul(FPoint pt)            { left *= pt.x; top *= pt.y; right *= pt.x; bottom *= pt.y; }
	public final void mul(double dx, double dy) { left *= dx;   top *= dy;   right *= dx;   bottom *= dy; }
	public final void mul(double d)             { left *= d;    top *= d;    right *= d;    bottom *= d; }

	public final void div(FPoint pt)            { left /= pt.x; top /= pt.y; right /= pt.x; bottom /= pt.y; }
	public final void div(double dx, double dy) { left /= dx;   top /= dy;   right /= dx;   bottom /= dy; }
	public final void div(double d)             { left /= d;    top /= d;    right /= d;    bottom /= d; }

	public final void inflate(FPoint pt)            { left -= pt.x; top -= pt.y; right += pt.x; bottom += pt.y; }
	public final void inflate(double dx, double dy) { left -= dx;   top -= dy;   right += dx;   bottom += dy; }
	public final void inflate(double d)             { left -= d;    top -= d;    right += d;    bottom += d; }

	public final FRect plus(FPoint pt)            { return new FRect(left + pt.x, top + pt.y, right + pt.x, bottom + pt.y); }
	public final FRect plus(double dx, double dy) { return new FRect(left + dx,   top + dy,   right + dx,   bottom + dy); }
	public final FRect plus(double d)             { return new FRect(left + d,    top + d,    right + d,    bottom + d); }

	public final FRect minus(FPoint pt)            { return new FRect(left - pt.x, top - pt.y, right - pt.x, bottom - pt.y); }
	public final FRect minus(double dx, double dy) { return new FRect(left - dx,   top - dy,   right - dx,   bottom - dy); }
	public final FRect minus(double d)             { return new FRect(left - d,    top - d,    right - d,    bottom - d); }

    public final FRect times(FPoint pt)            { return new FRect(left * pt.x, top * pt.y, right * pt.x, bottom * pt.y); }
    public final FRect times(double dx, double dy) { return new FRect(left * dx,   top * dy,   right * dx,   bottom * dy); }
    public final FRect times(double d)             { return new FRect(left * d,    top * d,    right * d,    bottom * d); }

    public final FRect over(FPoint pt)            { return new FRect(left / pt.x, top / pt.y, right / pt.x, bottom / pt.y); }
    public final FRect over(double dx, double dy) { return new FRect(left / dx,   top / dy,   right / dx,   bottom / dy); }
    public final FRect over(double d)             { return new FRect(left / d,    top / d,    right / d,    bottom / d); }

    public final FRect inflated(FPoint pt)            { return new FRect(left - pt.x, top - pt.y, right + pt.x, bottom + pt.y); }
    public final FRect inflated(double dx, double dy) { return new FRect(left - dx,   top - dy,   right + dx,   bottom + dy); }
    public final FRect inflated(double d)             { return new FRect(left - d,    top - d,    right + d,    bottom + d); }

	public final FPoint leftTop()                { return new FPoint(left, top); }
	public final FPoint centerTop()              { return new FPoint((left + right) / 2, top); }
	public final FPoint rightTop()               { return new FPoint(right, top); }
	public final FPoint leftCenter()             { return new FPoint(left, (top + bottom) / 2); }
	public final FPoint center()                 { return new FPoint((left + right) / 2, (top + bottom) / 2); }
	public final FPoint rightCenter()            { return new FPoint(right, (top + bottom) / 2); }
	public final FPoint leftBottom()             { return new FPoint(left, bottom); }
	public final FPoint centerBottom()           { return new FPoint((left + right) / 2, bottom); }
	public final FPoint rightBottom()            { return new FPoint(right, bottom); }

	public final FPoint getSize()                { return new FPoint(right - left, bottom - top); }
	public final double getWidth()               { return right - left; }
	public final double getHeight()              { return bottom - top; }

	public final FRect  scale(FPoint m, FPoint d) { return times(m.over(d)); }
	public final FRect  scale(double m, double d) { return times(m / d); }

	public final boolean contains(FPoint pt, double tol)
	{ return pt.x + tol >= left && pt.x - tol <= right && pt.y + tol >= top && pt.y - tol <= bottom; }
	public final boolean contains(FPoint pt)      { return contains(pt, 0); }

	public final boolean contains(FRect rc)
	{
		return rc != null && !rc.isEmpty()
			&& rc.left >= left && rc.right <= right
			&& rc.top >= top && rc.bottom <= bottom;
	}

    public final boolean intersects(FRect rc)
    { return left <= rc.right && top <= rc.bottom && right >= rc.left && bottom >= rc.top; }

	public final FRect union(FPoint pt)
	{
		if(pt == null)
			return this;
		return new FRect(Math.min(left, pt.x), Math.min(top, pt.y),
			Math.max(right, pt.y), Math.max(bottom, pt.y));
	}

	public final FRect union(FRect rc)
	{
		if(rc == null || rc.isEmpty())
			return this;
		return new FRect(Math.min(left, rc.left), Math.min(top, rc.top),
			Math.max(right, rc.right), Math.max(bottom, rc.bottom));
	}

	public final FRect bind(FPoint size) { return bind(Size(size)); }
	public final FRect bind(FRect out)
	{
		if(!intersects(out))
			return FRect.Empty();
		return new FRect(
			Math.max(left, out.left),
			Math.max(top, out.top),
			Math.min(right, out.right),
			Math.min(bottom, out.bottom));
	}
	public final FRect intersection(FRect out) { return bind(out); }

	public final FRect bindMove(FPoint size) { return bindMove(Size(size)); }
	public final FRect bindMove(FRect rc)
	{
		return plus(
			(left < rc.left ? rc.left - left : right  > rc.right  ? rc.right  - right  : 0),
			(top  < rc.top  ? rc.top  - top  : bottom > rc.bottom ? rc.bottom - bottom : 0));
	}

	public final double getDirDistance(FPoint pt)
	{
		double l = left - pt.x, r = pt.x - right, t = top - pt.y, b = pt.y - bottom;
		if(l < 0 && r < 0)
			return Math.abs(t) < Math.abs(b) ? t : b;
		if(t < 0 && b < 0)
			return Math.abs(l) < Math.abs(r) ? l : r;
		return Util.hypot(Math.max(l, t), Math.max(t, b));
	}
}
