///////////////////////////////////////////////////////////////////
// Slider: scrollbar body & thumb.

#include <utility/misc.jinc>

#define LLOG(x) // RLOG(x)

package utility;

import java.awt.*;
import java.awt.event.*;
import java.util.Vector;

public class Slider extends Component
{
	public Slider()                   { this(false); }
	public Slider(boolean horz)
	{
		this.horizontal = horz;
		setBackground(SystemColor.scrollbar);
		setForeground(SystemColor.control);
		adapter = new ActionAdapter(this)
		{
			protected void doPush()
			{
				Point pos = getPosition();
//                LLOG("thumb = " + thumb.toString());
//                LLOG("click = " + pos.toString());
				thumb_pos = horizontal
					? (pos.x < thumb.x ? -2 : pos.x >= thumb.x + thumb.width ? -1 : pos.x - thumb.x)
					: (pos.y < thumb.y ? -2 : pos.y >= thumb.y + thumb.height ? -1 : pos.y - thumb.y);
				if(thumb_pos < 0)
					doRepeat();
			}

			protected void doRelease()
			{
//            	LLOG("Slider.doRelease");
				setValue(pos, false);
			}

			protected void doRepeat()
			{
				switch(thumb_pos)
				{
				case -2: doPrevBlock(); break;
				case -1: doNextBlock(); break;
				}
			}
		};
		adapter.setPushAction(true);
		adapter.setRepeat();
		addMouseMotionListener(new MouseMotionAdapter()
		{
			public void mouseDragged(MouseEvent e)
			{
//            	LLOG("drag " + e.getPoint().toString());
				if(thumb_pos >= 0)
					setValue(distanceToValue((horizontal ? e.getPoint().x : e.getPoint().y) - thumb_pos), true);
			}
		});
	}

	public boolean isAutoSync() { return auto_sync; }
	public void setAutoSync(boolean _as) { auto_sync = _as; }

	public boolean isHorizontal() { return horizontal; }
	public void setHorizontal(boolean horizontal)
	{
		if(horizontal != this.horizontal)
		{
			this.horizontal = horizontal;
			invalidate();
			validate();
		}
	}

	public void set(int _min, int _max, int _page, int _val, boolean user)
	{
		LLOG("Slider.set(min " + _min + ", max " + _max + ", page " + _page + ", val " + _val + ", user " + user + ")");
		boolean redraw = false;
		if(_page != page || _min != min || _max != max || !getSize().equals(last_dim))
		{
			LLOG((horizontal ? "horz-set" : "vert-set") + ": min = " + _min + ", max = " + _max + ", page = " + _page + ", val = " + _val
				+ ", parent " + getParent());
			page = _page;
			min = _min;
			max = _max;
			calcPage();
			redraw = true;
		}
		_val = Util.minmax(_val, min, max - in_page);
		if(_val != pos || redraw)
		{
			LLOG("slider set: min = " + _min + ", max = " + _max + ", page = " + _page + ", val = " + _val
				+ ", parent " + getParent());
			pos = _val;
			calcThumb();
			redraw = true;
		}
		if(redraw && isShowing())
		{
			repaint();
			if(auto_sync)
			{ // redraw on the fly
				LLOG("auto-sync");
				Graphics g = getGraphics();
				paint(g);
				g.dispose();
			}
//				repaint();
		}
//		LLOG("Slider -> notifying listeners");
		for(int i = 0; i < listeners.size(); i++)
			((ScrollAdapter)listeners.elementAt(i)).doScroll(user);
//		LLOG("//Slider -> notifying listeners");
		LLOG("//Slider.set");
	}

	public int  getValue()                             { return pos; }
	public void setValue(int _value, boolean user)     { set(min, max, page, _value, user); }
	public void setValue(int _value)                   { set(min, max, page, _value, false); }

	public int  getVisibleAmount()
	{
		LLOG("Slider.getVisible = " + page);
		return page;
	}
	public void setAutoAmount()                        { setVisibleAmount(-1); }
	public void setVisibleAmount(int _page)            { set(min, max, _page, pos, false); }

	public void setRange(int _min, int _max)           { set(_min, _max, page, pos, false); }

	public int  getMinimum()                           { return min; }
	public void setMinimum(int _min)                   { set(_min, max, page, pos, false); }

	public int  getMaximum()                           { return max; }
	public void setMaximum(int _max)                   { set(min, _max, page, pos, false); }

	public int getOrientation()
	{ return horizontal ? Adjustable.HORIZONTAL : Adjustable.VERTICAL; }

	public int getUnitIncrement() { return unit_shift; }
	public void setUnitIncrement(int _shift) { unit_shift = _shift; }

	public int getBlockIncrement() { return block_shift; }
	public void setBlockIncrement(int _shift) { block_shift = _shift; }

	public int getRawBlockIncrement() { return block_shift > 0 ? block_shift : in_page - (in_page >> 3); }
	public int getRawUnitIncrement()  { return unit_shift > 0 ? unit_shift : in_page >> 4; }

	private int findScrollListener(ScrollAdapter a)
	{
		for(int i = 0; i < listeners.size(); i++)
			if(((ScrollAdapter)listeners.elementAt(i)).equals(a))
				return i;
		return -1;
	}

	public void addScrollListener(ScrollAdapter a)
	{
		if(findScrollListener(a) < 0)
			listeners.addElement(a);
	}

	public void removeScrollListener(ScrollAdapter a)
	{
		int f = findScrollListener(a);
		if(f >= 0)
			listeners.removeElementAt(f);
	}

	public Dimension getPreferredSize()
	{
		if(horizontal)
			return new Dimension(100, 18);
		else
			return new Dimension(18, 100);
	}

	public int distanceToValue(int distance)
	{
		int total = max - in_page - min;
		if(total <= 0)
			return min;
		Dimension dim = getSize();
		int over = (horizontal ? dim.width : dim.height) - raw_page;
		if(over <= 0)
			return min;
		return min + Util.scale(distance, total, over);
	}

	public int clientToValue(Point pos)
	{
		return distanceToValue(horizontal ? pos.x : pos.y);
	}

	public int valueToDistance(int amount)
	{
		int total = max - in_page - min;
		Dimension dim = getSize();
		if(total <= 0)
			return 0;
		int range = (horizontal ? dim.width : dim.height) - raw_page;
		return Util.scale(amount - min, range, total);
	}

	public Point valueToClient(int amount)
	{
		int v = valueToDistance(amount);
		return horizontal ? new Point(v, 0) : new Point(0, v);
	}

	public void doBegin()     { setValue(min, true); }
	public void doPrevBlock() { setValue(getValue() - getRawBlockIncrement(), true); }
	public void doPrevUnit()  { setValue(getValue() - getRawUnitIncrement(), true); }
	public void doNextUnit()  { setValue(getValue() + getRawUnitIncrement(), true); }
	public void doNextBlock() { setValue(getValue() + getRawBlockIncrement(), true); }
	public void doEnd()       { setValue(max - in_page, true); }

	public void doKey(KeyEvent e) { doKey(e, false); }
	public void doKey(KeyEvent e, boolean twin_keys)
	{
		if(horizontal || twin_keys)
			switch(e.getKeyCode())
			{
			case KeyEvent.VK_LEFT:  if(e.isControlDown()) doPrevBlock(); else doPrevUnit(); break;
			case KeyEvent.VK_RIGHT: if(e.isControlDown()) doNextBlock(); else doNextUnit(); break;
			case KeyEvent.VK_HOME:  doBegin(); break;
			case KeyEvent.VK_END:   doEnd(); break;
			}
		if(!horizontal || twin_keys)
			switch(e.getKeyCode())
			{
			case KeyEvent.VK_UP:        doPrevUnit(); break;
			case KeyEvent.VK_DOWN:      doNextUnit(); break;
			case KeyEvent.VK_PAGE_UP:   if(e.isControlDown()) doBegin(); else doPrevBlock(); break;
			case KeyEvent.VK_PAGE_DOWN: if(e.isControlDown()) doEnd();   else doNextBlock(); break;
			}
	}

	public void paint(Graphics g)
	{
		Dimension dim = getSize();
		if(last_dim == null || !dim.equals(last_dim))
			calcPage();
		Point pt = valueToClient(pos);
		LLOG("Slider.paint, dim = " + dim + ", pt = " + pt + ", parent " + getParent());
		g.setColor(getBackground());
		if(horizontal)
		{
			g.fillRect(0, 0, pt.x, dim.height);
			g.fillRect(pt.x + raw_page, 0, dim.width - pt.x - raw_page, dim.height);
			GUtil.drawRect(g, pt.x, pt.y, raw_page, dim.height, 1, GUtil.whiteGray, Color.black);
			GUtil.drawRect(g, pt.x + 1, pt.y + 1, raw_page - 2, dim.height - 2, 1, Color.white, Color.gray);
			g.setColor(getForeground());
			g.fillRect(pt.x + 2, pt.y + 2, raw_page - 4, dim.height - 4);
		}
		else
		{
			g.fillRect(0, 0, dim.width, pt.y);
			g.fillRect(0, pt.y + raw_page, dim.width, dim.height - pt.x - raw_page);
			GUtil.drawRect(g, pt.x, pt.y, dim.width, raw_page, 1, GUtil.whiteGray, Color.black);
			GUtil.drawRect(g, pt.x + 1, pt.y + 1, dim.width - 2, raw_page - 2, 1, Color.white, Color.gray);
			g.setColor(getForeground());
			g.fillRect(pt.x + 2, pt.y + 2, dim.width - 4, raw_page - 4);
		}
	}

//	public void update(Graphics g) { paint(g); }

	public void doLayout()
	{
		LLOG("Slider.doLayout");
		super.doLayout();
		calcPage();
	}

	private void calcPage()
	{
		last_dim = getSize();
		LLOG("Slider.calcPage -> dim = " + last_dim);
		int avail = (horizontal ? last_dim.width : last_dim.height);
		LLOG("Slider.calcPage -> avail = " + avail);
		if((in_page = page) < 0)
			page = avail;
		if(min >= max)
			raw_page = avail;
		else if(page > 0)
			raw_page = Util.minmax(Util.scale(page, avail, max - min), 6, avail);
		else
			raw_page = Math.min(avail >> 2, 16);
		LLOG("Slider.calcPage -> raw_page = " + raw_page + ", page = " + page);
		calcThumb();
/*
		if(horizontal)
		{
			LLOG("Slider.doLayout");
			LLOG("Dim:      " + dim);
			LLOG("avail:    " + avail);
			LLOG("page:     " + page);
			LLOG("in page:  " + in_page);
			LLOG("raw page: " + raw_page);
			LLOG("block:    " + block_shift);
		}
//*/
	}

	private void calcThumb()
	{
		Dimension dim = getSize();
		thumb = new Rectangle(valueToClient(pos), horizontal
			? new Dimension(raw_page, dim.height) : new Dimension(dim.width, raw_page));
		LLOG("Slider.calcThumb -> thumb = " + thumb);
	}

	private int block_shift, unit_shift;
	private int pos, min, max, page, in_page, raw_page;
	private Dimension last_dim = new Dimension(0, 0);
	private boolean horizontal;
	private Rectangle hotzone, thumb;
	private Vector listeners = new Vector();
	private int thumb_pos;
	private ActionAdapter adapter;
	private boolean auto_sync = true;
}
