#include "TestPtrMT.h"

namespace fileobj {

const int cycles = 5;
const int internalCycles = 3;

void Out(const char* text)
{
	INTERLOCKED Cout() << text << EOL;
}

File::File(const char* fname)
{
	Out(String().Cat() << "Creating file: " << fname);
}

File::~File()
{
	Out("Destroying file");
}

void File::Write(const char* text)
{
	Out(String().Cat() << "Writing text to file: " << text);
}

FileObject::FileObject() :
	impl(new Impl)
{
}

FileObject::FileObject(const shared_ptr<Impl>& impl_) :
	impl(impl_)
{
}

bool FileObject::IsOpened() const
{
	Mutex::Lock lock(impl->mutex);
	return impl->file;
}

void FileObject::Close()
{
	Mutex::Lock lock(impl->mutex);
	impl->file.reset();
}

void FileObject::Open(const char* fname)
{
	Mutex::Lock lock(impl->mutex);
	impl->file.reset(new File(fname));
}

void FileObject::Write(const char* text)
{
	Mutex::Lock lock(impl->mutex);
	if (!impl->file)
		throw Exc(String().Cat() << "Writing failed: file is not opened, text: " << text);
	impl->file->Write(text);
}

FileObject::Ref::Ref(FileObject file) :
	ref(file.impl)
{
}

FileObject FileObject::Ref::Get()
{
	shared_ptr<Impl> impl = ref.lock();
	if (!impl)
		throw Exc("File object have been destroyed");
	return impl;
}

void SetterThread()
{
	for (int i = 0; i < cycles; ++ i)
	{
		// create file object
		FileObject file;
		// assign reference to global variable
		*DataAccess::Access() = file;
		// create file itself
		file.Open("file.txt");
		// write some text, file will be opened because accesser doesn't use close
		// (try ... catch is not needed)
		file.Write(String().Cat() << "[" << i << "] setter");
		// close the file, accesser now cannot write into file
		file.Close();
	}
}

void AccesserThread()
{
	for (int i = 0; i < cycles; ++ i)
	{
		try
		{
			// try to get the real object from global reference
			FileObject file = DataAccess::Access()->Get();
			for (int j = 0; j < internalCycles; ++ j)
			{
				// try to write into file
				file.Write(String().Cat() << "[" << i << "," << j << "] accesser");	
			}
		}
		catch(Exc& e)
		{
			Out(String().Cat() << "[" << i << "] Accesser error: " << e);
		}
	}
}

}

