#ifndef _PainterSvg_Svg_h_
#define _PainterSvg_Svg_h_

#include "Style.h"
#include "Transform.h"

NAMESPACE_UPP

namespace SVG {

enum ElementType {
	etSvg,
	etG,
	etSwitch,
	etA,
	etDef,
	etAnimation,
	etVideo,
	etAudio,
	etUse,
	etCircle,
	etEllipse,
	etImage,
	etLine,
	etPath,
	etPolygon,
	etPolyline,
	etRect,
	etText,
	etTextArea,
	etAnimateColor,
	etTspan,
	etForeign,
};
	
enum BufferedRendering {
	brAuto,
	brDymamic,
	brStatic,
};

enum ColorRendering {
	crAuto,
	crQuality,
	crSpeed,
};

enum Direction {
	drLtr,
	drRtl,
};

enum DisplayAlign {
	daAuto,
	daBefore,
	daCenter,
	daAfter,
};

enum AttributeName {
	anWidth,
	anHeight,
	anFill,
};

enum CalcMode {
	cmDiscrete,
	cmLinear,
	cmPaced,
	cmSpline,
};

class BaseElement {
protected:
	ElementType et;
	String id;
	
public:
	virtual ~BaseElement() {
	}
	
	virtual void ParseAttr(XmlParser &xp) = 0;
	
	virtual void ParseChild(XmlParser &xp) {
		if (xp.IsTag()) {
			LOG("Unknown child " + xp.ReadTag());
			xp.SkipEnd();
		}
		else
			xp.Skip();
	}
	
	void Parse(XmlParser &xp) {
		ParseAttr(xp);
		while(!xp.End()) 
			ParseChild(xp);
	}
	
	String GetId() const{
		return id;
	}
	
	void SetVersion(const String& aId) {
		id = aId;
	}
};
	
class Element: public BaseElement {
public:
	Element* parent;
	SvgStyle style;
	Svg2DTransform transf;
	
	String title;
	String desc;
	
	Array<BaseElement> anims;
	
	Element(): parent(NULL) {
	}
	
	bool IsContainer() const {
		return et >= etSvg && et <= etDef;
	}
	
	bool IsMedia() const {
		return et >= etAnimation && et <= etAudio;
	}
	
	bool IsGraphics() const {
		return (et >= etCircle && et <= etTextArea) 
			|| et == etAnimation || et == etVideo || et == etUse;
	}
	
	bool IsText() const {
		return et == etText || et == etTextArea || et == etTspan;
	}
	
	bool Is(ElementType aet) const {
		return et == aet;
	}
	
	Element& GetStParrent() {
		if (parent == NULL)
			return *this;
		else
			return *parent;
	}
	
	void SetParent(Element* e) {
		parent = e;
		style = e->style;
		transf = e->transf;
	}
	
	virtual void Paint(Painter& sw) = 0;
	
	bool CommonItem(const String& attr, const String& val);
	bool StyleItem(const String& attr, const String& val);
	
	virtual double Get(AttributeName aName) const  {
		return Null;
	}
	
	virtual void Set(AttributeName aName, double aValue) {
	}
};
	
class Rect: public Element {
public:
	double x, y, width, height;
	
	Rect() {
		et = etRect;
		x = y = width = height = 0;
	}
	
	Rect(Element* e) {
		et = etRect;
		x = y = width = height = 0;
		SetParent(e);
	}
	
	void Paint(Painter& sw) {
		if (width == 0 || height == 0)
			return;
		transf.Apply(sw);
		sw.Rectangle(x, y, width, height);	
		style.Apply(sw);
	}

	void ParseAttr(XmlParser &xp);
	void ParseChild(XmlParser &xp);
	
	virtual double Get(AttributeName aName) const;
	virtual void Set(AttributeName aName, double aValue);
};

class Line: public Element {
public:
	double x1, y1, x2, y2;
	
	Line() {
		et = etLine;
		x1 = y1 = x2 = y2 = 0;
	}
	
	Line(Element* e) {
		et = etLine;
		x1 = y1 = x2 = y2 = 0;
		SetParent(e);
	}
	
	void Paint(Painter& sw) {
		transf.Apply(sw);
		sw.Move(x1, y1);
		sw.Line(x2, y2);	
		style.Apply(sw);
	}
	
	void ParseAttr(XmlParser &xp);
};

class Ellipse: public Rect {
public:
	Ellipse() {
		et = etEllipse;
	}
	
	Ellipse(Element* e) {
		et = etEllipse;
		SetParent(e);
	}
	
	void Paint(Painter& sw) {
		if (width == 0 || height == 0)
			return;
		transf.Apply(sw);
		sw.Ellipse(x, y, width, height);	
		style.Apply(sw);
	}
	
	void ParseAttr(XmlParser &xp);
};

class Circle: public Element {
public:
	double x, y, r;
	
	Circle() {
		et = etCircle;
		x = y = r = 0;
	}
	
	Circle(Element* e) {
		et = etCircle;
		x = y = r = 0;
		SetParent(e);
	}
	
	void Paint(Painter& sw) {
		if (r == 0)
			return;
		transf.Apply(sw);
		sw.Circle(x, y, r);		
		style.Apply(sw);
	}
	
	void ParseAttr(XmlParser &xp);
};

class G: public Element {
public:
	Array<Element> elements;
	 
	G() {
		et = etG;
	}
	
	G(Element* e) {
		et = etG;
		SetParent(e);
	}
	
	void Paint(Painter& sw) {
		for (int i = 0; i < elements.GetCount(); i++)
			elements[i].Paint(sw);
	}
	
	void ParseAttr(XmlParser &xp);
	void ParseChild(XmlParser &xp);
};

class A: public G {
public:
	A() {
		et = etA;
	}
	
	A(Element* e) {
		et = etA;
		SetParent(e);
	}
};

class Text: public Element {
public:
	double x, y;
	String text;
	
	Text() {
		et = etText;
		x = y = 0;
	}
	
	Text(Element* e) {
		et = etText;
		x = y = 0;
		SetParent(e);
	}
	
	void Paint(Painter& sw);	
	void ParseAttr(XmlParser &xp);
};

class Path: public Element {
public:
	String path;
	
	Path() {
		et = etPath;
	}
	
	Path(Element* e) {
		et = etPath;
		SetParent(e);
	}
	
	void Paint(Painter& sw);
	void ParseAttr(XmlParser &xp);
	void ParseChild(XmlParser &xp);
};

class Polyline: public Element {
public:
	Vector<Pointf> points;
	
	Polyline() {
		et = etPolyline;
	}
	
	Polyline(Element* e) {
		et = etPolyline;
		SetParent(e);
	}
	
	void Paint(Painter& sw);
	void ParseAttr(XmlParser &xp);
};

class Animate: public BaseElement {	
public:
	double to;
	double from; 
	bool accSum, addReplace;
	AttributeName attrName;
	double begin, dur;
	double repeatCount;
	CalcMode calcMode;
	bool fillFreeze;
	Vector<double> values;
	
	Animate() {
		to = from = begin = dur = repeatCount = 0;
		accSum = addReplace = false;
		fillFreeze = false;
	}
	
	void ParseAttr(XmlParser &xp);
	virtual void Apply(Element& aEl);
};

}

enum SvgBaseProfile {
	bpSvgNone,
	bpSvgFull,
	bpSvgTiny,
	bpSvgBasic,
};

class SvgImage: public SVG::G {
private:
	double version;
	SvgBaseProfile profile;
	double width, height;
	Rectf viewBox;
	
public:
	SvgImage(): version(1.0), profile(bpSvgNone), width(0), height(0), viewBox(0, 0, 0, 0) {
		et = SVG::etSvg;
	}
	
	void ParseAttr(XmlParser &xp);
	
	double GetVersion() const{
		return version;
	}
	
	void SetVersion(double aVersion) {
		version = aVersion;
	}

	SvgBaseProfile GetProfile() const{
		return profile;
	}
	
	void SetVersion(SvgBaseProfile aProfile) {
		profile = aProfile;
	}
	
	double GetWidth() const{
		return width;
	}
	
	double GetHeight() const{
		return height;
	}
};

END_UPP_NAMESPACE

#endif
