#include "Svg.h"
#include "Helpers.h"

NAMESPACE_UPP

String svgFileFolder = "";

namespace SVG {

typedef void (*PropFunction)(const String& attr, const String& val, Element& e);
VectorMap<String, PropFunction> properties;

void PropAudioLevel(const String& attr, const String& val, Element& e) {
	if (!(e.IsMedia() || e.IsContainer() || e.Is(etUse))) {
		LOG("property " + attr + " not available for element type");
		return;
	}
				
	double d;
	if (val == "inherit")
		d = e.GetStParrent().style.audio;
	else
		d = minmax(StrDbl(val), 0.0, 1.0);
	e.style.audio = d;
}

void PropBufferedRendering(const String& attr, const String& val, Element& e) {
	if (!(e.IsGraphics() || e.IsContainer())) {
		LOG("property " + attr + " not available for element type");
		return;
	}
	
	// does not affect implementation
	BufferedRendering br;
	if (val == "inherit")
		br = brAuto;
	else if (val == "auto")
		br = brAuto;
	else if (val == "dynamic")
		br = brDymamic;
	else if (val == "static")
		br = brStatic;
	else
		LOG("property " + attr + " has invalid value");
}

void PropColor(const String& attr, const String& val, Element& e) {
	Color c;
	if (val == "inherit")
		c = e.GetStParrent().style.color;
	else
		c = GetColorXml(val);
	e.style.color = c;
}

void PropColorRendering(const String& attr, const String& val, Element& e) {
	if (!(e.IsGraphics() || e.IsContainer() || e.Is(etAnimateColor))) {
		LOG("property " + attr + " not available for element type");
		return;
	}
	
	// does not affect implementation
	ColorRendering cr;
	if (val == "inherit")
		cr = SVG::crAuto;
	else if (val == "optimizeSpeed")
		cr = SVG::crSpeed;
	else if (val == "optimizeQuality")
		cr = SVG::crQuality;
	else
		LOG("property " + attr + " has invalid value");
}

void PropDirection(const String& attr, const String& val, Element& e) {
	if (!(e.IsText())) {
		LOG("property " + attr + " not available for element type");
		return;
	}
	
	LOG("property " + attr + " not implemented");
			
	SVG::Direction dr;
	if (val == "inherit")
		dr = SVG::drLtr;
	else if (val == "ltr")
		dr = SVG::drLtr;
	else if (val == "rtl")
		dr = SVG::drRtl;
	else
		LOG("property " + attr + " has invalid value");			
}

void PropDisplay(const String& attr, const String& val, Element& e) {
	if (!(e.Is(etSvg) || e.Is(etG) || e.Is(etSwitch) || e.Is(etA) ||
		e.Is(etForeign) || e.IsText() || e.IsGraphics() || e.IsMedia())) {
		LOG("property " + attr + "not available for element type");
		return;
	}
	
	Display ds;
	if (val == "inherit")
		ds = dsInline;
	else if (val == "none")
		ds = dsNone;
	else if (val == "inline")
		ds = dsInline;
	else if (val == "block")
		ds = dsBlock;
	else
		LOG("property " + attr + " not implement all values");
	e.style.display = ds;
}

void PropDisplayAlign(const String& attr, const String& val, Element& e) {
	if (!(e.Is(etTextArea))) {
		LOG("property " + attr + " not available for element type");
		return;
	}
	
	LOG("property " + attr + " not implemented");
			
	DisplayAlign da;
	if (val == "inherit")
		da = daAuto;
	else if (val == "before")
		da = daBefore;
	else if (val == "center")
		da = daCenter;
	else if (val == "after")
		da = daAfter;
	else
		LOG("property " + attr + " has invalid value");
}

void PropFill(const String& attr, const String& val, Element& e) {
	if (!(e.Is(etG) || e.IsGraphics() || e.IsText())) {
		LOG("property " + attr + " not available for element type");
		return;
	}
	
	Color f;
	if (val == "inherit")
		f = e.GetStParrent().style.fill;
	else
		f = GetColorXml(val);
	e.style.fill = f;
}

void PropFontFamily(const String& attr, const String& val, Element& e) {
	if (!(e.Is(etG) || e.IsText())) {
		LOG("property " + attr + " not available for element type");
		return;
	}
	
	int f;
	if (val == "inherit")
		f = e.GetStParrent().style.fontFamily;
	else if (val.Find("Roman") >= 0)
		f = Font::ROMAN;
	else if (val.Find("Arial") >= 0)
		f = Font::ARIAL;			
	else if (val.Find("Courier") >= 0)
		f = Font::COURIER;
	else 
		f = Font::ARIAL;
	e.style.fontFamily = f;
}

void PropFontSize(const String& attr, const String& val, Element& e) {
	if (!(e.Is(etG) || e.IsText())) {
		LOG("property " + attr + " not available for element type");
		return;
	}
	
	double f;
	if (val == "inherit")
		f = e.GetStParrent().style.fontSize;
	else
		f = StrDbl(val);
	e.style.fontSize = f;
}

void PropFontStyle(const String& attr, const String& val, Element& e) {
	if (!(e.Is(etG) || e.IsText())) {
		LOG("property " + attr + " not available for element type");
		return;
	}
	
	FontStyle fs;
	if (val == "inherit")
		fs = e.GetStParrent().style.fontStyle;
	else if (val == "normal")
		fs = fsNormal;
	else if (val == "italic")
		fs = fsItalic;
	else if (val == "oblique") {
		fs = fsOblique;
		LOG(attr + " 'oblique' implemented as italic");
	}
	else
		LOG("property " + attr + " has invalid value");
	e.style.fontStyle = fs;
}

void PropStroke(const String& attr, const String& val, Element& e) {
	if (!(e.Is(etG) || e.IsGraphics() || e.IsText())) {
		LOG("property " + attr + " not available for element type");
		return;
	}
	
	Color s;
	if (val == "inherit")
		s = e.GetStParrent().style.stroke;
	else
		s = GetColorXml(val);
	e.style.stroke = s;
}

void PropStrokeWidth(const String& attr, const String& val, Element& e) {
	if (!(e.Is(etG) || e.IsGraphics() || e.IsText())) {
		LOG("property " + attr + " not available for element type");
		return;
	}
	
	double s;
	if (val == "inherit")
		s = e.GetStParrent().style.strokeWidth;
	else
		s = StrDbl(val);
	e.style.strokeWidth = s;
}

void PropFillOpacity(const String& attr, const String& val, Element& e) {
	if (!(e.IsGraphics() || e.IsText())) {
		LOG("property " + attr + " not available for element type");
		return;
	}
	
	double s;
	if (val == "inherit")
		s = e.GetStParrent().style.fillOpacity;
	else
		s = StrDbl(val);
	e.style.fillOpacity = s;
}

void PropStrokeOpacity(const String& attr, const String& val, Element& e) {
	if (!(e.IsGraphics() || e.IsText())) {
		LOG("property " + attr + " not available for element type");
		return;
	}
	
	double s;
	if (val == "inherit")
		s = e.GetStParrent().style.strokeOpacity;
	else
		s = StrDbl(val);
	e.style.strokeOpacity = s;
}

void PropFillRule(const String& attr, const String& val, Element& e) {
	if (!(e.IsGraphics() || e.IsText())) {
		LOG("property " + attr + " not available for element type");
		return;
	}
	
	bool b;
	if (val == "inherit")
		b = e.GetStParrent().style.fillStyleEvenOdd;
	else if (val == "nonzero")
		b = false;
	else if (val == "evenodd")
		b = true;
	e.style.fillStyleEvenOdd = b;
}

}

void SvgPaint_Image(Painter& sw, XmlParser &xp, Svg2DTransform transf, SvgStyle style)
{
	int x, y, width, height;
	String fileName;
	
	for (int i = 0; i < xp.GetAttrCount(); ++i) {
		String attr = xp.GetAttr(i);
		if (attr == "x") 
			x = atoi(xp[i]);
		else if (attr == "y") 
			y = atoi(xp[i]);
		else if (attr == "width") 
			width = atoi(xp[i]);
		else if (attr == "height") 
			height = atoi(xp[i]);
		else if (attr == "xlink:href") 
			fileName = xp[i];
		else if (attr == "style") 
			style.Get(xp[i]);
		else if (attr == "transform") {
			Array<double> args;
			args = GetTransformArgs(xp[i], "translate");
			if (!args.IsEmpty()) 
				transf.Translate(args[0], args[1]);
			args = GetTransformArgs(xp[i], "scale");
			if (!args.IsEmpty()) 
				if (args.GetCount() == 2)
					transf.Scale(args[0], args[1]);
				else if (args.GetCount() == 1)
					transf.Scale(args[0], args[0]);		
		}
	}				
	if (!FileExists(fileName))
		fileName = AppendFileName(svgFileFolder, fileName);
	transf.Apply(sw);
	sw.Rectangle(x, y, width, height).Fill(StreamRaster::LoadFileAny(fileName), x, y, width, 0).Stroke(0, Black());
	style.Apply(sw);
}

void SvgPaint_Polygon(Painter& sw, XmlParser &xp, Svg2DTransform transf, SvgStyle style)
{
	Array<Point> points;
	
	for (int i = 0; i < xp.GetAttrCount(); ++i) {
		String attr = xp.GetAttr(i);
		if (attr == "points") {
			
		} 
		else if (attr == "style") 
			style.Get(xp[i]);
		else if (attr == "stroke-width") 
			style.strokeWidth = StrInt(xp[i]);
		else if (attr == "transform") {
			Array<double> args;
			args = GetTransformArgs(xp[i], "translate");
			if (!args.IsEmpty()) 
				transf.Translate(args[0], args[1]);
			args = GetTransformArgs(xp[i], "scale");
			if (!args.IsEmpty()) 
				if (args.GetCount() == 2)
					transf.Scale(args[0], args[1]);
				else if (args.GetCount() == 1)
					transf.Scale(args[0], args[0]);		
		}
	}			
	
}

int tick;

void ParseSVG(Painter& p, const char *svg, const char *folder, int ms)
{
	svgFileFolder = folder;
	
	XmlParser xp(svg);
	while(!xp.IsTag())
		xp.Skip();
	xp.PassTag("svg");
	
	tick = ms;	
	p.Begin();
	SvgImage g;
	g.Parse(xp);
	g.Paint(p);
	p.End();
}

INITBLOCK {
	using namespace SVG;
	
	properties.Add("audio-level", PropAudioLevel);
	properties.Add("buffered-rendering", PropBufferedRendering);
	properties.Add("color", PropColor);
	properties.Add("color-rendering", PropColorRendering);
	properties.Add("direction", PropDirection);
	properties.Add("display", PropDisplay);
	properties.Add("display-align", PropDisplayAlign);
	properties.Add("fill", PropFill);
	properties.Add("fill-rule", PropFillRule);
	properties.Add("fill-opacity", PropFillOpacity);
	properties.Add("font-family", PropFontFamily);
	properties.Add("font-size", PropFontSize);
	properties.Add("font-style", PropFontStyle);
	properties.Add("stroke", PropStroke);
	properties.Add("stroke-width", PropStrokeWidth);
	properties.Add("stroke-opacity", PropStrokeOpacity);
}

END_UPP_NAMESPACE
