#include "Svg.h"
#include "Helpers.h"


NAMESPACE_UPP

extern int tick;

namespace SVG {

Vector<Pointf> GetPolygonPointsXml(String &strpoints)
{
	Vector<Pointf> points;
	int newpos, pos;
	char separator;
	
	if (strpoints.Find(',') >= 0)
		separator = ',';
	else
		separator = ' ';
	
	pos = newpos = 0;
	while (newpos != strpoints.GetCount()) {
		int x, y;
		if ((newpos = strpoints.Find(separator, pos)) < 0)
			break;
		x = atoi(strpoints.Mid(pos, newpos+1));
		pos = newpos+1;
		if ((newpos = strpoints.Find(' ', pos+1 < strpoints.GetCount()? pos+1: pos)) < 0)	// Jump possible spaces 
			newpos = strpoints.GetCount();
		y = atoi(strpoints.Mid(pos, newpos+1));
		pos = newpos+1;
		Pointf &point = points.Add();
		point.x = x;
		point.y = y;
	}
	return points;
}
	
typedef void (*PropFunction)(const String& attr, const String& val, Element& e);

bool Element::CommonItem(const String& attr, const String& val) {
	if (attr == "transform") {
		Array<double> args;
		args = GetTransformArgs(val, "translate");
		if (args.GetCount() >= 2) 
			transf.Translate(args[0], args[1]);
		args = GetTransformArgs(val, "scale");
		if (!args.IsEmpty()) 
			if (args.GetCount() == 2)
				transf.Scale(args[0], args[1]);
			else if (args.GetCount() == 1)
				transf.Scale(args[0], args[0]);
		args = GetTransformArgs(val, "rotate");
		if (!args.IsEmpty())
			transf.Rotate(args[0] * M_PI / 180); 
		return true;
	}
	else if (attr == "id") {
		id = val;
		return true;
	}
	return false;
}

extern VectorMap<String, PropFunction> properties;

bool Element::StyleItem(const String& attr, const String& val) { 
	int ind = properties.Find(attr);
	if (ind != -1) {
		PropFunction f = properties[ind];
		f(attr, val, *this);
		return true;
	}
	return false;
}

void Rect::ParseAttr(XmlParser &xp) {
	for (int i = 0; i < xp.GetAttrCount(); i++) {
		String attr = xp.GetAttr(i);
		String val = xp[i];
		if (StyleItem(attr, val) || CommonItem(attr, val))
			continue;
		else if (attr == "x") 
			x = StrDbl(val);
		else if (attr == "y") 
			y = StrDbl(val);
		else if (attr == "width") 
			width = StrDbl(val);
		else if (attr == "height") 
			height = StrDbl(val);
		//else if (attr == "style") 
		//	style.Get(val);
		else
			LOG("Unknown attr " + attr);
	}
}

void Rect::ParseChild(XmlParser &xp) {
	if (xp.Tag("animate")) {
		Animate* a = new Animate();
		a->Parse(xp);
		a->Apply(*this);
		anims.Add(a);
	}
	else if (xp.IsTag()) {
		LOG("Unknown child " + xp.ReadTag());
		xp.SkipEnd();
	}
	else
		xp.Skip();
}

double Rect::Get(AttributeName aName) const {
	if (aName == anWidth)
		return width;
	else if (aName == anHeight)
		return height;
	else 
		return Null;
}

void Rect::Set(AttributeName aName, double aValue) {
	if (aName == anWidth)
		width = aValue;
	else if (aName == anHeight)
		height = aValue;
}
	
void Line::ParseAttr(XmlParser &xp) {
	for (int i = 0; i < xp.GetAttrCount(); i++) {
		String attr = xp.GetAttr(i);
		String val = xp[i];
		if (StyleItem(attr, val) || CommonItem(attr, val))
			continue;
		else if (attr == "x1") 
			x1 = StrDbl(val);
		else if (attr == "y1") 
			y1 = StrDbl(val);
		else if (attr == "x2") 
			x2 = StrDbl(val);
		else if (attr == "y2") 
			y2 = StrDbl(val);
		//else if (attr == "style") 
		//	style.Get(val);
		else
			LOG("Unknown attr " + attr);
	}					
}

void Ellipse::ParseAttr(XmlParser &xp) {
	for (int i = 0; i < xp.GetAttrCount(); i++) {
		String attr = xp.GetAttr(i);
		String val = xp[i];
		if (StyleItem(attr, val) || CommonItem(attr, val))
			continue;
		else if (attr == "cx") 
			x = StrDbl(val);
		else if (attr == "cy") 
			y = StrDbl(val);
		else if (attr == "rx") 
			width = StrDbl(val);
		else if (attr == "ry") 
			height = StrDbl(val);
		//else if (attr == "style") 
		//	style.Get(xp[i]);
		else
			LOG("Unknown attr " + attr);
	}				
}

void Circle::ParseAttr(XmlParser &xp) {
	for (int i = 0; i < xp.GetAttrCount(); ++i) {
		String attr = xp.GetAttr(i);
		String val = xp[i];
		if (StyleItem(attr, val) || CommonItem(attr, val))
			continue;
		else if (attr == "cx") 
			x = StrDbl(val);
		else if (attr == "cy") 
			y = StrDbl(val);
		else if (attr == "r") 
			r = StrDbl(val);
		//else if (attr == "style") 
		//	style.Get(xp[i]);
		else
			LOG("Unknown attr " + attr);
	}		
}

void G::ParseAttr(XmlParser &xp) {	
	for (int i = 0; i < xp.GetAttrCount(); ++i) {
		String attr = xp.GetAttr(i);
		String val = xp[i];
		if (StyleItem(attr, val) || CommonItem(attr, val))
			continue;
		//else if (attr == "style") 
		//	style.Get(xp[i]);
		else if (attr == "stroke-width") 
			style.strokeWidth = StrInt(val);
		else if (attr == "font-size") 
			style.fontSize = StrInt(val);
		else if (attr == "text-anchor") {
			if (xp[i] == "middle")
				style.anchor = 1;
			else if (xp[i] == "end")
				style.anchor = 2;
		}
		else
			LOG("Unknown attr " + attr);
	}
}

void G::ParseChild(XmlParser &xp) {
	if(xp.Tag("rect"))
		elements.Add(new SVG::Rect(this)).Parse(xp);
	else if(xp.Tag("ellipse")) 
		elements.Add(new SVG::Ellipse(this)).Parse(xp);
	else if(xp.Tag("circle"))
		elements.Add(new SVG::Circle(this)).Parse(xp);
	else if(xp.Tag("line")) 
		elements.Add(new SVG::Line(this)).Parse(xp);
	else if(xp.Tag("polygon"))
		elements.Add(new SVG::Polyline(this)).Parse(xp);			
	else if(xp.Tag("polyline"))
		elements.Add(new SVG::Polyline(this)).Parse(xp);	
	else if(xp.Tag("path")) 
		elements.Add(new SVG::Path(this)).Parse(xp);
	/*else if(xp.TagE("image"))
		SvgPaint_Image(p, xp, transf, style);*/		
	else if(xp.Tag("text")) {
		Element& e = elements.Add(new SVG::Text(this));
		e.Parse(xp);
	}	
	else if(xp.Tag("g")) {
		elements.Add(new SVG::G(this));
		elements.Top().ParseAttr(xp);
		while (!xp.End())
			elements.Top().ParseChild(xp);
	}
	else if(xp.Tag("a")) {
		elements.Add(new SVG::A(this));
		elements.Top().ParseAttr(xp);
		while (!xp.End())
			elements.Top().ParseChild(xp);
	}
	else if(xp.Tag("desc"))
		desc = xp.ReadTextE();
	else if(xp.Tag("title"))
		title = xp.ReadTextE();
	else if (xp.IsTag()) {
		LOG("Unknown tag " + xp.ReadTag());
		xp.SkipEnd();
	}
	else
		xp.Skip();
}

void Text::ParseAttr(XmlParser &xp) {
	text = xp.ReadText();
	for (int i = 0; i < xp.GetAttrCount(); ++i) {
		String attr = xp.GetAttr(i);
		String val = xp[i];
		if (StyleItem(attr, val) || CommonItem(attr, val))
			continue;
		else if (attr == "x") 
			x = StrDbl(val);
		else if (attr == "y") 
			y = StrDbl(val);
		else if (attr == "text-anchor") {
			if (xp[i] == "middle")
				style.anchor = 1;
			else if (xp[i] == "end")
				style.anchor = 2;
		/*}
		else if (attr == "style") { 
			style.Get(xp[i]);
			
			if (style.fill.IsNullInstance())
				style.fill = style.stroke;
			String fontText;
			fontText = GetValueStringXml(xp[i], "font-family:");
			if (fontText.Find("Roman") >= 0)
				f.Face(Font::ROMAN);
			else if (fontText.Find("Arial") >= 0)
				f.Face(Font::ARIAL);			
			else if (fontText.Find("Courier") >= 0)
				f.Face(Font::COURIER);		
			style.fontSize = atoi(GetValueStringXml(xp[i], "font-size:")) * factor;
			if (style.fontSize == 0)
				style.fontSize = 12 * factor;
			f.Height(style.fontSize);*/
		}
		else
			LOG("Unknown attr " + attr);
	}
}
	
void Text::Paint(Painter& sw) {
	transf.Translate(x, y - style.fontSize);
	transf.Apply(sw);
	Font f;
	f.Face(style.fontFamily);
	f.Height((int)style.fontSize);
	if (style.fontStyle == fsItalic || style.fontStyle == fsOblique)
		f.Italic();
	int dx;
	if (style.anchor == 1)
		dx = GetTextSize(text, f).cx / 2;
	else if (style.anchor == 2)
		dx = GetTextSize(text, f).cx;
	else
		dx = 0;
	sw.Text(-dx, 0, text, f);
	if (style.stroke.IsNullInstance() && style.fill.IsNullInstance())
		style.fill = Black();
	style.Apply(sw);
}

void Path::Paint(Painter& sw) {
	transf.Apply(sw);
	SvgStyle temp = style;	
	if (!path.IsEmpty()) {
		if (!style.fill.IsNullInstance()) {
			sw.Path(path);
			SvgStyle temp = style;	
			temp.strokeWidth = 1;
			temp.stroke = Null;
			temp.Apply(sw);
		}
		if (!style.stroke.IsNullInstance()) {
			sw.Path(path);
			style.fill = Null;
			style.Apply(sw);
		}
	}
}

void Path::ParseAttr(XmlParser &xp) {
	for (int i = 0; i < xp.GetAttrCount(); ++i) {
		String attr = xp.GetAttr(i);
		String val = xp[i];
		if (StyleItem(attr, val) || CommonItem(attr, val))
			continue;
		//else if (attr == "style") 
		//	style.Get(xp[i]);
		else if (attr == "d")
			path = val;
		else
			LOG("Unknown attr " + attr);
	}
}

void Path::ParseChild(XmlParser &xp) {
	if (xp.Tag("animateMotion")) {
		for (int i = 0; i < xp.GetAttrCount(); i++) {
			String attr = xp.GetAttr(i);
			String val = xp[i];
			if (attr == "to") { 
				Vector<String> to = Split(val, ',');
				transf.Translate(StrDbl(to[0]), StrDbl(to[1]));
			}
			else if (attr == "values") { 
				Vector<String> vals = Split(val, ';');
				Vector<String> to = Split(vals[vals.GetCount() - 1], ',');
				transf.Translate(StrDbl(to[0]), StrDbl(to[1]));
			}
			else if (attr == "path") { 
				Vector<String> vals = Split(val, ' ');
				Vector<String> to = Split(vals[vals.GetCount() - 1], ',');
				transf.Translate(StrDbl(to[0]), StrDbl(to[1]));
			}
			//else
			//LOG("Unknown attr " + attr);
		}
		xp.SkipEnd();
	}
	else
		xp.Skip();
}

void Polyline::ParseAttr(XmlParser &xp) {
	for (int i = 0; i < xp.GetAttrCount(); ++i) {
		String attr = xp.GetAttr(i);
		String val = xp[i];
		if (StyleItem(attr, val) || CommonItem(attr, val))
			continue;
		else if (attr == "points") {
			String strpoints;
			strpoints = NormalizeSpaces(xp[i]);
			points = GetPolygonPointsXml(strpoints);
		}
		else
			LOG("Unknown attr " + attr);
	}
}

void Polyline::Paint(Painter& sw) {
	if (points.GetCount()) {
		transf.Apply(sw);
		sw.Move(points[0].x, points[0].y);
		for (int i = 1; i < points.GetCount(); ++i)				
			sw.Line(points[i].x, points[i].y);
		style.Apply(sw);
	}
}

void Animate::ParseAttr(XmlParser &xp) {
	for (int i = 0; i < xp.GetAttrCount(); ++i) {
		String attr = xp.GetAttr(i);
		String val = xp[i];
		if (attr == "to") 
			to = StrDbl(val);
		else if (attr == "from") 
			from = StrDbl(val);
		else if (attr == "additive") {
			if (val == "replace")
				addReplace = true;
			else
				addReplace = false;
		}
		else if (attr == "accumulate") {
			if (val == "sum")
				accSum = true;
			else
				accSum = false;
		}
		else if (attr == "attributeName") {
			if (val == "width")
				attrName = anWidth;
			else if (val == "height")
				attrName = anHeight;
			else if (val == "fill")
				attrName = anFill;
			else
				LOG("attributeName invalid " + val);
		}
		else if (attr == "calcMode") {
			if (val == "discrete")
				calcMode = cmDiscrete;
			else if (val == "linear")
				calcMode = cmLinear;
			else if (val == "paced")
				calcMode = cmPaced;
			else if (val == "spline")
				calcMode = cmSpline;
			else
				LOG("calcMode invalid");
		}
		else if (attr == "repeatCount") 
			repeatCount = StrDbl(val);
		else if (attr == "begin") 
			begin = StrDbl(val);
		else if (attr == "dur") 
			dur = StrDbl(val);
		else if (attr == "fill") {
			if (val == "freeze")
				fillFreeze = true;
			else if (val == "remove")
				fillFreeze = false;
			else
				LOG("fill invalid");
		}
		else if (attr == "id")
			id = val;
		else if (attr == "values") {
			Vector<String> v = Split(val, ';');
			for (int i = 0; i < v.GetCount(); i++)
				values.Add(StrDbl(v[i]));
		}			
		else
			LOG("Unknown attr " + attr);
	}
}

void Animate::Apply(Element& aEl) {
	double orig = aEl.Get(attrName);
	if (from && to) {
		double delta = from - to;
		double f = (400.0 - tick) / 400.0;
		if (f < 0)
			f = 0;
		delta = to + delta * f;
		double h = delta;
		if (!addReplace) 
			h += orig;
		if (accSum)
			h += orig;
		aEl.Set(attrName, h);
	}
	else if (values.GetCount()) {
		long l = tick;
		double f = tick / 800.0;
		int i = (int)f;
		if (i < 0)
			i = 0;
		if (i > values.GetCount() - 1)
			i = values.GetCount() - 1;
		aEl.Set(attrName, values[i]);
		/*if (!a->addReplace) 
			height += orig;
		if (a->accSum)
			height += orig;*/
	}
}

}

void SvgImage::ParseAttr(XmlParser &xp) {	
	for (int i = 0; i < xp.GetAttrCount(); ++i) {
		String attr = xp.GetAttr(i);
		String val = xp[i];
		if (attr == "version") {
			if (val == "1.0")
				version = 1.0;
			else if (val == "1.1")
				version = 1.1;
			else if (val == "1.2")
				version = 1.2;
			else
				LOG("Invalid SVG version");
		}
		else if (attr == "baseProfile") {
			if (val == "none")
				profile = bpSvgNone;
			else if (val == "basic")
				profile = bpSvgBasic;
			else if (val == "full")
				profile = bpSvgFull;
			else if (val == "tiny")
				profile = bpSvgTiny;
			else
				LOG("Invalid SVG baseProfile");
		}
		else if (attr == "xmlns") {
			if (val != "http://www.w3.org/2000/svg")
				LOG("Invalid SVG xmlns");
		}
		else if (attr == "xmlns:xlink") {
			if (val != "http://www.w3.org/1999/xlink")
				LOG("Invalid SVG xmlns:xlink");
		}
		else if (attr == "id")
			id = val;
		else if (attr == "width") 
			width = StrDbl(val);
		else if (attr == "height") 
			height = StrDbl(val);
		else if (attr == "viewBox") {
			Vector<String> v = Split(val, ' ');
			if (v.GetCount() == 4) {
				viewBox.left = StrDbl(v[0]);
				viewBox.top = StrDbl(v[1]);	
				viewBox.right = StrDbl(v[2]);
				viewBox.bottom = StrDbl(v[3]);
			}
			else
				LOG("Invalid SVG viewBox");
		}
		else
			LOG("Unknown attr " + attr);
	}
}

END_UPP_NAMESPACE