//USQLite3.cpp
//modified from Rob Groves's CppSQLite3.cpp,see http://www.codeproject.com/database/CppSQLite.asp


#include "USQLite3.h"
//#include <cstdlib>


// Named constant for passing to USQLite3Exception when passing it a string
// that cannot be deleted.
//static const bool DONT_DELETE_MSG=false;
////////////////////////////////////////////////////////////////////////////////

USQLite3Exception::USQLite3Exception(const int nErrCode,String szErrMess) :mnErrCode(nErrCode)
{
    mpszErrMess = errorCodeAsString(nErrCode)+AsString(nErrCode)+szErrMess;
}

									
USQLite3Exception::USQLite3Exception(const USQLite3Exception&  e) :
									mnErrCode(e.mnErrCode)
{
	mpszErrMess.Clear();
	if (!e.mpszErrMess.IsEmpty())
	{
		mpszErrMess = e.mpszErrMess;
	}
}


const String USQLite3Exception::errorCodeAsString(int nErrCode)
{
	switch (nErrCode)
	{
		case SQLITE_OK          : return "SQLITE_OK";
		case SQLITE_ERROR       : return "SQLITE_ERROR";
		case SQLITE_INTERNAL    : return "SQLITE_INTERNAL";
		case SQLITE_PERM        : return "SQLITE_PERM";
		case SQLITE_ABORT       : return "SQLITE_ABORT";
		case SQLITE_BUSY        : return "SQLITE_BUSY";
		case SQLITE_LOCKED      : return "SQLITE_LOCKED";
		case SQLITE_NOMEM       : return "SQLITE_NOMEM";
		case SQLITE_READONLY    : return "SQLITE_READONLY";
		case SQLITE_INTERRUPT   : return "SQLITE_INTERRUPT";
		case SQLITE_IOERR       : return "SQLITE_IOERR";
		case SQLITE_CORRUPT     : return "SQLITE_CORRUPT";
		case SQLITE_NOTFOUND    : return "SQLITE_NOTFOUND";
		case SQLITE_FULL        : return "SQLITE_FULL";
		case SQLITE_CANTOPEN    : return "SQLITE_CANTOPEN";
		case SQLITE_PROTOCOL    : return "SQLITE_PROTOCOL";
		case SQLITE_EMPTY       : return "SQLITE_EMPTY";
		case SQLITE_SCHEMA      : return "SQLITE_SCHEMA";
		case SQLITE_TOOBIG      : return "SQLITE_TOOBIG";
		case SQLITE_CONSTRAINT  : return "SQLITE_CONSTRAINT";
		case SQLITE_MISMATCH    : return "SQLITE_MISMATCH";
		case SQLITE_MISUSE      : return "SQLITE_MISUSE";
		case SQLITE_NOLFS       : return "SQLITE_NOLFS";
		case SQLITE_AUTH        : return "SQLITE_AUTH";
		case SQLITE_FORMAT      : return "SQLITE_FORMAT";
		case SQLITE_RANGE       : return "SQLITE_RANGE";
		case SQLITE_ROW         : return "SQLITE_ROW";
		case SQLITE_DONE        : return "SQLITE_DONE";
		case USQLITE3_ERROR    : return "USQLITE3_ERROR";
		default: return "UNKNOWN_ERROR";
	}
}


USQLite3Exception::~USQLite3Exception()
{
	if (!mpszErrMess.IsEmpty())
	{
		mpszErrMess.Clear();
	}
}


////////////////////////////////////////////////////////////////////////////////

USQLite3Query::USQLite3Query()
{
	mpVM = 0;
	mbEof = true;
	mnCols = 0;
	mbOwnVM = false;
}


USQLite3Query::USQLite3Query(const USQLite3Query& rQuery)
{
	mpVM = rQuery.mpVM;
	// Only one object can own the VM
	const_cast<USQLite3Query&>(rQuery).mpVM = 0;
	mbEof = rQuery.mbEof;
	mnCols = rQuery.mnCols;
	mbOwnVM = rQuery.mbOwnVM;
}


USQLite3Query::USQLite3Query(sqlite3* pDB,
							sqlite3_stmt* pVM,
							bool bEof,
							bool bOwnVM/*=true*/)
{
	mpDB = pDB;
	mpVM = pVM;
	mbEof = bEof;
	mnCols = sqlite3_column_count(mpVM);
	mbOwnVM = bOwnVM;
}


USQLite3Query::~USQLite3Query()
{
	try
	{
		finalize();
	}
	catch (...)
	{
	}
}


USQLite3Query& USQLite3Query::operator=(const USQLite3Query& rQuery)
{
	try
	{
		finalize();
	}
	catch (...)
	{
	}
	mpVM = rQuery.mpVM;
	// Only one object can own the VM
	const_cast<USQLite3Query&>(rQuery).mpVM = 0;
	mbEof = rQuery.mbEof;
	mnCols = rQuery.mnCols;
	mbOwnVM = rQuery.mbOwnVM;
	return *this;
}


int USQLite3Query::numFields()
{
	checkVM();
	return mnCols;
}


const String USQLite3Query::fieldValue(int nField)
{
	checkVM();

	if (nField < 0 || nField > mnCols-1)
	{
		throw USQLite3Exception(USQLITE3_ERROR,
								"Invalid field index requested");
	}

	return (const char*)sqlite3_column_text(mpVM, nField);
}


const String USQLite3Query::fieldValue(const String szField)
{
	int nField = fieldIndex(szField);
	return (const char*)sqlite3_column_text(mpVM, nField);
}


int USQLite3Query::getIntField(int nField, int nNullValue/*=0*/)
{
	if (fieldDataType(nField) == SQLITE_NULL)
	{
		return nNullValue;
	}
	else
	{
		return sqlite3_column_int(mpVM, nField);
	}
}


int USQLite3Query::getIntField(const String szField, int nNullValue/*=0*/)
{
	int nField = fieldIndex(szField);
	return getIntField(nField, nNullValue);
}


double USQLite3Query::getFloatField(int nField, double fNullValue/*=0.0*/)
{
	if (fieldDataType(nField) == SQLITE_NULL)
	{
		return fNullValue;
	}
	else
	{
		return sqlite3_column_double(mpVM, nField);
	}
}


double USQLite3Query::getFloatField(const String szField, double fNullValue/*=0.0*/)
{
	int nField = fieldIndex(szField);
	return getFloatField(nField, fNullValue);
}


const String USQLite3Query::getStringField(int nField, const String szNullValue/*=""*/)
{
	if (fieldDataType(nField) == SQLITE_NULL)
	{
		return szNullValue;
	}
	else
	{
		return (const char*)sqlite3_column_text(mpVM, nField);
	}
}


const String USQLite3Query::getStringField(const String szField, const String szNullValue/*=""*/)
{
	int nField = fieldIndex(szField);
	return getStringField(nField, szNullValue);
}


const String USQLite3Query::getBlobField(int nField, int& nLen)
{
	checkVM();

	if (nField < 0 || nField > mnCols-1)
	{
		throw USQLite3Exception(USQLITE3_ERROR,
								"Invalid field index requested");
	}

	nLen = sqlite3_column_bytes(mpVM, nField);
	return (const char*)sqlite3_column_blob(mpVM, nField);
}


const String USQLite3Query::getBlobField(const String szField, int& nLen)
{
	int nField = fieldIndex(szField);
	return getBlobField(nField, nLen);
}


bool USQLite3Query::fieldIsNull(int nField)
{
	return (fieldDataType(nField) == SQLITE_NULL);
}


bool USQLite3Query::fieldIsNull(const String szField)
{
	int nField = fieldIndex(szField);
	return (fieldDataType(nField) == SQLITE_NULL);
}


int USQLite3Query::fieldIndex(const String szField)
{
	checkVM();

	if (!szField.IsEmpty())
	{
		for (int nField = 0; nField < mnCols; nField++)
		{
			const String szTemp = AsString(sqlite3_column_name(mpVM, nField));

			if (szField==szTemp)
			{
				return nField;
			}
		}
	}

	throw USQLite3Exception(USQLITE3_ERROR,
							"Invalid field name requested");
}


const String USQLite3Query::fieldName(int nCol)
{
	checkVM();

	if (nCol < 0 || nCol > mnCols-1)
	{
		throw USQLite3Exception(USQLITE3_ERROR,
								"Invalid field index requested");
	}

	return sqlite3_column_name(mpVM, nCol);
}


const String USQLite3Query::fieldDeclType(int nCol)
{
	checkVM();

	if (nCol < 0 || nCol > mnCols-1)
	{
		throw USQLite3Exception(USQLITE3_ERROR,
								"Invalid field index requested");
	}

	return sqlite3_column_decltype(mpVM, nCol);
}


int USQLite3Query::fieldDataType(int nCol)
{
	checkVM();

	if (nCol < 0 || nCol > mnCols-1)
	{
		throw USQLite3Exception(USQLITE3_ERROR,
								"Invalid field index requested");
	}

	return sqlite3_column_type(mpVM, nCol);
}


bool USQLite3Query::eof()
{
	checkVM();
	return mbEof;
}


void USQLite3Query::nextRow()
{
	checkVM();

	int nRet = sqlite3_step(mpVM);

	if (nRet == SQLITE_DONE)
	{
		// no rows
		mbEof = true;
	}
	else if (nRet == SQLITE_ROW)
	{
		// more rows, nothing to do
	}
	else
	{
		nRet = sqlite3_finalize(mpVM);
		mpVM = 0;
		const String szError = sqlite3_errmsg(mpDB);
		throw USQLite3Exception(nRet,szError);
	}
}


void USQLite3Query::finalize()
{
	if (mpVM && mbOwnVM)
	{
		int nRet = sqlite3_finalize(mpVM);
		mpVM = 0;
		if (nRet != SQLITE_OK)
		{
			const String szError = sqlite3_errmsg(mpDB);
			throw USQLite3Exception(nRet, szError);
		}
	}
}


void USQLite3Query::checkVM()
{
	if (mpVM == 0)
	{
		throw USQLite3Exception(USQLITE3_ERROR,
								"Null Virtual Machine pointer");
	}
}


////////////////////////////////////////////////////////////////////////////////

USQLite3Table::USQLite3Table()
{
	mpaszResults = 0;
	mnRows = 0;
	mnCols = 0;
	mnCurrentRow = 0;
}


USQLite3Table::USQLite3Table(const USQLite3Table& rTable)
{
	mpaszResults = rTable.mpaszResults;
	// Only one object can own the results
	const_cast<USQLite3Table&>(rTable).mpaszResults = 0;
	mnRows = rTable.mnRows;
	mnCols = rTable.mnCols;
	mnCurrentRow = rTable.mnCurrentRow;
}


USQLite3Table::USQLite3Table(char** paszResults, int nRows, int nCols)
{
	mpaszResults = paszResults;
	mnRows = nRows;
	mnCols = nCols;
	mnCurrentRow = 0;
}


USQLite3Table::~USQLite3Table()
{
	try
	{
		finalize();
	}
	catch (...)
	{
	}
}


USQLite3Table& USQLite3Table::operator=(const USQLite3Table& rTable)
{
	try
	{
		finalize();
	}
	catch (...)
	{
	}
	mpaszResults = rTable.mpaszResults;
	// Only one object can own the results
	const_cast<USQLite3Table&>(rTable).mpaszResults = 0;
	mnRows = rTable.mnRows;
	mnCols = rTable.mnCols;
	mnCurrentRow = rTable.mnCurrentRow;
	return *this;
}


void USQLite3Table::finalize()
{
	if (mpaszResults)
	{
		sqlite3_free_table(mpaszResults);
		mpaszResults = 0;
	}
}


int USQLite3Table::numFields()
{
	checkResults();
	return mnCols;
}


int USQLite3Table::numRows()
{
	checkResults();
	return mnRows;
}


const String USQLite3Table::fieldValue(int nField)
{
	checkResults();

	if (nField < 0 || nField > mnCols-1)
	{
		throw USQLite3Exception(USQLITE3_ERROR,
								"Invalid field index requested");
	}

	int nIndex = (mnCurrentRow*mnCols) + mnCols + nField;
	return mpaszResults[nIndex];
}


const String USQLite3Table::fieldValue(const String szField)
{
	checkResults();

	if (!szField.IsEmpty())
	{
		for (int nField = 0; nField < mnCols; nField++)
		{
			if (strcmp(szField, mpaszResults[nField]) == 0)
			{
				int nIndex = (mnCurrentRow*mnCols) + mnCols + nField;
				return mpaszResults[nIndex];
			}
		}
	}

	throw USQLite3Exception(USQLITE3_ERROR,
							"Invalid field name requested");
}


int USQLite3Table::getIntField(int nField, int nNullValue/*=0*/)
{
	if (fieldIsNull(nField))
	{
		return nNullValue;
	}
	else
	{
		return atoi(fieldValue(nField));
	}
}


int USQLite3Table::getIntField(const String szField, int nNullValue/*=0*/)
{
	if (fieldIsNull(szField))
	{
		return nNullValue;
	}
	else
	{
		return atoi(fieldValue(szField));
	}
}


double USQLite3Table::getFloatField(int nField, double fNullValue/*=0.0*/)
{
	if (fieldIsNull(nField))
	{
		return fNullValue;
	}
	else
	{
		return atof(fieldValue(nField));
	}
}


double USQLite3Table::getFloatField(const String szField, double fNullValue/*=0.0*/)
{
	if (fieldIsNull(szField))
	{
		return fNullValue;
	}
	else
	{
		return atof(fieldValue(szField));
	}
}


const String USQLite3Table::getStringField(int nField, const String szNullValue/*=""*/)
{
	if (fieldIsNull(nField))
	{
		return szNullValue;
	}
	else
	{
		return fieldValue(nField);
	}
}


const String USQLite3Table::getStringField(const String szField, const String szNullValue/*=""*/)
{
	if (fieldIsNull(szField))
	{
		return szNullValue;
	}
	else
	{
		return fieldValue(szField);
	}
}


bool USQLite3Table::fieldIsNull(int nField)
{
	checkResults();
	return (fieldValue(nField).IsEmpty());
}


bool USQLite3Table::fieldIsNull(const String szField)
{
	checkResults();
	return (fieldValue(szField).IsEmpty());
}


const String USQLite3Table::fieldName(int nCol)
{
	checkResults();

	if (nCol < 0 || nCol > mnCols-1)
	{
		throw USQLite3Exception(USQLITE3_ERROR,
								"Invalid field index requested");
	}

	return mpaszResults[nCol];
}


void USQLite3Table::setRow(int nRow)
{
	checkResults();

	if (nRow < 0 || nRow > mnRows-1)
	{
		throw USQLite3Exception(USQLITE3_ERROR,
								"Invalid row index requested");
	}

	mnCurrentRow = nRow;
}


void USQLite3Table::checkResults()
{
	if (mpaszResults == 0)
	{
		throw USQLite3Exception(USQLITE3_ERROR,
								"Null Results pointer");
	}
}


////////////////////////////////////////////////////////////////////////////////

USQLite3Statement::USQLite3Statement()
{
	mpDB = 0;
	mpVM = 0;
}


USQLite3Statement::USQLite3Statement(const USQLite3Statement& rStatement)
{
	mpDB = rStatement.mpDB;
	mpVM = rStatement.mpVM;
	// Only one object can own VM
	const_cast<USQLite3Statement&>(rStatement).mpVM = 0;
}


USQLite3Statement::USQLite3Statement(sqlite3* pDB, sqlite3_stmt* pVM)
{
	mpDB = pDB;
	mpVM = pVM;
}


USQLite3Statement::~USQLite3Statement()
{
	try
	{
		finalize();
	}
	catch (...)
	{
	}
}


USQLite3Statement& USQLite3Statement::operator=(const USQLite3Statement& rStatement)
{
	mpDB = rStatement.mpDB;
	mpVM = rStatement.mpVM;
	// Only one object can own VM
	const_cast<USQLite3Statement&>(rStatement).mpVM = 0;
	return *this;
}


int USQLite3Statement::execDML()
{
	checkDB();
	checkVM();

	int nRet = sqlite3_step(mpVM);

	if (nRet == SQLITE_DONE)
	{
		int nRowsChanged = sqlite3_changes(mpDB);

		nRet = sqlite3_reset(mpVM);

		if (nRet != SQLITE_OK)
		{
			const String szError =sqlite3_errmsg(mpDB);
			throw USQLite3Exception(nRet,szError);
		}

		return nRowsChanged;
	}
	else
	{
		nRet = sqlite3_reset(mpVM);
        const String szError = sqlite3_errmsg(mpDB);
		throw USQLite3Exception(nRet, szError);
	}
}


USQLite3Query USQLite3Statement::execQuery()
{
	checkDB();
	checkVM();

	int nRet = sqlite3_step(mpVM);

	if (nRet == SQLITE_DONE)
	{
		// no rows
		return USQLite3Query(mpDB, mpVM, true/*eof*/, false);
	}
	else if (nRet == SQLITE_ROW)
	{
		// at least 1 row
		return USQLite3Query(mpDB, mpVM, false/*eof*/, false);
	}
	else
	{
		nRet = sqlite3_reset(mpVM);
		const String szError = sqlite3_errmsg(mpDB);
		throw USQLite3Exception(nRet, szError);
	}
}


void USQLite3Statement::bind(int nParam, const String szValue)
{
	checkVM();
	int nRes = sqlite3_bind_text(mpVM, nParam, szValue, -1, SQLITE_TRANSIENT);

	if (nRes != SQLITE_OK)
	{
		throw USQLite3Exception(nRes,
								"Error binding string param");
	}
}


void USQLite3Statement::bind(int nParam, const int nValue)
{
	checkVM();
	int nRes = sqlite3_bind_int(mpVM, nParam, nValue);

	if (nRes != SQLITE_OK)
	{
		throw USQLite3Exception(nRes,
								"Error binding int param");
	}
}


void USQLite3Statement::bind(int nParam, const double dValue)
{
	checkVM();
	int nRes = sqlite3_bind_double(mpVM, nParam, dValue);

	if (nRes != SQLITE_OK)
	{
		throw USQLite3Exception(nRes,
								"Error binding double param");
	}
}


void USQLite3Statement::bind(int nParam, const String blobValue, int nLen)
{
	checkVM();
	int nRes = sqlite3_bind_blob(mpVM, nParam,
								(const void*)blobValue, nLen, SQLITE_TRANSIENT);

	if (nRes != SQLITE_OK)
	{
		throw USQLite3Exception(nRes,
								"Error binding blob param");
	}
}

	
void USQLite3Statement::bindNull(int nParam)
{
	checkVM();
	int nRes = sqlite3_bind_null(mpVM, nParam);

	if (nRes != SQLITE_OK)
	{
		throw USQLite3Exception(nRes,
								"Error binding NULL param");
	}
}


void USQLite3Statement::reset()
{
	if (mpVM)
	{
		int nRet = sqlite3_reset(mpVM);

		if (nRet != SQLITE_OK)
		{
			const String szError = sqlite3_errmsg(mpDB);
			throw USQLite3Exception(nRet, szError);
		}
	}
}


void USQLite3Statement::finalize()
{
	if (mpVM)
	{
		int nRet = sqlite3_finalize(mpVM);
		mpVM = 0;

		if (nRet != SQLITE_OK)
		{
			const String szError = sqlite3_errmsg(mpDB);
			throw USQLite3Exception(nRet, szError);
		}
	}
}


void USQLite3Statement::checkDB()
{
	if (mpDB == 0)
	{
		throw USQLite3Exception(USQLITE3_ERROR,
								"Database not open");
	}
}


void USQLite3Statement::checkVM()
{
	if (mpVM == 0)
	{
		throw USQLite3Exception(USQLITE3_ERROR,
								"Null Virtual Machine pointer");
	}
}


////////////////////////////////////////////////////////////////////////////////

USQLite3DB::USQLite3DB()
{
	mpDB = 0;
	mnBusyTimeoutMs = 60000; // 60 seconds
}


USQLite3DB::USQLite3DB(const USQLite3DB& db)
{
	mpDB = db.mpDB;
	mnBusyTimeoutMs = 60000; // 60 seconds
}


USQLite3DB::~USQLite3DB()
{
	close();
}


USQLite3DB& USQLite3DB::operator=(const USQLite3DB& db)
{
	mpDB = db.mpDB;
	mnBusyTimeoutMs = 60000; // 60 seconds
	return *this;
}


void USQLite3DB::open(const String szFile)
{
	int nRet = sqlite3_open(szFile, &mpDB);

	if (nRet != SQLITE_OK)
	{
		const String szError = sqlite3_errmsg(mpDB);
		throw USQLite3Exception(nRet,szError);
	}

	setBusyTimeout(mnBusyTimeoutMs);
}


void USQLite3DB::close()
{
	if (mpDB)
	{
		sqlite3_close(mpDB);
		mpDB = 0;
	}
}


USQLite3Statement USQLite3DB::compileStatement(const String szSQL)
{
	checkDB();

	sqlite3_stmt* pVM = compile(szSQL);
	return USQLite3Statement(mpDB, pVM);
}


bool USQLite3DB::tableExists(const String szTable)
{
	String szSQL;
	szSQL="select count(*) from sqlite_master where type='table' and name='"+szTable+"';";
	int nRet = execScalar(szSQL);
	return (nRet > 0);
}


int USQLite3DB::execDML(const String szSQL)
{
	checkDB();

	char* szError=0;

	int nRet = sqlite3_exec(mpDB, szSQL, 0, 0, &szError);

	if (nRet == SQLITE_OK)
	{
		return sqlite3_changes(mpDB);
	}
	else
	{
		const String Error=(char*) szError;
			if(szError)sqlite3_free(szError);
		throw USQLite3Exception(nRet, Error);
	}
}


USQLite3Query USQLite3DB::execQuery(const String szSQL)
{
	checkDB();

	sqlite3_stmt* pVM = compile(szSQL);

	int nRet = sqlite3_step(pVM);

	if (nRet == SQLITE_DONE)
	{
		// no rows
		return USQLite3Query(mpDB, pVM, true/*eof*/);
	}
	else if (nRet == SQLITE_ROW)
	{
		// at least 1 row
		return USQLite3Query(mpDB, pVM, false/*eof*/);
	}
	else
	{
		nRet = sqlite3_finalize(pVM);
		const String szError= sqlite3_errmsg(mpDB);
		throw USQLite3Exception(nRet, szError);
	}
}


int USQLite3DB::execScalar(const String szSQL)
{
	USQLite3Query q = execQuery(szSQL);

	if (q.eof() || q.numFields() < 1)
	{
		throw USQLite3Exception(USQLITE3_ERROR,
								"Invalid scalar query");
	}

	return atoi(q.fieldValue(0));
}


USQLite3Table USQLite3DB::getTable(const String szSQL)
{
	checkDB();

	char* szError;
	char** paszResults=0;
	int nRet;
	int nRows(0);
	int nCols(0);

	nRet = sqlite3_get_table(mpDB, szSQL, &paszResults, &nRows, &nCols, &szError);
    
    
	if (nRet == SQLITE_OK)
	{
		return USQLite3Table(paszResults, nRows, nCols);
	}
	else
	{
		const String Error=(char*) szError;
			if(szError)sqlite3_free(szError);
		throw USQLite3Exception(nRet, Error);
	}
}


sqlite_int64 USQLite3DB::lastRowId()
{
	return sqlite3_last_insert_rowid(mpDB);
}


void USQLite3DB::setBusyTimeout(int nMillisecs)
{
	mnBusyTimeoutMs = nMillisecs;
	sqlite3_busy_timeout(mpDB, mnBusyTimeoutMs);
}


void USQLite3DB::checkDB()
{
	if (!mpDB)
	{
		throw USQLite3Exception(USQLITE3_ERROR,
								"Database not open");
	}
}


sqlite3_stmt* USQLite3DB::compile(const String szSQL)
{
	checkDB();

	String szError;
	const char* szTail=0;
	sqlite3_stmt* pVM;

	int nRet = sqlite3_prepare(mpDB, szSQL, -1, &pVM, &szTail);

	if (nRet != SQLITE_OK)
	{
		throw USQLite3Exception(nRet, szError);
	}

	return pVM;
}


////////////////////////////////////////////////////////////////////////////////

USQLite3Buffer::USQLite3Buffer()
{
  clear();
}


USQLite3Buffer::~USQLite3Buffer()
{

}


void USQLite3Buffer::clear()
{
	if (!mpBuf.IsEmpty())
	{
      mpBuf.Clear();
	}
}


const String USQLite3Buffer::format(const String szFormat, ...)
{
	clear();
	va_list va;
	va_start(va, szFormat);
	mpBuf = sqlite3_vmprintf(szFormat, va);
	va_end(va);
	return mpBuf;
}
