
#include "CascadeDefs.h"

#include "CascadeView.h"

#ifdef WIN32
#include <Visualization/WNT/Window.hxx>
#include <Visualization/Graphic3d/WNTGraphicDevice.hxx>
#else
#include <GL/glx.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/Xmu/StdCmap.h>
#include <X11/Xlib.h>
#include <Visualization/Xw/Window.hxx>
#include <Visualization/Graphic3d/GraphicDevice.hxx>
#endif

//#include <Visualization/V3d/View.hxx>

using namespace Upp;

/////////////////////////////////////////////////////////////////////////////////////////
// Constructor
CascadeView::CascadeView(CascadeDocument *Doc)
{
	// Stores the document
	Document = Doc;
	
	// Resets view pointer
	View.Nullify();
	
	// Resets panning and rotate flag
	isPanning = false;
	isRotating = false;
	
	// Resets Pan start point to a default value
	PanStartPoint.SetNull();
	
} // END Constructor class CascadeView

/////////////////////////////////////////////////////////////////////////////////////////
// Destructor
CascadeView::~CascadeView()
{
	// Resets document handle
	Document = 0;
	
	// Resets view handle
	View.Nullify();
	
} // END Destructor class CascadeView


/////////////////////////////////////////////////////////////////////////////////////////
// Method to choose the correct visual
XVisualInfo *CascadeView::CreateVisual(void)
{
	int visualAttr[] =
	{
		GLX_RGBA,
		GLX_DEPTH_SIZE,		1,
		GLX_RED_SIZE,		1,
		GLX_GREEN_SIZE,		1,
		GLX_BLUE_SIZE,		1,
		GLX_DOUBLEBUFFER,	None
	};
    XVisualInfo *pVisualInfo = ::glXChooseVisual( (Display *)Xdisplay, DefaultScreen((Display *)Xdisplay), visualAttr );
    
    return pVisualInfo;
	
} // END CascadeView::CreateVisual()
		
/////////////////////////////////////////////////////////////////////////////////////////
// Method for attribute setting
void CascadeView::SetAttributes(unsigned long &ValueMask, XSetWindowAttributes &attr)
{
	ValueMask |=
			CWBackPixel
		|	CWBorderPixel
	;
	attr.background_pixel = 0;
	attr.border_pixel = 0;
	
} // END CascadeView::SetAttributes()
		

/////////////////////////////////////////////////////////////////////////////////////////
// GLInit method
void CascadeView::AfterInit(bool isError)
{
	if(isError)
		return;
	
	// Gets the window handle
	Window WindowHandle = GetWindow();

	// Creates the view object
	View = Document->getViewer()->CreateView();
	
	// Creates the OpenCascade window handle
    short lo = (short) WindowHandle;
    short hi = (short) (WindowHandle >> 16);
    Handle(Xw_Window) hWnd = new Xw_Window(Handle(Graphic3d_GraphicDevice)::DownCast(Document->getViewer()->Device()),(int) hi,(int) lo,Xw_WQ_SAMEQUALITY);
//    Handle(Xw_Window) hWnd = new Xw_Window(Document->getGraphicDevice(), WindowHandle);

	// Sets window handle in view
    View->SetWindow(hWnd);
    
    // Maps the view if needed
    if ( !hWnd->IsMapped() )
        hWnd->Map();
    
    // Sets the background color
    View->SetBackgroundColor(Quantity_NOC_BLACK);
    
    // Sets up the triedron
    View->TriedronDisplay(Aspect_TOTP_LEFT_LOWER, Quantity_NOC_WHITE, 0.1, V3d_ZBUFFER);
    View->TriedronEcho(Aspect_TOTE_ORIGIN); // ???

    // Activate the grid
    Document->getViewer()->ActivateGrid(Aspect_GT_Rectangular, Aspect_GDM_Lines);
    Document->getViewer()->SetGridEcho(Standard_True);
    
    // Signals resize needed
    View->MustBeResized();
    
} // END CascadeView::AfterInit()


/////////////////////////////////////////////////////////////////////////////////////////
// These is called just before termination
void CascadeView::BeforeTerminate(void)
{
	
} // END CascadeView::BeforeTerminate()
		

/////////////////////////////////////////////////////////////////////////////////////////
// Paint method
void CascadeView::Paint(Draw &draw)
{
	if(View.IsNull())
		return;
	
Rect r = GetView();
//draw.DrawLine(r.TopLeft(), r.BottomRight());	
//draw.DrawLine(r.TopRight(), r.BottomLeft());	
    View->MustBeResized();
	View->Redraw();
	
} // END CascadeView::Paint()


/////////////////////////////////////////////////////////////////////////////////////////
// Resize method
void CascadeView::Resize(int w, int h)
{
	if(View.IsNull())
		return;
    View->MustBeResized();
    
} // END CascadeView::Resize()


/////////////////////////////////////////////////////////////////////////////////////////
// Fit all in view
void CascadeView::FitAll()
{
	if(View.IsNull())
		return;
    View->FitAll();
    View->ZFitAll();
    View->Redraw();

} // END CascadeView::FitAll()


///////////////////////////////////////////////////////////////////////////////////////////////
// middle up - resets pan/3dorbit behaviour
void CascadeView::MiddleUp(Point p, dword keyflags)
{
	isPanning = false;
	PanStartPoint.SetNull();
	
	isRotating = false;
	
} // END CascadeView::MiddleUp()


///////////////////////////////////////////////////////////////////////////////////////////////
// mouse move handler
void CascadeView::MouseMove(Point p, dword keyflags)
{
	/////////////////// DISPLAY HANDLING (PAN / ROTATE VIEW  /////////////////////
	if(isPanning)
	{
		View->Pan(p.x - PanPrevPoint.x, PanPrevPoint.y - p.y);
		PanPrevPoint = p;
	}
	
	if(isRotating)
		View->Rotation(p.x, p.y);
	
	///////////////////   /////////////////////

} // END CascadeView::MiddleDown()

		
///////////////////////////////////////////////////////////////////////////////////////////////
// middle drag - pan or 3dorbit if in addition to ctrl-shift keys
void CascadeView::MiddleDrag(Point p, dword keyflags)
{
	PanStartPoint = p;
	PanPrevPoint = p;
	
	// if both CTRL and SHIFT keys are pressed, the view is rotated (3dorbit)
	// if not, the pan function is activated
	if( (keyflags & K_SHIFT) && (keyflags & K_CTRL) )
	{
		isRotating = true;
		View->StartRotation(p.x, p.y, 0.4);
	}
	else
		isPanning = true;
	
} // END CascadeView::MiddleDrag()
		

///////////////////////////////////////////////////////////////////////////////////////////////
// mouse wheel - zoom
void CascadeView::MouseWheel(Point p, int zdelta, dword keyflags)
{
	Standard_EXPORT Quantity_Factor	CurScale = View->Scale();
	if(zdelta > 0)
		CurScale *= (1.0 + zdelta / 200.0);
	else
		CurScale /= (1.0 - zdelta / 200.0);
	View->SetScale(CurScale);
	
	View->Redraw();
	
} // END CascadeView::MouseWheel()
		

///////////////////////////////////////////////////////////////////////////////////////////////
// double middle click - zoom extents
void CascadeView::MiddleDouble(Point p, dword keyflags)
{
  View->FitAll();	
  View->Redraw();
	
} // END CascadeView::MiddleDouble()


///////////////////////////////////////////////////////////////////////////////////////////////
// mouse enter handler
void   CascadeView::MouseEnter(Point p, dword keyflags)
{
	// if mouse middle button is pressed on enter, resume panning or rotate
	if(GetMouseMiddle())
	{
		// sets starting and current pan points if needed
		if(PanStartPoint.IsNullInstance())
		{
			PanStartPoint = p;
			PanPrevPoint = p;
		}

		// pans or rotates depending on CTRL-SHIFT keys
		if( (keyflags & K_SHIFT) && (keyflags & K_CTRL) )
		{
			isRotating = true;
			View->StartRotation(p.x, p.y, 0.4);
		}
		else
			isPanning = true;
	}
	
} // END CascadeView::MouseEnter()


///////////////////////////////////////////////////////////////////////////////////////////////
// mouse leave handler
void   CascadeView::MouseLeave()
{
	// resets panning & rotate state on window exit
	isPanning = false;
	isRotating = false;
	
} // END CascadeView::MouseLeave()
