#include "Skips.h"
#include "PatchIncludes.h"

////////////////////////////////////////////////////////////////////////////////////////////////
// DIVIDES A CASCADE FILE NAME LIKE xxxxxx_yyyyy.zzz
// INTO 2 PARTS :
//			aaaaa_bbbbb.zzz ==> aaaaa + bbbbb.zzz
// SPECIAL CASES :
//  1)- NO '_' CHARACTER :
//			aaaaa.zzz ==> aaaaa + aaaaa.zzz
//  2)- PARTS BEFORE AND AFTER UNDERSCORE EQUAL :
//			aaaaa_aaaaa.zzz ==> aaaaa + aaaaa_aaaaa.zzz
//  3)- HANDLE NAMES :
//			Handle_aaaaa_bbbbb.zzz ==> aaaaa + Handle_bbbbb.zzz
// ERROR IF NAME DOESN'T CONTAIN NEITHER '_' NOR '.'
bool SplitCascadeName(String const &Name, String &First, String &Second)
{
	int pos;
	
	// SPECIAL CASE, recfile.xx (FOUND IN StepFile...)
	if(Name.Mid(0, 7) == "recfile")
	{
		First = "StepFile";
		Second = Name;
	}
	// SPECIAL CASE, NO '_'
	else if((pos = Name.Find("_")) < 0)
	{
		if((pos = Name.Find(".")) < 0)
			return false;
		First = Name.Mid(0, pos);
		Second = Name;
	}
	else
	{
		First = Name.Mid(0, pos);
		Second = Name.Mid(pos+1);

		// SPECIAL CASE, Handle_...
		if(First == "Handle")
		{
			String F2, S2;
			SplitCascadeName(Second, F2, S2);
			First = F2;
			Second = "Handle_" + S2;
		}
		
		// SPECIAL CASE, aaaaa_aaaaa.zzz
		if( (pos = Second.Find(".")) >= 0)
		{
			String S2 = Second.Mid(0, pos);
			if(First == S2)
				Second = First + "_" + Second;
		}
	}
	if(First == "" || Second == "" || isspace(First[0]) || isspace(Second[0]))
		return false;
	return true;
	
} // END SplitCascadeName()

/////////////////////////////////////////////////////////////////////////////////////////////////
// TRANSLATES AN INCLUDE OPENCASCADE PATH TO ITS FINAL LOCATION
// IF IT'S NOT AN OPENCASCADE PATH, RETURNS NULL STRING
String TranslateIncludePath(String const &IncPath, String const &CasRoot, const Index<String> &SubPackages, const Array<String> &MainPackages)
{
	String First, Second;
	int iSubPackage;
	
	// splits include path, return null string if can't
	if(!SplitCascadeName(IncPath, First, Second))
		return "";
	
	// checks if it's an opencascade path
	if((iSubPackage = SubPackages.Find(First)) >= 0)
	{
		// should be an opencascade one
		// let's check if file exist and where is it
		// and build correct dest path
		if(FileExists(CasRoot + "/inc/" + IncPath))
			return MainPackages[iSubPackage] + "/" + SubPackages[iSubPackage] + "/" + Second;
		else if(FileExists(CasRoot + "/drv/" + SubPackages[iSubPackage] + "/"  + IncPath))
			return "drv/" + SubPackages[iSubPackage] + "/" + IncPath;
		else if(FileExists(CasRoot + "/src/" + SubPackages[iSubPackage] + "/"  + IncPath))
			return "src/" + SubPackages[iSubPackage] + "/" + IncPath;
		else
			return "";
	}
	else
		return "";

} // END TranslateIncludePath()

/////////////////////////////////////////////////////////////////////////////////////////////////
// PATCHES #include DIRECTIVES INSIDE A FILE
// IT DOES A LIMITATE CHECK ON FILE, SO IT' NOT ERROR-FREE
// IF SOME STRANGE CODE IS USED
String PatchIncludes(const String &Src, const String &CasRoot, const Index<String> &SubPackages, const Array<String> &MainPackages)
{
	// from previous buffer processing, #include are on start of line
	// so processing here is very simple

	String Dest;
	char c;
	int p = 0;
	while(Src[p])
	{
		// test if #include
		if(Src.Mid(p, 8) == "#include")
		{
			// yes, process it
			
			// first, copy the #include statement
			Dest.Cat("#include");
			p+=8;
			
			// then copy/skip all whitespaces/comments
			SkipSpacesAndComments(Src, p, Dest);
			
			// checks if there's a delimiter
			// if not, the include name is a macro, so must not be changed
			char del = Src[p];
			if(del == '<' || del == '"')
			{
				// ok, process the include
				
				// first adds the starting delimiter
				Dest.Cat(del);
				
				// gets last delimiter
				if(del == '<')
					del = '>';
				p++;
				int p2 = p;
				while(Src[p2] && Src[p2] != del)
					p2++;
				
				// gets the include path
				String IncPath = Src.Mid(p, p2-p);
				
				// translates the include path
				if( (IncPath = TranslateIncludePath(IncPath, CasRoot, SubPackages, MainPackages)) != "")
				{
				
					// replaces the translated path
					Dest.Cat(IncPath);
				
					// adds closing delimiter
					Dest.Cat(del);
				
					// advances the pointer
					p = p2+1;
				}
				
			}

		} // end if include
		// test if #define
		else if(Src.Mid(p, 7) == "#define")
		{
			// yes, process it
			
			// first, copy the #define statement
			Dest.Cat("#define");
			p+=7;
			
			// then copy/skip all whitespaces/comments
			SkipSpacesAndComments(Src, p, Dest);
			
			// read the identifier
			while( (c = Src[p]) != 0 && (isalnum(c) || c == '_'))
			{
				Dest.Cat(c);
				p++;
			}
			
			// copy/skip whitespaces, but stops on end of line
			// as macros can be null defines
			if(SkipSpacesAndComments(Src, p, Dest, true))
				continue;
			
			// now, here is the value; it can be an #include define
			// *only* if it's delimited by '"' or '<>'
			char del  = Src[p];
			if(del != '"' && del != '<')
			{
				SkipToEol(Src, p, Dest);
				continue;
			}
			
			// well, it can be a #define include... gets the value
			if(del == '<')
				del = '>';
			p++;
			int p2 = p;
			while(Src[p2] && Src[p2] != del)
				p2++;
			
			// gets the include path
			String IncPath = Src.Mid(p, p2-p);
			
			// translates the include path
			if( (IncPath = TranslateIncludePath(IncPath, CasRoot, SubPackages, MainPackages)) != "")
			{
				Dest.Cat('"');
			
				// replaces the translated path
				Dest.Cat(IncPath);
			
				// adds closing delimiter
				Dest.Cat('"');
			
				// advances the pointer
				p = p2+1;
			}
			else
				// not an include opencascade path
				// go back to include delimiter in copy
				p--;
			
		} // end if define

		// just copy the line
		SkipToEol(Src, p, Dest);

	} // while !end of buffer
	
	// adds a null delimiter as usual
	Dest.Cat(0);
	
	// returns modified buffer
	return Dest;
	
} // END PatchIncludes()
