#include "ReplaceToken.h"
#include "PackSource.h"
#include "RemoveGlobalScopes.h"
#include "AddNamespace.h"
#include "FormatSource.h"
#include "PatchIncludes.h"

#include "OcasReformatter.h"

// OPTIONS PER PACKAGE AND PER FILE
//#define PACKAGE_OPTIONS		"-ffunction-sections  -fPIC -funsigned-char -Wall -fmessage-length=0 -DLIN -DLININTEL -DOCC_CONVERT_SIGNALS  -DNOPROTECTION -DCSFDB -DHAVE_WOK_CONFIG_H  -DHAVE_CONFIG_H -DNDEBUG -DNo_Exception -MD"
//#define CXX_OPTIONS			"-ffriend-injection -fpermissive"
#define PACKAGE_OPTIONS		"-ffunction-sections  -fPIC -funsigned-char -Wall -fmessage-length=0 -DLIN -DLININTEL -DOCC_CONVERT_SIGNALS  -DNOPROTECTION -DCSFDB -DHAVE_WOK_CONFIG_H  -DHAVE_CONFIG_H -DNDEBUG -DNo_Exception -MD -ffriend-injection -fpermissive"
#define CXX_OPTIONS			""
#define C_OPTIONS			""


////////////////////////////////////////////////////////////////////////////////////////////////
// GETS OPENCASCADE ROOT PATH
String GetCasRoot(void)
{
	String RootVar("CASROOT");
	String CasRoot;
	int i = Environment().Find(RootVar);
	if(i >= 0)
		CasRoot = Environment()[i];
	else
		CasRoot = "/opt/OpenCASCADE6.2.0/ros";
	if(!DirectoryExists(CasRoot))
		CasRoot = "";
	return CasRoot;
	
} // END getCasRoot()

////////////////////////////////////////////////////////////////////////////////////////////////
// GETS DIRECTORY CONTENT
Array<String> GetDirectoryContent(const String &Path, bool Folders, bool FullPath = false)
{
	
	// GETS FOLDER CONTENTS
	Array<FileSystemInfo::FileInfo> DirFiles;
	DirFiles = StdFileSystemInfo().Find(Path);
	
	// COPIES ALL NAMES INSIDE AN ARRAY
	Array<String> DirArray;
	for(int i = 0 ; i < DirFiles.GetCount() ; i++)
	{
		if( (Folders && DirFiles[i].is_folder) || (!Folders &&  !DirFiles[i].is_folder) && DirFiles[i].filename != "." && DirFiles[i].filename != "..")
		{
			if(FullPath)
			  DirArray << Path + "/" + DirFiles[i].filename;
			else
			  DirArray << DirFiles[i].filename;
		}
	}
	
	return DirArray;

} // END GetDirectoryContent()

////////////////////////////////////////////////////////////////////////////////////////////////
// GETS ALL THE INCLUDE FILES FROM OPENCASCADE 'inc' DIR
Array<String> GetIncludeNames(const String &CasRoot)
{
	// GETS INCLUDE FOLDER CONTENTS
	Array<String> IncArray;
	IncArray = GetDirectoryContent(CasRoot + "/inc/*.*", false);
	
	return IncArray;

} // END GetIncludeNames()

////////////////////////////////////////////////////////////////////////////////////////////////
// GET ALL FOLDERS FROM OPENCASCADE 'src' DIR
Array<String> GetSourceFolders(const String &CasRoot)
{
	// GETS SOURCE FOLDER CONTENT, ONLY FOLDERS
	Array<String> SourceArray;
	SourceArray = GetDirectoryContent(CasRoot + "/src/*.*", true);
	
	return SourceArray;

} // END GetSourceFolders()

////////////////////////////////////////////////////////////////////////////////////////////////
// GET ALL FOLDERS FROM OPENCASCADE 'drv' DIR
Array<String> GetDrvFolders(const String &CasRoot)
{
	// GETS SOURCE FOLDER CONTENT, ONLY FOLDERS
	Array<String> DrvArray;
	DrvArray = GetDirectoryContent(CasRoot + "/drv/*.*", true);
	
	return DrvArray;

} // END GetDrvFolders()

////////////////////////////////////////////////////////////////////////////////////////////////
// FROM THE ARRAY OF INCLUDE FILE NAMES, GETS THE ARRAY OF SUBPACKAGE NAMES
Array<String> GetSubPackageNames(const Array<String> &IncArray)
{
	Array<String> SubPackageNames;
	String First, Second;
	for(int i = 0 ; i < IncArray.GetCount() ; i++)
	{
		if(!SplitCascadeName(IncArray[i], First, Second))
			continue;
		if(FindIndex(SubPackageNames, First) < 0)
			SubPackageNames << First;
	}
	
	return SubPackageNames;
	
} // END GetSubPackageNames()

////////////////////////////////////////////////////////////////////////////////////////////////
// FROM DEST PATH GETS TEMPLATE FILE NAMES
Array<String> GetTemplateNames(const String &DestRoot)
{
	// GETS ALL TEMPLATE FILE NAMES ON DEST PATH
	// EXTRACT TEMPLATE NAMES FROM FILENAMES
	Array<String> TemplateFileNames = GetDirectoryContent(DestRoot + "/*.template", false);
	String TemplateName;
	Array<String> TemplateNames;
	for(int i = 0 ; i < TemplateFileNames.GetCount() ; i++)
	{
		TemplateName = TemplateFileNames[i];
		int p = TemplateName.Find(".");
		if(p == -1)
			continue;
		TemplateNames << (TemplateName.Mid(0, p));
	}
	
	return TemplateNames;
	
} // END GetTemplateNames()

////////////////////////////////////////////////////////////////////////////////////////////////
// READS A TEMPLATE FILE, RETURNING AN ARRAY OF PACKAGE NAMES
Array<String> ReadTemplate(const String &TemplatePath)
{
	Array<String> TemplateLines;
	String Line;
	
	FileIn f(TemplatePath);
	while(!f.IsEof())
	{
		Line = f.GetLine();
		if(Line == "" || isspace(Line[0]))
			continue;
		TemplateLines << Line;
	}
	
	return TemplateLines;
	
} // END ReadTemplate()

////////////////////////////////////////////////////////////////////////////////////////////////
// CHECKS WHETHER THE NAMED FILE IS A C/C++ SOURCE/INCLUDE FILE
// C/C++ SOURCE FILES ARE OF EXTENSIONS : .c .cpp .cxx .h .hxx .gxx .ixx .jxx
bool IsSourceFile(String const &Name)
{
	String Ext = GetFileExt(Name);
	if(
		Ext == ".c"		||	Ext == ".cpp"	||
		Ext == ".cxx"	||	Ext == ".h"		||
		Ext == ".hxx"	||	Ext == ".gxx"	||
		Ext == ".ixx"	||	Ext == ".jxx"	||
		Ext == ".lxx"	||	Ext == ".pxx"	||
		Ext == ".ph"	||	Ext == ".pc"
	)
		return true;
	return false;

} // END IsSourceFile()

////////////////////////////////////////////////////////////////////////////////////////////////
// CHECKS WHETHER THE NAMED FILE IS A C++ SOURCE/INCLUDE FILE
// C/C++ SOURCE FILES ARE OF EXTENSIONS : .c .cpp .cxx .h .hxx .gxx .ixx .jxx
bool IsCppSourceFile(String const &Name)
{
	String Ext = GetFileExt(Name);
	if(
		Ext == ".cpp"	||	Ext == ".cxx"	||
		Ext == ".h"		||	Ext == ".hxx"	||
		Ext == ".gxx"	||	Ext == ".ixx"	||
		Ext == ".jxx"	||	Ext == ".lxx"	||
		Ext == ".pxx"
	)
		return true;
	return false;

} // END IsCppSourceFile()

////////////////////////////////////////////////////////////////////////////////////////////////
// SCANS A Makefile.am FOR DEPENDENT LIBRARIES
Array<String> GetMakefileLibs(const String &RootPath, const String &Name)
{
	Array<String> res;
	String FilePath = RootPath + "/" + Name + "/Makefile.am";
	String Pattern = "lib" + Name + "_la_LIBADD";
	FileIn f(FilePath);
	String Line;
	while(!f.IsEof())
	{
		Line = f.GetLine();
		if(Line.Find(Pattern) >= 0)
			break;
	}
	if(!f.IsEof())
	{
		int k = Line.Find('=');
		ASSERT(k > 0);
		k++;
		while(k < Line.GetCount() && (isspace(Line[k]) || Line[k] == '\\'))
			k++;
		if(k >= Line.GetCount())
			Line = f.GetLine();
		else
			Line = Line.Mid(k);
		while(!f.IsEof() && Line.Find("../") == 0)
		{
			int k = Line.GetCount()-1;
			while(k >= 3 && Line[k] != '.')
				k--;
			if(k > 3)
			{
				Line = Line.Mid(3, k-3);
				k = 0;
				while(k < Line.GetCount() && Line[k] != '/')
					k++;
				k += 3; // skips 'lib' header
				if(k < Line.GetCount()-1)
					res.Add(Line.Mid(k+1));
			}
			Line = f.GetLine();
		}
	}
	
	return res;
	
} // END GetMakefileLibs()

////////////////////////////////////////////////////////////////////////////////////////////////
// SCANS A Makefile.am FOR SOURCE FILES
Array<String> GetMakefileSources(const String &RootPath, const String &Name)
{
	Array<String> res;
	String FilePath = RootPath + "/" + Name + "/Makefile.am";
	String Pattern = "lib" + Name + "_la_SOURCES";
	FileIn f(FilePath);
	String Line;
	while(!f.IsEof())
	{
		Line = f.GetLine();
		if(Line.Find(Pattern) >= 0)
			break;
	}
	if(!f.IsEof())
	{
		int k = Line.Find('=');
		ASSERT(k > 0);
		k++;
		while(k < Line.GetCount() && (isspace(Line[k]) || Line[k] == '\\'))
			k++;
		if(k >= Line.GetCount())
			Line = f.GetLine();
		else
			Line = Line.Mid(k);

		while(!f.IsEof() && Line.Find("@top_srcdir@/") == 0)
		{
			int k = Line.GetCount()-1;
			while(k >= 13 && (isspace(Line[k]) || Line[k] == '\\'))
				k--;
			if(k > 13)
			{
				Line = Line.Mid(13, k-12);
				res.Add(Line);
			}
			Line = f.GetLine();
		}
	}
	
	return res;
	
} // END GetMakefileSources()


////////////////////////////////////////////////////////////////////////////////////////////////
// CHECKS IF FILE MUST HAVE NAMESPACE INSIDE
bool RequiresNamespace(String const &filename)
{
	char *Excludes[] =
	{
/*
			"config.h",		"Shiftjis.h",	"gb2312.h",			"f2c.h"
		,	"SysBase.h",	"MathBase.h",	"ApproxF2var.h",	"cgmtypes.h"
		,	"cgmstruc.h",	"cgmatt.h"	,	"cgmerr.h",			"cgmout.h"
		,	"cgmobin.h",	"cgmochar.h",	"cgminit.h",		"cgmtypes.h"
		,	"cgmmach.h",	"cgmelem.h",	"cgmpar.h",			"Data_Data.h"
		,	"Triangle.hxx",	"triangle.h",	"Extension.h",		"ImageBox.h"
		,	"tgl_all.h",	"LightBox.h",	"telem.h",			"Cextern.hxx"
		,	"tgl_all.h"
		,	"cmn_varargs.h"
		,	"telem.h"
		,	"cmn_memory.h"
		,	"telem_attri.h"
		,	"tsm.h"
		,	"telem_util.h"
		,	"telem_highlight.h"
		,	"telem_view.h"
		,	"degeneration.h"
		,	"TextureBox.h"
		,	"telem_filters.h"
		,	"telem_inquire.h"
		,	"cmn_htbl.h"
		,	"PrimitiveArray.hxx"
		,	"cmn_stg_tbl.h"
		,	"telem_depthcue.h"
		,	"telem_ws.h"
		,	"tgl.h"
		,	"X11.hxx"
		,	"Aspect.hxx"
		,	"Drawable.hxx"
		,	"Graphic3d.hxx"
		,	"Visual3d.hxx"
		,	"GraphicCallbackProc.hxx"
		,	"Display.hxx"
		,	"RenderingContext.hxx"
		,	"inquire.h"
		,	"tgl_util.h"
		,	"tgl_tox.h"
		,	"tXfm.h"
		,	"animation.h"
		,	"trsf_stack.h"
		,	"tgl_funcs.h"
		,	"telem_pick.h"
		,	"PolygonOffsetMode.hxx"
		,	"transform_persistence.h"
		,	"gl2ps.h"
		,	"tgl_subrvis.h"
		,	"triedron.h"
		,	"tgl_subrs.h"
		,	"tgl_elems.h"
		,	"txgl.h"
		,	"Labels.hxx"
		,	"tgl_pick.h"
		,	"context.h"
		,	"tgl_utilgr.h"
		,	"tgl_vis.h"
		,	"callback.h"
		,	"f2c.h"
		,	"ExprIntrp.tab.h"
		,	"rule.h"
		,	"EDL.tab.h"
		,	"ParseDelivery.h"
		,	"DELIVERY.tab.h"
		,	"regexp.h"
		,	"IDL.tab.h"
		,	"rules.h"
		,	"defines.hxx"
		,	"CDL.tab.h"
		,	"CallFailure.hxx"
		,	"step.tab.h"
		,	"recfile.ph"
		,	"recfile.pc"
		,	"igesread.h"
		,	"GeomToStep_MakeAxis1Placement_gen.pxx"
*/	
			"Cextern.hxx"
	};
	#define nExcludes (sizeof(Excludes) / sizeof(char *))

	String Name = GetFileName(filename);
	
	// no namespaces on *.pxx files... are included in middle of code
	// (once again, awful style of coding....)
	if(Name.Find(".pxx") >= 0)
		return false;

	for(int i = 0 ; i < nExcludes ; i++)
		if(Name == Excludes[i])
			return false;
	if(!IsCppSourceFile(filename))
		return false;
	
	return true;
	
} // END RequiresNamespace()


////////////////////////////////////////////////////////////////////////////////////////////////
// HANDLE SPECIAL CASES
void HandleSpecialCases(const String &FileName, String &Buf)
{
	// files requiring pow() patch
	char *PowPatch[] =
	{
		"Units_Token.cxx",
		"Units_Measurement.cxx",
		"Units_UnitsDictionary.cxx"
	};
	#define nPowPatch (sizeof(PowPatch) / sizeof(char *))
	
	// if required, apply pow() patches
	for(int i = 0 ; i < nPowPatch ; i++)
	{
		if(FileName.Find(PowPatch[i]) >=0)
		{
			ReplaceToken(Buf, "pow(thevalue,", "::pow(thevalue,");
			ReplaceToken(Buf, "pow(themeasurement,", "::pow(themeasurement,");
		}
	}

	// files requiring Tcl patch
	char *TclPatch[] =
	{
		"PInterp.hxx"
	};
	#define nTclPatch (sizeof(TclPatch) / sizeof(char *))
	
	// if required, apply Tcl() patches
	// again, ugly way of coding opencascade....
	for(int i = 0 ; i < nTclPatch ; i++)
	{
		if(FileName.Find(TclPatch[i]) >=0)
		{
			ReplaceToken(Buf, "struct Tcl_Interp;", "\n#include<tcl.h>\n");
		}
	}
	
	// files requiring Edl patch
	char *EdlPatch[] =
	{
		"EDL_Interpretor.cxx"
	};
	#define nEdlPatch (sizeof(EdlPatch) / sizeof(char *))
	
	// if required, apply Edl() patches
	// again, ugly way of coding opencascade....
	for(int i = 0 ; i < nEdlPatch ; i++)
	{
		if(FileName.Find(EdlPatch[i]) >=0)
		{
			ReplaceToken(Buf, ";edl_set_var(varname,astr);", ";::edl_set_var(varname,astr);");
		}
	}
	
} // END HandleSpecialCases()

String __CURFILE__;

////////////////////////////////////////////////////////////////////////////////////////////////
// COPIES THE FILES PATCHING THEM
void CopyPatch(String const &CasRoot, String const &Source, String const &Dest, const Index<String> &SubPackages, const Array<String> &MainPackages)
{
	__CURFILE__ = Source;
	String Buf = LoadFile(Source);
	String SpaceName = "OpenCascade";

	// packs source file
	Buf = PackSource(Buf);

	// remove global scopes from it
	Buf = RemoveGlobalScopes(Buf, SpaceName);
	
	// patches the include directives
	Buf = PatchIncludes(Buf, CasRoot, SubPackages, MainPackages);
	
	// handles special cases
	HandleSpecialCases(Source, Buf);
	
	// adds namespace to file
	if(RequiresNamespace(Dest))
		Buf = AddNamespace(Buf, SpaceName);
	
	// reformats the buffer
//	Buf = FormatSource(Buf);

Buf = Buf.Mid(0, Buf.GetCount()-1);
Buf.Cat('\n');

	SaveFile(Dest, Buf);
	
} // END CopyPatch()

////////////////////////////////////////////////////////////////////////////////////////////////
// CREATES UPP PACKAGES FOR LIBRARY BUILD
bool CreatePackages(const String &CasRoot, const String &DestRoot)
{
	// CREATES MAKEILE ROOT PATH
	String MakeRootPath = CasRoot + "/adm/make";
	
	// CREATE PACKAGES ROOT PATH
	String PackagesRoot = DestRoot + "/OpenCascadeBuild";
	
	// READS IN THE LIBRARY NAMES
	Array<String> LibNames = GetDirectoryContent(MakeRootPath + "/*", true);
	
	// FOR EACH LIBRARY, CREATES THE CORRESPONDING UPP PACKAGE
	for(int iLib = 0 ; iLib < LibNames.GetCount() ; iLib++)
	{
		String LibName = LibNames[iLib];
		Array<String> Uses = GetMakefileLibs(MakeRootPath, LibName);
		Array<String> Sources = GetMakefileSources(MakeRootPath, LibName);
		
		// CREATES THE PACKAGE FILE
		String PackagePath = PackagesRoot + "/" + LibName;
		RealizeDirectory(PackagePath);
		PackagePath = PackagePath + "/" + LibName + ".upp";
		FileOut f(PackagePath);
		
		// TARGET NAME
		f.PutLine("target");
		f.PutLine("\x09lib" + LibName + ".so;");
		f.PutLine("");
		
		// OPTIONS
		f.PutLine("options");
		f.PutLine("\x09\"-DHAVE_CONFIG_H -DHAVE_WOK_CONFIG_H\",");
		f.PutLine("\x09\"-DLIN -DLININTEL\",");
		f.PutLine("\x09\"-DOCC_CONVERT_SIGNALS -DNOPROTECTION -DCSFDB\",");
		f.PutLine("\x09\"-DNDEBUG -DNo_Exception\",");
		f.PutLine("\x09\"-ffriend-injection -fpermissive\",");
		f.PutLine("\x09\"-ffunction-sections -fPIC\",");
		f.PutLine("\x09-fmessage-length=0,");
		f.PutLine("\x09-funsigned-char,");
		f.PutLine("\x09-MD,");
		f.PutLine("\x09-Wall,");
		f.PutLine("\x09\"-Wno-non-virtual-dtor -Wno-cast-qual\",");
		f.PutLine("\x09\"-Wno-unused-variable -Wno-unused-function\";");
		f.PutLine("");
		
		// USED PACKAGES
		if(Uses.GetCount())
			f.PutLine("uses");
		for(int iUse = 0 ; iUse < Uses.GetCount()-1; iUse++)
			f.PutLine("\x09" + Uses[iUse] + ",");
		if(Uses.GetCount())
			f.PutLine("\x09" + Uses[Uses.GetCount()-1] + ";");
		f.PutLine("");
			
		// SOURCE FILES
		if(Sources.GetCount())
			f.PutLine("file");
		for(int iSource = 0 ; iSource < Sources.GetCount()-1; iSource++)
			f.PutLine("\x09../../OpenCascadeSrc/" + Sources[iSource] +",");
		if(Sources.GetCount())
			f.PutLine("\x09../../OpenCascadeSrc/" + Sources[Sources.GetCount()-1] +";");
		f.PutLine("");

		// MAIN CONFIG		
//		f.PutLine("mainconfig");
//		f.PutLine("\x09\"\" = \"DLL \";");
//		f.PutLine("");
		
	} // for iLib
	
	// CREATES DUMMY TOP-LEVEL PACKAGE TO HAVE ALL LIBS BUILT
	String TopPath = PackagesRoot + "/BUILDALL";
	RealizeDirectory(TopPath);
	TopPath = TopPath + "/BUILDALL.upp";
	FileOut f(TopPath);
	
	if(LibNames.GetCount())
		f.PutLine("uses");
	for(int iLib = 0 ; iLib < LibNames.GetCount()-1 ; iLib++)
		f.PutLine("\x09" + LibNames[iLib] + ",");
	if(LibNames.GetCount())
		f.PutLine("\x09" + LibNames[LibNames.GetCount()-1] + ";");
	f.PutLine("");
	
	f.PutLine("mainconfig");	
	f.PutLine("\x09\"\" = \"\"");
	f.PutLine("");
	
} // END CreatePackages()


////////////////////////////////////////////////////////////////////////////////////////////////
// PACKS OPENCASCADE FOR UPP USAGE
bool PackageCascade(const String &CasRoot, const String &DestRoot)
{
	// GETS THE TEMPLATE NAMES FROM DESTINATION DIRECTORY
	// (MUST EXIST AND FILLED WITH INCLUDE FILE NAMES)
	Array<String> TemplateNames = GetTemplateNames(DestRoot);

	// FOR EACH TEMPLATE, EXTRACTS INCLUDED PACKAGE NAMES
	// GETTING THEM FROM INCLUDE FILE NAMES AND MAPS THEM TO TEMPLATE NAME
	Index<String> SubPackages;
	Array<String> MainPackages;
	for(int i = 0 ; i < TemplateNames.GetCount() ; i++)
	{
		// READS THE TEMPLATE FILE
		Array<String> TemplateLines = ReadTemplate(DestRoot + "/" + TemplateNames[i] + ".template");
		
		// EXTRACTS THE PACKAGE NAMES
		Array<String> TemplatePackages = GetSubPackageNames(TemplateLines);
		
		// MAPS THE PACKAGE NAMES WITH CORRESPONDING TEMPLATE
		// AND FILLS THE GLOBAL PACKAGES ARRAY FROM TEMPLATES
		for(int j = 0 ; j < TemplatePackages.GetCount() ; j++)
		{
			SubPackages.Add(TemplatePackages[j]);
			MainPackages.Add(TemplateNames[i]);
		}
	}
	
	// HERE WE MUST DEAL WITH INCLUDE FILES THAT ARE NOT INCLUDED IN PACKAGES
	// INSIDE THE TEMPLATES; SHOULD NOT HAPPEN, BUT... WE MAP THEM TO 'Others' TEMPLATE
	
	// FIRST, WE READ THE INCLUDE FILES....
	Array<String> IncludeFiles = GetIncludeNames(CasRoot);
	
	// ...FROM THEM WE EXTRACT A PACKAGE NAMES LIST....
	Array<String> PackagesFromIncludes = GetSubPackageNames(IncludeFiles);
	
	// AND WE LOOK FOR PACKAGES NOT INCLUDED IN TEMPLATES
	// MAPPING THEM TO 'Others' CATEGORY
	bool FoundOrphan = false;
	for(int i = 0 ; i < PackagesFromIncludes.GetCount() ; i++)
	{
		if(SubPackages.Find(PackagesFromIncludes[i]) < 0)
		{
			SubPackages.Add(PackagesFromIncludes[i]);
			MainPackages.Add("Others");
			FoundOrphan = true;
		}
	}
	if(FoundOrphan)
		TemplateNames << "Others";
	
	////////////////////////////////////////////////////////////////////////////////////////////
	// HERE WE HAVE :
	//  All template (category) names in TemplateNames
	//  2 Arrays mapping  Package names with corresponding Templates in SubPackages+MainPackages
	//  the complete array of include file names in IncludeFiles
	//
	// We must now create package folders on destination path
	// and copy include files there, changing #include directives if necessary

	// NOW WE MUST COPY ALL INCLUDE FILES, PATCHING THE INCLUDE DIRECTIVES
	// WE PUT THEM ON CHILD OF OpenCascade FOLDER IN ROOT DEST
	for(int i = 0 ; i < IncludeFiles.GetCount() ; i++)
	{
		String First, Second;
		if(!SplitCascadeName(IncludeFiles[i], First, Second))
		  continue;
		String Source = CasRoot + "/inc/" + IncludeFiles[i];
		int iSubPackage = SubPackages.Find(First);
		String MainPackage = MainPackages[iSubPackage];
		String DestPath = DestRoot + "/OpenCascade/" + MainPackage + "/" + First;
		RealizeDirectory(DestPath);
		String Dest = DestPath + "/" + Second;
		CopyPatch(CasRoot, Source, Dest, SubPackages, MainPackages);
	}

	//////////////////////////////////////////////////////////////////////////////////////////////
	// NOW WE MUST DEAL WITH SOURCE FILES
	// WE PUT THEM IN CHILD OF OpenCascadeSrc/src FOLDER IN ROOT DEST
	
	// AT FIRST, WE GET A LIST OF SOURCE FILE FOLDERS....
	Array<String> SourceFolders = GetSourceFolders(CasRoot);
	
	// NOW WE PROCESS EACH SOURCE FOLDER
	for(int i = 0 ; i < SourceFolders.GetCount() ; i++)
	{
		// WE CREATE THE SOURCE FULL PATH
		String SourcePath = CasRoot + "/src/" + SourceFolders[i] + "/";
		
		// WE CREATE THE DEST FULL PATH AND THE FOLDER, IF NOT THERE
		String DestPath = DestRoot + "/OpenCascadeSrc/src/" + SourceFolders[i] + "/";
		RealizeDirectory(DestPath);
		
		// WE'RE DONE WITH FOLDERS, NOW WE DEAL WITH FILES ON EACH FOLDER
		Array<String> SourceFiles = GetDirectoryContent(CasRoot + "/src/" + SourceFolders[i] + "/*.*", false);
		for(int j = 0 ; j < SourceFiles.GetCount() ; j++)
		{
			// WE GET THE FILENAME
			String SourceFile = SourceFiles[j];
			
			// SKIP NON-SOURCE FILES
			if(!IsSourceFile(SourceFile))
				continue;
			
			// WE CREATE SOURCE AND DEST FULL PATHS AND COPY/PATCH THE FILE
			String Source = SourcePath + SourceFile;
			String Dest = DestPath + SourceFile;
			CopyPatch(CasRoot, Source, Dest, SubPackages, MainPackages);
						
		}
	}
	
	//////////////////////////////////////////////////////////////////////////////////////////////
	// NOW WE MUST DEAL WITH DRV FILES
	// WE PUT THEM IN CHILD OF OpenCascadeSrc/drv FOLDER IN ROOT DEST
	
	// AT FIRST, WE GET A LIST OF DRV FILE FOLDERS....
	Array<String> DrvFolders = GetDrvFolders(CasRoot);
	
	// NOW WE PROCESS EACH DRV FOLDER
	for(int i = 0 ; i < DrvFolders.GetCount() ; i++)
	{
		// WE CREATE THE SOURCE FULL PATH
		String SourcePath = CasRoot + "/drv/" + DrvFolders[i] + "/";
		
		// WE CREATE THE DEST FULL PATH AND THE FOLDER, IF NOT THERE
		String DestPath = DestRoot + "/OpenCascadeSrc/drv/" + SourceFolders[i] + "/";
		RealizeDirectory(DestPath);
		
		// WE'RE DONE WITH FOLDERS, NOW WE DEAL WITH FILES ON EACH FOLDER
		Array<String> DrvFiles = GetDirectoryContent(CasRoot + "/drv/" + DrvFolders[i] + "/*.*", false);
		for(int j = 0 ; j < DrvFiles.GetCount() ; j++)
		{
			// WE GET THE FILENAME
			String DrvFile = DrvFiles[j];
			
			// SKIP NON-SOURCE FILES
			if(!IsSourceFile(DrvFile))
				continue;
			
			// WE CREATE SOURCE AND DEST FULL PATHS AND COPY/PATCH THE FILE
			String Source = SourcePath + DrvFile;
			String Dest = DestPath + DrvFile;
			CopyPatch(CasRoot, Source, Dest, SubPackages, MainPackages);
		}
	}

	return true;

} // END PackageCascade()

CONSOLE_APP_MAIN
{
	int argc = CommandLine().GetCount();
	const Vector<String>& argv = CommandLine();
	String CasRoot;

/*
String Source = "/opt/OpenCASCADE6.2.0/ros/src/V3d/V3d_View.cxx";
String Dest = "/home/massimo/xxxxx.cxx";

	String Buf = LoadFile(Source);
	Buf = PackSource(Buf);
	Buf = AddNamespace(Buf, "OpenCascade");
//	Buf = FormatSource(Buf);
	SaveFile(Dest, Buf);
	SetExitCode(0);
	exit(0);
*/
	if(argc == 1)
	{
		// CHECKS FOR DESTINATION FOLDER
		if(!FileSystemInfo().FolderExists(argv[0]))
		{
			Cout() << "Destination folder '" << argv[0] << "' don't exist";
			exit(1);
		}
		
		// CHECKS FOR EXISTING OPENCASCADE
		CasRoot = GetCasRoot();
		if(CasRoot == "")
		{
			Cout() << "OpenCASCADE folder not found\nPlease check OpenCASCADE setup\n";
			exit(1);
		}

		PackageCascade(CasRoot, argv[0]);
		
		CreatePackages(CasRoot, argv[0]);
		
		SetExitCode(0);
	}
	else
	{
		Cout() << "USAGE : PackCascade <dest root path>\n";
		exit(1);
	}
}

