#include "Skips.h"
#include "PackSource.h"

#define		SKIP_EOL			0x01
#define		LEAVE_ONE_SPACE		0x02

// reads next char -- checks for line continuation
inline char ReadChar(String const &Src, int &p)
{
	if(Src[p] == '\\' && isspace(Src[p+1]))
	{
		int p2 = p+1;
		while(Src[p2] && isspace(Src[p2]) && Src[p2] != '\n')
			p2++;
		if(Src[p2] == '\n')
		{
			p2++;
			char c = ReadChar(Src, p2);
			p = p2;
			return c;
		}
	}
	return Src[p];
	
} // END ReadChar()

// reads spaces skipping them
bool ReadSpaces(String const &Src, int &p, String &Dest, int flags = 0)
{
	// gets next char
	char c = ReadChar(Src, p);
	
	// if no spaces, just return
	if(!isspace(c))
		return false;
	
	// at least one space found, if needed, append a space to dest
	if(flags & LEAVE_ONE_SPACE)
		Dest.Cat(' ');
	
	// now skip remaining spaces
	while(c && isspace(c) && (c != '\n' || (flags & SKIP_EOL)))
	{
		p++;
		c = ReadChar(Src, p);
	}
	
	if(c == '\n')
	{
		p++;
		return true;
	}
	else
		return false;
	
} // END ReadSpaces()

// Reads a using directive
void ReadUsing(String const &Src, int &p, String &Dest)
{
	// If not at start of file, adds a newline before using
	if(p != 0 && Dest.GetCount() && Dest[Dest.GetCount()-1] != '\n')
		Dest.Cat('\n');
	
	// Just copy all directive till terminating ';', eventually putting multiline directive
	// on a single line
	char c;
	while( (c = Src[p]) != 0)
	{
		if(c == '\n')
		{
			p++;
			Dest.Cat(' ');
			continue;
		}
		Dest.Cat(c);
		p++;
		if(c == ';')
			break;
	}
	Dest.Cat('\n');
	
} // END ReadUsing()

// Reads an extern directive
void ReadExtern(String const &Src, int &p, String &Dest)
{
	// If not at start of file, adds a newline before extern
	if(p != 0 && Dest.GetCount() && Dest[Dest.GetCount()-1] != '\n')
		Dest.Cat('\n');
	
	// Just copy all directive till terminating ';'
	char c;
	while( (c = Src[p]) != 0)
	{
		if(c == '\n')
		{
			p++;
			Dest.Cat(' ');
			continue;
		}
		Dest.Cat(c);
		p++;
		if(c == ';')
			break;
	}
	Dest.Cat('\n');
	
} // END ReadExtern()

// Reads a DEFINE_XXXX macro
void ReadDEFINE_(String const &Src, int &p, String &Dest)
{
	// If not at start of file, adds a newline before DEFINE_xxxx
	if(p != 0 && Dest.GetCount() && Dest[Dest.GetCount()-1] != '\n')
		Dest.Cat('\n');
	
	// copies chars till eol
	char c;
	while( (c = Src[p]) != 0)
	{
		Dest.Cat(c);
		p++;
		if(c == '\n')
			break;
	}
	
} // END ReadDEFINE_()

// read c++ comment
void ReadCppComment(String const &Src, int &p, String &Dest)
{
	// tries to figure if comment was at start of line
	// in that case, keeps it there whithout eating whitespaces
	int p2 = p-1;
	while(p2 >= 0 && isspace(Src[p2]) && Src[p2] != '\n')
		p2--;
	if(p2 >= 0 && Src[p2] == '\n')
	{
		p = p2+1;
		if(Dest.GetCount() && Dest[Dest.GetCount()-1] != '\n')
			Dest.Cat('\n');
	}
	
	// Copies till end of line
	char c;
	while( (c = Src[p]) != 0)
	{
		Dest.Cat(c);
		p++;
		if(c == '\n')
			break;
	}
	
} // END ReadCppComment()

// read c comment
void ReadCComment(String const &Src, int &p, String &Dest, bool InsidePreproc = false)
{
	// tries to figure if comment was at start of line
	// in that case, keeps it there whithout eating whitespaces
	if(!InsidePreproc)
	{
		int p2 = p-1;
		while(p2 >= 0 && isspace(Src[p2]) && Src[p2] != '\n')
			p2--;
		if(p2 >= 0 && Src[p2] == '\n')
		{
			if(Dest.GetCount() && Dest[Dest.GetCount()-1] != '\n')
				Dest.Cat('\n');
			Dest.Cat(Src.Mid(p2, p-p2-1));
		}
	}
	
	// Copies comment till closing, checking also for nested comments
	Dest.Cat(Src.Mid(p, 2));
	p+=2;
	char c;
	while( (c = Src[p]) != 0)
	{
		// Check for nested comment
//		if(c == '/' && Src[p+1] == '*')
//			ReadCComment(Src, p, Dest);
//		else
		if(c == '*' && Src[p+1] == '/')
		{
			Dest.Cat(Src.Mid(p, 2));
			p+=2;
			break;
		}
		else
		{
			Dest.Cat(c);
			p++;
		}
	}
	
} // END ReadCComment()
		
// read identifier, returns true if stopped on eol
bool ReadIdentifier(String const &Src, int &p, String &Dest, bool SkipEol = true)
{
	char c;
	while( (c = Src[p]) != 0 && (isalnum(c) || c == '_'))
	{
		Dest.Cat(c);
		p++;
	}
	
	// Skip Spaces
	int flags = (SkipEol ? SKIP_EOL : 0);
	if(ReadSpaces(Src, p, Dest, flags))
	{
		// here only when must not skip eol
		return true;
	}
	
	// If another identifier follows, adds a whitespace to separate them
	c = Src[p];
	if(isalnum(c) || c == '_')
		Dest.Cat(' ');
	
	return false;
	
} // END ReadIdentifier()

// Reads preprocessor directive
void ReadPreproc(String const &Src, int &p, String &Dest)
{
	char c;

	// If not at start of file, adds a newline before preproc
	if(p != 0 && Dest.GetCount() && Dest[Dest.GetCount()-1] != '\n')
		Dest.Cat('\n');
	
	// copies the starting '#'
	Dest.Cat('#');
	p++;
	
	// skip whitespaces (some bad behaviour... write # define instead #define
	// returns if eol found
	if(ReadSpaces(Src, p, Dest))
	{
		Dest.Cat('\n');
		return;
	}
	
	// reads the preproc directive itself
	while( (c = ReadChar(Src, p)) != 0 && (isalnum(c) || c == '_'))
	{
		Dest.Cat(c);
		p++;
	}
	
	// skips spaces (leaves 1)
	if(ReadSpaces(Src, p, Dest, LEAVE_ONE_SPACE))
	{
		Dest.Cat('\n');
		return;
	}
	
	// reads directive first arg
	while((c = ReadChar(Src, p)) != 0 && (isalnum(c) || c == '_'))
	{
		Dest.Cat(c);
		p++;
	}
	
	// takes in the first space (if any)
	// then skips remaining spaces
	if(ReadSpaces(Src, p, Dest, LEAVE_ONE_SPACE))
	{
		Dest.Cat('\n');
		return;
	}

	// copy the remaining of directive till eol, eventually putting multiline directive
	// on a single line
	while( 1 )
	{
		// eats whitespaces
		if(ReadSpaces(Src, p, Dest))
		{
			Dest.Cat('\n');
			return;
		}
		
		// gets next char
		if( (c = ReadChar(Src, p)) == 0)
			break;
		
		// don't skip space in (rare) case of '. ASCII'
		if(c == '.' && isspace(Src[p+1]))
		{
			int p2 = p+1;
			ReadSpaces(Src, p2, Dest);
			if(isalpha(Src[p2]))
			{
				Dest.Cat(". ");
				p = p2;
			}
			else
			{
				Dest.Cat(c);
				p++;
			}
			continue;
		}

		// don't skip space in (rare) case of '/ *'
		else if(c == '/' && isspace(Src[p+1]))
		{
			int p2 = p+1;
			ReadSpaces(Src, p2, Dest);
			if(Src[p2] == '*')
			{
				Dest.Cat("/ *");
				p = p2+1;
			}
			else
			{
				Dest.Cat(c);
				p++;
			}
			continue;
		}
		
		// don't skip spaces in rare case of ': ::'
		else if(c == ':' && isspace(Src[p+1]))
		{
			int p2 = p+1;
			ReadSpaces(Src, p2, Dest);
			if(Src[p2] == ':' && Src[p2+1] == ':')
			{
				Dest.Cat(": ::");
				p = p2+2;
			}
			else
			{
				Dest.Cat(c);
				p++;
			}
			continue;
		}
		
		// Checks for comments
		else if(c == '/' && Src[p+1] == '/')
		{
			ReadCppComment(Src, p, Dest);
			Dest.Cat('\n');
			break;
		}
		
		else if(c == '/' && Src[p+1] == '*')
		{
			ReadCComment(Src, p, Dest, true);
			continue;
		}
		
		// Checks for strings
		else if(c == '"')
		{
			SkipString(Src, p, Dest);
			continue;
		}
		
		// checks for single char (avoids dropping space inside
		else if(c == '\'')
		{
			SkipSingleChar(Src, p, Dest);
			continue;
		}
		
		// Checks for identifier
		else if(isalnum(c) || c == '_')
		{
			if(ReadIdentifier(Src, p, Dest, false))
			{
				Dest.Cat('\n');
				return;
			}
			continue;
		}
		
		// just add char, if none of the above
		else
		{
			Dest.Cat(c);
			p++;
		}

	}
	
} // END ReadPreproc()

// Packs source file - removing all unneeded whitespaces, cr, etc
// Preprocessor directives are put on start of lines, in a single line
// comments and strings are left unchanged
String PackSource(String &Src)
{
	// At first, adds \0 terminator to buffer
	// (make simpler buffer handling)
	Src.Cat(0);
	
	// Initializes dest buffer
	String Dest;
	
	// Initializes buffer pointer
	int p = 0;
	
	// Packing loop
	char c;
//	char prevc = 0;
	while(1)
	{
		ReadSpaces(Src, p, Dest, SKIP_EOL);
		if( (c = ReadChar(Src, p)) == 0)
			break;

		// don't skip space in (rare) case of '. ASCII'
		if(c == '.' && isspace(Src[p+1]))
		{
			int p2 = p+1;
			ReadSpaces(Src, p2, Dest, SKIP_EOL);
			if(isalpha(Src[p2]))
			{
				Dest.Cat(". ");
				p = p2;
			}
			else
			{
				Dest.Cat(c);
				p++;
			}
			continue;
		}

		// don't skip space in (rare) case of '/ *'
		else if(c == '/' && isspace(Src[p+1]))
		{
			int p2 = p+1;
			ReadSpaces(Src, p2, Dest, SKIP_EOL);
			if(Src[p2] == '*')
			{
				Dest.Cat("/ *");
				p = p2+1;
			}
			else
			{
				Dest.Cat(c);
				p++;
			}
			continue;
		}
		
		// don't skip spaces in rare case of ': ::'
		else if(c == ':' && isspace(Src[p+1]))
		{
			int p2 = p+1;
			ReadSpaces(Src, p2, Dest, SKIP_EOL);
			if(Src[p2] == ':' && Src[p2+1] == ':')
			{
				Dest.Cat(": ::");
				p = p2+2;
			}
			else
			{
				Dest.Cat(c);
				p++;
			}
			continue;
		}
		
		// Checks for preprocessor directives
		else if(c == '#')
		{
			ReadPreproc(Src, p, Dest);
			continue;
		}
		
		// Checks for comments
		else if(c == '/' && Src[p+1] == '/')
		{
			ReadCppComment(Src, p, Dest);
			continue;
		}
		else if(c == '/' && Src[p+1] == '*')
		{
			ReadCComment(Src, p, Dest);
			continue;
		}
		
		// Checks for strings
		else if(c == '"')
		{
			SkipString(Src, p, Dest);
			continue;
		}
		
		// checks for single char (avoids dropping space inside
		else if(c == '\'')
		{
			SkipSingleChar(Src, p, Dest);
			continue;
		}
		
		// checks for using directive
		else if(Src.Mid(p, 5) == "using" && isspace(Src[p+5]))
		{
			ReadUsing(Src, p, Dest);
			continue;
		}
/*		
		// checks for extern directive
		else if(Src.Mid(p, 6) == "extern" && isspace(Src[p+6]))
			ReadExtern(Src, p, Dest);
*/
		
		// checks for DEFINE_xxxx macro
		else if(Src.Mid(p, 7) == "DEFINE_")
		{
			ReadDEFINE_(Src, p, Dest);
			continue;
		}
		
		// Checks for identifier
		else if(isalnum(c) || c == '_')
		{
			ReadIdentifier(Src, p, Dest);
			continue;
		}
		
		// Else, just copy character to dest
		else
		{
			Dest.Cat(c);
			p++;
		}
	} // END Packing loop
	
	// Add \0 terminator to buffer
	// (make simpler buffer handling)
	Dest.Cat(0);
	
	return Dest;
	
} // END PackSource()
