#include	<Core/Core.h>

#include "LibCascade.h"

#define USE_NAMESPACE

// OPTIONS PER PACKAGE AND PER FILE
//#define PACKAGE_OPTIONS		"-ffunction-sections  -fPIC -funsigned-char -Wall -fmessage-length=0 -DLIN -DLININTEL -DOCC_CONVERT_SIGNALS  -DNOPROTECTION -DCSFDB -DHAVE_WOK_CONFIG_H  -DHAVE_CONFIG_H -DNDEBUG -DNo_Exception -MD"
//#define CXX_OPTIONS			"-ffriend-injection -fpermissive"
#define PACKAGE_OPTIONS		"-ffunction-sections  -fPIC -funsigned-char -Wall -fmessage-length=0 -DLIN -DLININTEL -DOCC_CONVERT_SIGNALS  -DNOPROTECTION -DCSFDB -DHAVE_WOK_CONFIG_H  -DHAVE_CONFIG_H -DNDEBUG -DNo_Exception -MD -ffriend-injection -fpermissive"
#define CXX_OPTIONS			""
#define C_OPTIONS			""


////////////////////////////////////////////////////////////////////////////////////////////////
// GETS OPENCASCADE ROOT PATH
String GetCasRoot(void)
{
	String RootVar("CASROOT");
	String CasRoot;
	int i = Environment().Find(RootVar);
	if(i >= 0)
		CasRoot = Environment()[i];
	else
		CasRoot = "/opt/OpenCASCADE6.2.0/ros";
	if(!DirectoryExists(CasRoot))
		CasRoot = "";
	return CasRoot;
	
} // END getCasRoot()

////////////////////////////////////////////////////////////////////////////////////////////////
// DIVIDES A CASCADE FILE NAME LIKE xxxxxx_yyyyy.zzz
// INTO 2 PARTS :
//			aaaaa_bbbbb.zzz ==> aaaaa + bbbbb.zzz
// SPECIAL CASES :
//  1)- NO '_' CHARACTER :
//			aaaaa.zzz ==> aaaaa + aaaaa.zzz
//  2)- PARTS BEFORE AND AFTER UNDERSCORE EQUAL :
//			aaaaa_aaaaa.zzz ==> aaaaa + aaaaa_aaaaa.zzz
//  3)- HANDLE NAMES :
//			Handle_aaaaa_bbbbb.zzz ==> aaaaa + Handle_bbbbb.zzz
// ERROR IF NAME DOESN'T CONTAIN NEITHER '_' NOR '.'
bool SplitCascadeName(String const &Name, String &First, String &Second)
{
	int pos;
	
	// SPECIAL CASE, NO '_'
	if((pos = Name.Find("_")) < 0)
	{
		if((pos = Name.Find(".")) < 0)
			return false;
		First = Name.Mid(0, pos);
		Second = Name;
	}
	else
	{
		First = Name.Mid(0, pos);
		Second = Name.Mid(pos+1);

		// SPECIAL CASE, Handle_...
		if(First == "Handle")
		{
			String F2, S2;
			SplitCascadeName(Second, F2, S2);
			First = F2;
			Second = "Handle_" + S2;
		}
		
		// SPECIAL CASE, aaaaa_aaaaa.zzz
		if( (pos = Second.Find(".")) >= 0)
		{
			String S2 = Second.Mid(0, pos);
			if(First == S2)
				Second = First + "_" + Second;
		}
	}
	if(First == "" || Second == "" || isspace(First[0]) || isspace(Second[0]))
		return false;
	return true;
	
} // END SplitCascadeName()

////////////////////////////////////////////////////////////////////////////////////////////////
// GETS DIRECTORY CONTENT
Array<String> GetDirectoryContent(const String &Path, bool Folders, bool FullPath = false)
{
	
	// GETS FOLDER CONTENTS
	Array<FileSystemInfo::FileInfo> DirFiles;
	DirFiles = StdFileSystemInfo().Find(Path);
	
	// COPIES ALL NAMES INSIDE AN ARRAY
	Array<String> DirArray;
	for(int i = 0 ; i < DirFiles.GetCount() ; i++)
	{
		if( (Folders && DirFiles[i].is_folder) || (!Folders &&  !DirFiles[i].is_folder) && DirFiles[i].filename != "." && DirFiles[i].filename != "..")
		{
			if(FullPath)
			  DirArray << Path + "/" + DirFiles[i].filename;
			else
			  DirArray << DirFiles[i].filename;
		}
	}
	
	return DirArray;

} // END GetDirectoryContent()

////////////////////////////////////////////////////////////////////////////////////////////////
// GETS ALL THE INCLUDE FILES FROM OPENCASCADE 'inc' DIR
Array<String> GetIncludeNames(const String &CasRoot)
{
	// GETS INCLUDE FOLDER CONTENTS
	Array<String> IncArray;
	IncArray = GetDirectoryContent(CasRoot + "/inc/*.*", false);
	
	return IncArray;

} // END GetIncludeNames()

////////////////////////////////////////////////////////////////////////////////////////////////
// GET ALL FOLDERS FROM OPENCASCADE 'src' DIR
Array<String> GetSourceFolders(const String &CasRoot)
{
	// GETS SOURCE FOLDER CONTENT, ONLY FOLDERS
	Array<String> SourceArray;
	SourceArray = GetDirectoryContent(CasRoot + "/src/*.*", true);
	
	return SourceArray;

} // END GetSourceFolders()

////////////////////////////////////////////////////////////////////////////////////////////////
// GET ALL FOLDERS FROM OPENCASCADE 'drv' DIR
Array<String> GetDrvFolders(const String &CasRoot)
{
	// GETS SOURCE FOLDER CONTENT, ONLY FOLDERS
	Array<String> DrvArray;
	DrvArray = GetDirectoryContent(CasRoot + "/drv/*.*", true);
	
	return DrvArray;

} // END GetDrvFolders()

////////////////////////////////////////////////////////////////////////////////////////////////
// FROM THE ARRAY OF INCLUDE FILE NAMES, GETS THE ARRAY OF SUBPACKAGE NAMES
Array<String> GetSubPackageNames(const Array<String> &IncArray)
{
	Array<String> SubPackageNames;
	String First, Second;
	for(int i = 0 ; i < IncArray.GetCount() ; i++)
	{
		if(!SplitCascadeName(IncArray[i], First, Second))
			continue;
		if(FindIndex(SubPackageNames, First) < 0)
			SubPackageNames << First;
	}
	
	return SubPackageNames;
	
} // END GetSubPackageNames()

////////////////////////////////////////////////////////////////////////////////////////////////
// FROM DEST PATH GETS TEMPLATE FILE NAMES
Array<String> GetTemplateNames(const String &DestRoot)
{
	// GETS ALL TEMPLATE FILE NAMES ON DEST PATH
	// EXTRACT TEMPLATE NAMES FROM FILENAMES
	Array<String> TemplateFileNames = GetDirectoryContent(DestRoot + "/*.template", false);
	String TemplateName;
	Array<String> TemplateNames;
	for(int i = 0 ; i < TemplateFileNames.GetCount() ; i++)
	{
		TemplateName = TemplateFileNames[i];
		int p = TemplateName.Find(".");
		if(p == -1)
			continue;
		TemplateNames << (TemplateName.Mid(0, p));
	}
	
	return TemplateNames;
	
} // END GetTemplateNames()

////////////////////////////////////////////////////////////////////////////////////////////////
// READS A TEMPLATE FILE, RETURNING AN ARRAY OF PACKAGE NAMES
Array<String> ReadTemplate(const String &TemplatePath)
{
	Array<String> TemplateLines;
	String Line;
	
	FileIn f(TemplatePath);
	while(!f.IsEof())
	{
		Line = f.GetLine();
		if(Line == "" || isspace(Line[0]))
			continue;
		TemplateLines << Line;
	}
	
	return TemplateLines;
	
} // END ReadTemplate()

/*
////////////////////////////////////////////////////////////////////////////////////////////////
// THIS FUNCTIONS ADDS A #define TO AN OPENED FILE
void AddDefine(FileOut &fOut, String const &Name, String const &Value)
{
	fOut.PutLine("#ifndef " + Name);
	fOut.PutLine("#define " + Name + " " + Value);
	fOut.PutLine("#endif");
	             
} // END AddDefine()
*/

////////////////////////////////////////////////////////////////////////////////////////////////
// CHECKS WHETHER THE NAMED FILE IS A C/C++ SOURCE/INCLUDE FILE
// C/C++ SOURCE FILES ARE OF EXTENSIONS : .c .cpp .cxx .h .hxx .gxx .ixx .jxx
bool IsSourceFile(String const &Name)
{
	String Ext = GetFileExt(Name);
	if(
		Ext == ".c"		||	Ext == ".cpp"	||
		Ext == ".cxx"	||	Ext == ".h"		||
		Ext == ".hxx"	||	Ext == ".gxx"	||
		Ext == ".ixx"	||	Ext == ".jxx"	||
		Ext == ".pxx"
	)
		return true;
	return false;

} // END IsSourceFile()


////////////////////////////////////////////////////////////////////////////////////////////////
// REPLACES A TOKEN INSIDE A LINE
void ReplaceToken(String &Line, String const &OldToken, String const &NewToken)
{
	int Start = 0;
	bool fin = false;
	int k=1;
	while(k >= 0)
	{
		k = Line.Find(OldToken, Start);
		if( k < 0 && OldToken[0] == ' ')
		{
			String MyToken = "\x09" + OldToken.Mid(1);
			k = Line.Find(MyToken, Start);
			if(k >= 0)
			{
				MyToken = NewToken;
				if(isspace(OldToken[0]))
				   MyToken.Set(0, OldToken[0]);
				Line = Line.Mid(0, k) + MyToken + Line.Mid(k + MyToken.GetCount());
				Start = k + MyToken.GetCount();
			}
		}
		else if(k >= 0)
		{
			Line = Line.Mid(0, k) + NewToken + Line.Mid(k + OldToken.GetCount());
			Start = k + OldToken.GetCount();
		}
	}

} // END ReplaceToken()

////////////////////////////////////////////////////////////////////////////////////////////////
// INSERT cas:: IN PLACE OF ::
void AddNameSpace(String &Line)
{
	String L = Line;
	if(L.Find("HASHCODE") >= 0)
	   L = Line;
	
	int Start = 0;
	int k=0;
	while(k >= 0 && Start < Line.GetCount())
	{
		// CERCA L' INIZIO DEL' MARKER DI SCOPO (::)
		k = Line.Find("::", Start);
		if(k < 0)
			return;

		// MEMORIZZA LA POSIZIONE DEL MARKER		
		Start = k;
		// VA A RITROSO, PER VEDERE CHE C'E' PRIMA, SALTANDO GLI SPAZI
		while(k > 0 && isspace(Line[k-1]))
			k--;
		
		// HA TROVATO UN CARATTERE NON NULLO; ORA, PER PRIMA COSA,
		// CONTROLLA IL CASO PARTICOLARE DI 'return', POI SE C'E'
		// UN CARATTERE NON ALFANUMERICO O IL ')'
		// IN ENTREMBI I CASI, DEVE SOSTITUIRE....
		if(
			(k >= 6 && Line.Mid(k-6,6) == "return") ||
			(k >= 4 && Line.Mid(k-4,4) == "else") ||
			k == 0 ||
			(k > 0 && (!isalnum(Line[k-1]) && Line[k-1] != ')' && Line[k-1] != '>'))
		)
		{
			Line = Line.Mid(0, k) + " ocf" + Line.Mid(Start);
			Start += 5;
			continue;
		}

		// NON SOSTITUITO, SALTA E PROSEGUE RICERCA..
		Start += 2;
	}

} // END AddNameSpace()


////////////////////////////////////////////////////////////////////////////////////////////////
// START AND END OF NAMESPACE BLOCK
void StartNameSpace(FileOut &f)
{
	f.PutLine("#ifdef __cplusplus");
	f.PutLine("namespace ocf {");
	f.PutLine("#endif");
	
} // END StartNameSpace()

void EndNameSpace(FileOut &f)
{
	f.PutLine("#ifdef __cplusplus");
	f.PutLine("} // namespace ocf");
	f.PutLine("#endif");
	
} // END EndNameSpace()


////////////////////////////////////////////////////////////////////////////////////////////////
// COPIES A FILE PATCHING #INCLUDE DIRECTIVES INSIDE
// IT DOES A LIMITATE CHECK ON FILE, SO IT' NOT ERROR-FREE
// IF SOME STRANGE CODE IS USED
bool CopyPatch(const String &CasRoot, const String &Source, const String &Dest, const Index<String> &SubPackages, const Array<String> &MainPackages)
{
	FileIn fIn(Source);
	FileOut fOut(Dest);
	String Line;
	char Delimiter;
	int IncludeStart, IncludeLen;
	String IncludeName;
	String First, Second;
	int iSubPackage;
	
	enum { none, preproc, inString, inComment, end } State;
	
#ifdef USE_NAMESPACE
	
	// ENTERS NAME SPACE
	StartNameSpace(fOut);

#endif

	State = none;
	int Indent = 0;
#ifdef USE_NAMESPACE
	bool ToggleNameSpace = false;
#endif
	while(!fIn.IsEof())
	{
		Line = fIn.GetLine();
		
		// AT FIRST, SOME SPECIAL CASES
		// ADDS NAMESPACE TO :: GLOBAL NAMES (HOPING ALL ARE OPENCASCADE...)
#ifdef USE_NAMESPACE
		AddNameSpace(Line);
		ReplaceToken(Line, "pow(themeasurement,anexponent)", "::pow(themeasurement,anexponent)");
		ReplaceToken(Line, "pow(thevalue,atoken->Value()", "::pow(thevalue,atoken->Value()");
		ReplaceToken(Line, "pow(thevalue,anexponent)", "::pow(thevalue,anexponent)");

		ReplaceToken(Line, "struct Tcl_Interp;", "} struct Tcl_Interp; namespace ocf {");
		ReplaceToken(Line, "typedef Tcl_Interp *Draw_PInterp;", "typedef ::Tcl_Interp *Draw_PInterp;");
		ReplaceToken(Line, "OSD::SecSleep", "ocf::OSD::SecSleep");
		ReplaceToken(Line, "ocf::DownCast", "::DownCast");
		ReplaceToken(Line, " ::FUN_ScanInterfList", " ocf::FUN_ScanInterfList");
 		ReplaceToken(Line, " ::FUN_makeUisoLineOnSphe", " ocf::FUN_makeUisoLineOnSphe");
 		ReplaceToken(Line, " ::FUN_reducepure2dI0", " ocf::FUN_reducepure2dI0");
 		ReplaceToken(Line, "void Pack_End()", "void add_cpp_comment_to_method(); void Pack_End()");
		ReplaceToken(Line, ")::CATSHA", ")ocf::CATSHA");

		ReplaceToken(Line, "::SetPoint", "ocf::::SetPoint");
		ReplaceToken(Line, "::FUN_ChkIntg", "ocf::FUN_ChkIntg");
/*
		ReplaceToken(Line, " ::FUN_ChkIntgInterf", " ocf::FUN_ChkIntgInterf");
		ReplaceToken(Line, " ::FUN_ChkIntgSamDomain", " ocf::FUN_ChkIntgSamDomain");
		ReplaceToken(Line, " ::FUN_ChkVertex", " ocf::FUN_ChkVertex");
		ReplaceToken(Line, "", "");
		ReplaceToken(Line, "", "");
		ReplaceToken(Line, "", "");
		ReplaceToken(Line, "", "");
		ReplaceToken(Line, "", "");
		ReplaceToken(Line, "", "");
		ReplaceToken(Line, "", "");
		ReplaceToken(Line, " ::HashCode", " ocf::HashCode");
		ReplaceToken(Line, " ::HASHCODE", " ocf::HASHCODE");
		ReplaceToken(Line, " ::ISSIMILAR", " ocf::ISSIMILAR");
		ReplaceToken(Line, " ::ShallowDump", " ocf::ShallowDump");
		ReplaceToken(Line, "::isName", "ocf::isName");
		ReplaceToken(Line, "::getEncodingName", "ocf::getEncodingName");
		ReplaceToken(Line, "::TryXmlDriverType", "ocf::TryXmlDriverType");
		ReplaceToken(Line, "::ToCharacter", "ocf::ToCharacter");
		ReplaceToken(Line, "::ToExtCharacter", "ocf::ToExtCharacter");
		ReplaceToken(Line, "::EdgesIntersector_checkT1D", "ocf::EdgesIntersector_checkT1D");
		ReplaceToken(Line, "::TransitionToOrientation", "ocf::TransitionToOrientation");
		ReplaceToken(Line, "::MakeCPVInterference", "ocf::MakeCPVInterference");
		ReplaceToken(Line, "::MakeEPVInterference", "ocf::MakeEPVInterference");
		ReplaceToken(Line, "::TopOpeBRepBuild_FUN_aresamegeom", "ocf::TopOpeBRepBuild_FUN_aresamegeom");
		ReplaceToken(Line, "::sectionedgesON", "ocf::sectionedgesON");
		ReplaceToken(Line, "::allIonsectionedges", "ocf::allIonsectionedges");
		ReplaceToken(Line, "::BASISCURVE2D", "ocf::BASISCURVE2D");

		ReplaceToken(Line, "::FUN_", "ocf::FUN_");
*/
/*
		ReplaceToken(Line, "::FUN_ProjectPoint", "ocf::FUN_ProjectPoint");
		ReplaceToken(Line, "::FUN_Ang", "ocf::FUN_Ang");
		ReplaceToken(Line, "::FUN_getSTA", "ocf::FUN_getSTA");
		ReplaceToken(Line, "::FUN_getstate", "ocf::FUN_getstate");
		ReplaceToken(Line, "::FUN_OO", "ocf::FUN_OO");
		ReplaceToken(Line, "::FUN_refnearest", "ocf::FUN_refnearest");
		ReplaceToken(Line, "::FUN_nearestISO", "ocf::FUN_nearestISO");
		ReplaceToken(Line, "::FUN_shareNOG", "ocf::FUN_shareNOG");
		ReplaceToken(Line, "::FUN_GeomTrans", "ocf::FUN_GeomTrans");
		ReplaceToken(Line, "::FUN_brep_ONfirstP", "ocf::FUN_brep_ONfirstP");
		ReplaceToken(Line, "::FUN_haslastvpon0", "ocf::FUN_haslastvpon0");
		ReplaceToken(Line, "::FUN_ALINETOWLINE", "ocf::FUN_ALINETOWLINE");
		ReplaceToken(Line, "::FUN_EqualPonR", "ocf::FUN_EqualPonR");
		ReplaceToken(Line, "::FUN_LineRestF", "ocf::FUN_LineRestF");
		ReplaceToken(Line, "::FUN_transForWL", "ocf::FUN_transForWL");
		ReplaceToken(Line, "::FUN_ScanInterfList", "ocf::FUN_ScanInterfList");
		ReplaceToken(Line, "::FUN_selectGinterference", "ocf::FUN_selectGinterference");
		ReplaceToken(Line, "::FUN_selectTRAISHAinterference", "ocf::FUN_selectTRAISHAinterference");
		ReplaceToken(Line, "::FUN_onedge", "ocf::FUN_onedge");
		ReplaceToken(Line, "::FUN_ScanInterfList", "ocf::FUN_ScanInterfList");
		ReplaceToken(Line, "::FUN_findeSD", "ocf::FUN_findeSD");
		ReplaceToken(Line, "::FUN_isonbound", "ocf::FUN_isonbound");
		ReplaceToken(Line, "::FUN_validF1edge", "ocf::FUN_validF1edge");
		ReplaceToken(Line, "::FUN_changev", "ocf::FUN_changev");
		ReplaceToken(Line, "::FUN_updatev", "ocf::FUN_updatev");
		ReplaceToken(Line, "::FUN_sortplcy", "ocf::FUN_sortplcy");
		ReplaceToken(Line, "::FUN_comparekoletgesh", "ocf::FUN_comparekoletgesh");
		ReplaceToken(Line, "::FUN_rebuildfc", "ocf::FUN_rebuildfc");
		ReplaceToken(Line, "::FUN_addf", "ocf::FUN_addf");
		ReplaceToken(Line, "::FUN_onboundsper", "ocf::FUN_onboundsper");
		ReplaceToken(Line, "::FUN_UisoLineOnSphe", "ocf::FUN_UisoLineOnSphe");
		ReplaceToken(Line, "::FUN_makeUisoLineOnSphe", "ocf::FUN_makeUisoLineOnSphe");
		ReplaceToken(Line, "::FUN_reducepure2dI0", "ocf::FUN_reducepure2dI0");
		ReplaceToken(Line, "::FUN_reducepure2dI(", "ocf::FUN_reducepure2dI(");
*/
/*
		ReplaceToken(Line, "::Parameter", "ocf::Parameter");
		ReplaceToken(Line, "::FDS_Tdata", "ocf::FDS_Tdata");
		ReplaceToken(Line, "::MakeCurve3DfromWLineApprox(", "ocf::MakeCurve3DfromWLineApprox(");
		ReplaceToken(Line, "::MakeCurve2DfromWLineApproxAndPlane(", "ocf::MakeCurve2DfromWLineApproxAndPlane(");
		ReplaceToken(Line, "::MakeCurve2DfromWLineApprox(", "ocf::MakeCurve2DfromWLineApprox(");
		ReplaceToken(Line, "::MakePCurve", "ocf::MakePCurve");
		ReplaceToken(Line, "::ValueString", "ocf::ValueString");
		ReplaceToken(Line, "::GeometriesString", "ocf::GeometriesString");
		ReplaceToken(Line, "::PlaneString", "ocf::PlaneString");
		ReplaceToken(Line, "::TypeString", "ocf::TypeString");
		ReplaceToken(Line, "::StatusString", "ocf::StatusString");
		ReplaceToken(Line, "::ConRadiusString", "ocf::ConRadiusString");
		ReplaceToken(Line, "::ConDiameterString", "ocf::ConDiameterString");
		ReplaceToken(Line, "::ConMinRadiusString", "ocf::ConMinRadiusString");
		ReplaceToken(Line, "::ConMajRadiusString", "ocf::ConMajRadiusString");
		ReplaceToken(Line, "::ConTangentString", "ocf::ConTangentString");
		ReplaceToken(Line, "::ConParallelString", "ocf::ConParallelString");
		ReplaceToken(Line, "::ConPerpendicularString", "ocf::ConPerpendicularString");

		ReplaceToken(Line, "::ConConcentricString", "ocf::ConConcentricString");
		ReplaceToken(Line, "::ConCoincidentString", "ocf::ConCoincidentString");
		ReplaceToken(Line, "::ConDistanceString", "ocf::ConDistanceString");
		ReplaceToken(Line, "::ConAngleString", "ocf::ConAngleString");
		ReplaceToken(Line, "::ConEqualRadiusString", "ocf::ConEqualRadiusString");
		ReplaceToken(Line, "::ConSymmetryString", "ocf::ConSymmetryString");
		ReplaceToken(Line, "::ConMidPointString", "ocf::ConMidPointString");
		ReplaceToken(Line, "::ConEqualDistanceString", "ocf::ConEqualDistanceString");
		ReplaceToken(Line, "::ConFixString", "ocf::ConFixString");
		ReplaceToken(Line, "::ConRigidString", "ocf::ConRigidString");
		ReplaceToken(Line, "::ConFromString", "ocf::ConFromString");
		ReplaceToken(Line, "::ConAxisString", "ocf::ConAxisString");
		ReplaceToken(Line, "::ConMateString", "ocf::ConMateString");
		ReplaceToken(Line, "::ConAlignFacesString", "ocf::ConAlignFacesString");
		ReplaceToken(Line, "::ConAlignAxesString", "ocf::ConAlignAxesString");
		ReplaceToken(Line, "::ConAxesAngleString", "ocf::ConAxesAngleString");
		ReplaceToken(Line, "::ConFacesAngleString", "ocf::ConFacesAngleString");
		ReplaceToken(Line, "::ConRoundString", "ocf::ConRoundString");
		ReplaceToken(Line, "::ConOffsetString", "ocf::ConOffsetString");

::LowerString
::UpperString
::IndString
::take_time
*/
/*


		ReplaceToken(Line, "", "");
		ReplaceToken(Line, "", "");
		ReplaceToken(Line, "", "");
		ReplaceToken(Line, "", "");
		ReplaceToken(Line, "", "");
		ReplaceToken(Line, "", "");
		ReplaceToken(Line, "", "");
*/
#endif
		
		int i = 0 ;
		while(i < Line.GetCount())
		{
			switch(State)
			{
				// NORMAL STATE, LOOKS FOR ALL
				case none :
					if(Line[i] == '#')
						State = preproc;
					else if(Line[i] == '"')
						State = inString;
					else if(Line[i] == 0x5c)
						i++;
					else if(Line[i] == '/' && i < Line.GetCount()-1)
					{
						// C++ STYLE COMMENT ?
						if(Line[i+1] == '/')
						{
							// YES, SKIP REST OF LINE...
							i = Line.GetCount();
						}
						// C STYLE COMMENT ?
						else if(Line[i+1] == '*')
						{
							// YES, CHANGE STATE TO inComment
							// AND GO LOOK FOR END OF COMMENT
							State = inComment;
							i++;
						}
					}
					// COUNTS BLOCK LEVEL... NOT 100% ERROR FREE, BUT SHOULD DO THE JOB
					else if(Line[i] == '{')
						Indent++;
					else if(Line[i] == '}')
						Indent--;

				// INSIDE STRING STATE, LOOKS ONLY FOR END OF STRING
				// OR END OF LINE, WHCH TERMINATES THE STRING TOO
				case inString :
					if(Line[i] == '"' || i >= Line.GetCount()-1)
						State = none;
					else if(Line[i] == 0x5c)
						i++;
					break;
					
				// INSIDE C-STYLE COMMENT, LOOKS ONLY FOR END OF COMMENT
				case inComment :
					if(i < Line.GetCount() -1 && Line[i] == '*' && Line[i+1] == '/')
					{
						State = none;
						i++;
					}
					break;
					
				// PREPROCESSOR DIRECTIVE FOUND, LOOK IF IT'S #include
				case preproc :

					// RESETS STATE FOR FOLLOWING OPERATIONE
					State = none;
					
					// SKIP BLANKS....
					while(i < Line.GetCount() && isspace(Line[i]))
						i++;
					
					if(i < Line.GetCount() - 7 && Line.Mid(i, 7) == "include")
					{
#ifdef USE_NAMESPACE
						// MARKS TOGGLE NAMESPACE IF NEEDED
						if(Indent == 0)
							ToggleNameSpace = true;
#endif
						
						// GO TO THE END OF DIRECTIVE
						i += 7;
						
						// SKIP BLANKS
						while(i < Line.GetCount() && isspace(Line[i]))
							i++;
						
						// IF END OF LINE, ERROR IN DIRECTIVE, SO SKIP IT
						if(i >= Line.GetCount()-2) // 2 is space for 2 delimiters
						{
							i = Line.GetCount();
							continue;
						}
						
						// GETS INCLUDE DELIMITER AND START POS IF INCLUDE NAME
						Delimiter = Line[i++];
						if(Delimiter == '<')
							Delimiter = '>';
						IncludeStart = i;
						
						// LOOKS FORWARD FOR CLOSING DELIMITER
						while(i < Line.GetCount() && Line[i] != Delimiter)
							i++;
						
						// IF END OF LINE, ERROR IN DIRECTIVE, SO SKIP IT
						if(i >= Line.GetCount())
						{
							continue;
						}
						
						// SAVES INCLUDE NAME LENGTH
						IncludeLen = i - IncludeStart;
						
						// LOOKS IF INCLUDE FILE IS AN OPENCASCADE PACKAGE
						IncludeName = Line.Mid(IncludeStart, IncludeLen);
						if(!SplitCascadeName(IncludeName, First, Second))
						{
							i = Line.GetCount();
							continue;
						}
						if((iSubPackage = SubPackages.Find(First)) >= 0)
						{
							// SHOULD BE AN OPENCASCADE ONE
							// LET'S CHECK IF FILE EXISTS... AND WHERE IS IT
							// AND BUILD CORRECT DEST PATH
							if(FileExists(CasRoot + "/inc/" + IncludeName))
								IncludeName = MainPackages[iSubPackage] + "/" + SubPackages[iSubPackage] + "/" + Second;
							else if(FileExists(CasRoot + "/drv/" + SubPackages[iSubPackage] + "/"  + IncludeName))
								IncludeName = "drv/" + SubPackages[iSubPackage] + "/" + IncludeName;
							else if(FileExists(CasRoot + "/src/" + SubPackages[iSubPackage] + "/"  + IncludeName))
								IncludeName = "src/" + SubPackages[iSubPackage] + "/" + IncludeName;
							else
							{
								i = Line.GetCount();
								continue;
							}
							
							// SUBSTITUTE THE CORRECT INCLUDE IN SOURCE LINE
							Line = Line.Mid(0, IncludeStart-1) + "\"" + IncludeName + "\"" + Line.Mid(IncludeStart+IncludeLen+2);
							
							// ADJUST THE LINE INDEX
							i = IncludeStart + IncludeName.GetCount() + 1;
						}
					} // if #include
					
					// HERE a DIRTY TRICK TO DEAL WITH #includes REFERRING TO #defined FILES..
					// IT REPLaCES ONLY THE VALUE OF THE MACRO IF AND ONLY IF IT' S SURROUNDED
					// BY <> AND IS PART OF OpenCascade THREE
					// DON'T DEAL WITH INCLUDES WITH '"' DELIMITER... FOR NOW
					if(i < Line.GetCount() - 6 && Line.Mid(i, 6) == "define")
					{
						// GO TO THE END OF DIRECTIVE
						i += 6;
						
						// SKIP BLANKS
						while(i < Line.GetCount() && isspace(Line[i]))
							i++;
						
						// SEARCH FOR < DELIMITER
						while(i < Line.GetCount() && Line[i] != '<')
							i++;
						
						// IF END OF LINE, NOT FOUND, SO SKIP IT
						if(i >= Line.GetCount())
							continue;
						
						// GETS START OF INCLUDE NAME
						i++;
						IncludeStart = i;
						
						// LOOKS FORWARD FOR CLOSING DELIMITER
						while(i < Line.GetCount() && Line[i] != '>')
							i++;
						
						// IF END OF LINE, ERROR IN DIRECTIVE, SO SKIP IT
						if(i >= Line.GetCount())
							continue;
						
						// SAVES INCLUDE NAME LENGTH
						IncludeLen = i - IncludeStart;
						
						// LOOKS IF INCLUDE FILE IS AN OPENCASCADE PACKAGE
						IncludeName = Line.Mid(IncludeStart, IncludeLen);
						if(!SplitCascadeName(IncludeName, First, Second))
							continue;
						if((iSubPackage = SubPackages.Find(First)) >= 0)
						{
							// SHOULD BE AN OPENCASCADE ONE
							// LET'S CHECK IF FILE EXISTS... AND WHERE IS IT
							// AND BUILD CORRECT DEST PATH
							if(FileExists(CasRoot + "/inc/" + IncludeName))
								IncludeName = MainPackages[iSubPackage] + "/" + SubPackages[iSubPackage] + "/" + Second;
							else if(FileExists(CasRoot + "/drv/" + SubPackages[iSubPackage] + "/"  + IncludeName))
								IncludeName = "drv/" + SubPackages[iSubPackage] + "/" + IncludeName;
							else if(FileExists(CasRoot + "/src/" + SubPackages[iSubPackage] + "/"  + IncludeName))
								IncludeName = "src/" + SubPackages[iSubPackage] + "/" + IncludeName;
							else
								continue;

							// SUBSTITUTE THE CORRECT INCLUDE IN SOURCE LINE
							Line = Line.Mid(0, IncludeStart-1) + "\"" + IncludeName + "\"" + Line.Mid(IncludeStart+IncludeLen+2);
							
							// ADJUST THE LINE INDEX
							i = IncludeStart + IncludeName.GetCount() + 1;
						}
					} // if #define
							
					break;
					
			} // switch
			i++;
		}
#ifdef USE_NAMESPACE
		// CHECKS WHETHER MUST TOGGLE NAME SPACE
		if(ToggleNameSpace)
			EndNameSpace(fOut);
#endif			
		// OUTPUTS THE PATCHED LINE
		fOut.PutLine(Line);

#ifdef USE_NAMESPACE
		// CHECKS WHETHER MUST TOGGLE NAME SPACE
		if(ToggleNameSpace)
			StartNameSpace(fOut);
		ToggleNameSpace = false;
#endif
		
	}
	
#ifdef USE_NAMESPACE
	// CLOSE NAMESPACE
	EndNameSpace(fOut);
#endif

	return true;

} // END CopyPatch()

////////////////////////////////////////////////////////////////////////////////////////////////
// SCANS A Makefile.am FOR DEPENDENT LIBRARIES
Array<String> GetMakefileLibs(const String &RootPath, const String &Name)
{
	Array<String> res;
	String FilePath = RootPath + "/" + Name + "/Makefile.am";
	String Pattern = "lib" + Name + "_la_LIBADD";
	FileIn f(FilePath);
	String Line;
	while(!f.IsEof())
	{
		Line = f.GetLine();
		if(Line.Find(Pattern) >= 0)
			break;
	}
	if(!f.IsEof())
	{
		int k = Line.Find('=');
		ASSERT(k > 0);
		k++;
		while(k < Line.GetCount() && (isspace(Line[k]) || Line[k] == '\\'))
			k++;
		if(k >= Line.GetCount())
			Line = f.GetLine();
		else
			Line = Line.Mid(k);
		while(!f.IsEof() && Line.Find("../") == 0)
		{
			int k = Line.GetCount()-1;
			while(k >= 3 && Line[k] != '.')
				k--;
			if(k > 3)
			{
				Line = Line.Mid(3, k-3);
				k = 0;
				while(k < Line.GetCount() && Line[k] != '/')
					k++;
				k += 3; // skips 'lib' header
				if(k < Line.GetCount()-1)
					res.Add(Line.Mid(k+1));
			}
			Line = f.GetLine();
		}
	}
	
	return res;
	
} // END GetMakefileLibs()

////////////////////////////////////////////////////////////////////////////////////////////////
// SCANS A Makefile.am FOR SOURCE FILES
Array<String> GetMakefileSources(const String &RootPath, const String &Name)
{
	Array<String> res;
	String FilePath = RootPath + "/" + Name + "/Makefile.am";
	String Pattern = "lib" + Name + "_la_SOURCES";
	FileIn f(FilePath);
	String Line;
	while(!f.IsEof())
	{
		Line = f.GetLine();
		if(Line.Find(Pattern) >= 0)
			break;
	}
	if(!f.IsEof())
	{
		int k = Line.Find('=');
		ASSERT(k > 0);
		k++;
		while(k < Line.GetCount() && (isspace(Line[k]) || Line[k] == '\\'))
			k++;
		if(k >= Line.GetCount())
			Line = f.GetLine();
		else
			Line = Line.Mid(k);

		while(!f.IsEof() && Line.Find("@top_srcdir@/") == 0)
		{
			int k = Line.GetCount()-1;
			while(k >= 13 && (isspace(Line[k]) || Line[k] == '\\'))
				k--;
			if(k > 13)
			{
				Line = Line.Mid(13, k-12);
				res.Add(Line);
			}
			Line = f.GetLine();
		}
	}
	
	return res;
	
} // END GetMakefileSources()

////////////////////////////////////////////////////////////////////////////////////////////////
// CREATES UPP PACKAGES FOR LIBRARY BUILD
bool CreatePackages(const String &CasRoot, const String &DestRoot)
{
	// CREATES MAKEILE ROOT PATH
	String MakeRootPath = CasRoot + "/adm/make";
	
	// CREATE PACKAGES ROOT PATH
	String PackagesRoot = DestRoot + "/OpenCascadeBuild";
	
	// READS IN THE LIBRARY NAMES
	Array<String> LibNames = GetDirectoryContent(MakeRootPath + "/*", true);
	
	// FOR EACH LIBRARY, CREATES THE CORRESPONDING UPP PACKAGE
	for(int iLib = 0 ; iLib < LibNames.GetCount() ; iLib++)
	{
		String LibName = LibNames[iLib];
		Array<String> Uses = GetMakefileLibs(MakeRootPath, LibName);
		Array<String> Sources = GetMakefileSources(MakeRootPath, LibName);
		
		// CREATES THE PACKAGE FILE
		String PackagePath = PackagesRoot + "/" + LibName;
		RealizeDirectory(PackagePath);
		PackagePath = PackagePath + "/" + LibName + ".upp";
		FileOut f(PackagePath);
		
		// TARGET NAME
		f.PutLine("target");
		f.PutLine("\x09lib" + LibName + ".so;");
		f.PutLine("");
		
		// OPTIONS
		f.PutLine("options");
		f.PutLine("\x09\"-DHAVE_CONFIG_H -DHAVE_WOK_CONFIG_H\",");
		f.PutLine("\x09\"-DLIN -DLININTEL\",");
		f.PutLine("\x09\"-DOCC_CONVERT_SIGNALS -DNOPROTECTION -DCSFDB\",");
		f.PutLine("\x09\"-DNDEBUG -DNo_Exception\",");
		f.PutLine("\x09\"-ffriend-injection -fpermissive\",");
		f.PutLine("\x09\"-ffunction-sections -fPIC\",");
		f.PutLine("\x09-fmessage-length=0,");
		f.PutLine("\x09-funsigned-char,");
		f.PutLine("\x09-MD,");
		f.PutLine("\x09-Wall,");
		f.PutLine("\x09\"-Wno-non-virtual-dtor -Wno-cast-qual\",");
		f.PutLine("\x09\"-Wno-unused-variable -Wno-unused-function\";");
		f.PutLine("");
		
		// USED PACKAGES
		if(Uses.GetCount())
			f.PutLine("uses");
		for(int iUse = 0 ; iUse < Uses.GetCount()-1; iUse++)
			f.PutLine("\x09" + Uses[iUse] + ",");
		if(Uses.GetCount())
			f.PutLine("\x09" + Uses[Uses.GetCount()-1] + ";");
		f.PutLine("");
			
		// SOURCE FILES
		if(Sources.GetCount())
			f.PutLine("file");
		for(int iSource = 0 ; iSource < Sources.GetCount()-1; iSource++)
			f.PutLine("\x09../../OpenCascadeSrc/" + Sources[iSource] +",");
		if(Sources.GetCount())
			f.PutLine("\x09../../OpenCascadeSrc/" + Sources[Sources.GetCount()-1] +";");
		f.PutLine("");

		// MAIN CONFIG		
//		f.PutLine("mainconfig");
//		f.PutLine("\x09\"\" = \"DLL \";");
//		f.PutLine("");
		
	} // for iLib
	
	// CREATES DUMMY TOP-LEVEL PACKAGE TO HAVE ALL LIBS BUILT
	String TopPath = PackagesRoot + "/BUILDALL";
	RealizeDirectory(TopPath);
	TopPath = TopPath + "/BUILDALL.upp";
	FileOut f(TopPath);
	
	if(LibNames.GetCount())
		f.PutLine("uses");
	for(int iLib = 0 ; iLib < LibNames.GetCount()-1 ; iLib++)
		f.PutLine("\x09" + LibNames[iLib] + ",");
	if(LibNames.GetCount())
		f.PutLine("\x09" + LibNames[LibNames.GetCount()-1] + ";");
	f.PutLine("");
	
	f.PutLine("mainconfig");	
	f.PutLine("\x09\"\" = \"\"");
	f.PutLine("");
	
} // END CreatePackages()


////////////////////////////////////////////////////////////////////////////////////////////////
// PACKS OPENCASCADE FOR UPP USAGE
bool PackageCascade(const String &CasRoot, const String &DestRoot)
{
	// GETS THE TEMPLATE NAMES FROM DESTINATION DIRECTORY
	// (MUST EXIST AND FILLED WITH INCLUDE FILE NAMES)
	Array<String> TemplateNames = GetTemplateNames(DestRoot);

	// FOR EACH TEMPLATE, EXTRACTS INCLUDED PACKAGE NAMES
	// GETTING THEM FROM INCLUDE FILE NAMES AND MAPS THEM TO TEMPLATE NAME
	Index<String> SubPackages;
	Array<String> MainPackages;
	for(int i = 0 ; i < TemplateNames.GetCount() ; i++)
	{
		// READS THE TEMPLATE FILE
		Array<String> TemplateLines = ReadTemplate(DestRoot + "/" + TemplateNames[i] + ".template");
		
		// EXTRACTS THE PACKAGE NAMES
		Array<String> TemplatePackages = GetSubPackageNames(TemplateLines);
		
		// MAPS THE PACKAGE NAMES WITH CORRESPONDING TEMPLATE
		// AND FILLS THE GLOBAL PACKAGES ARRAY FROM TEMPLATES
		for(int j = 0 ; j < TemplatePackages.GetCount() ; j++)
		{
			SubPackages.Add(TemplatePackages[j]);
			MainPackages.Add(TemplateNames[i]);
		}
	}
	
	// HERE WE MUST DEAL WITH INCLUDE FILES THAT ARE NOT INCLUDED IN PACKAGES
	// INSIDE THE TEMPLATES; SHOULD NOT HAPPEN, BUT... WE MAP THEM TO 'Others' TEMPLATE
	
	// FIRST, WE READ THE INCLUDE FILES....
	Array<String> IncludeFiles = GetIncludeNames(CasRoot);
	
	// ...FROM THEM WE EXTRACT A PACKAGE NAMES LIST....
	Array<String> PackagesFromIncludes = GetSubPackageNames(IncludeFiles);
	
	// AND WE LOOK FOR PACKAGES NOT INCLUDED IN TEMPLATES
	// MAPPING THEM TO 'Others' CATEGORY
	bool FoundOrphan = false;
	for(int i = 0 ; i < PackagesFromIncludes.GetCount() ; i++)
	{
		if(SubPackages.Find(PackagesFromIncludes[i]) < 0)
		{
			SubPackages.Add(PackagesFromIncludes[i]);
			MainPackages.Add("Others");
			FoundOrphan = true;
		}
	}
	if(FoundOrphan)
		TemplateNames << "Others";
	
	////////////////////////////////////////////////////////////////////////////////////////////
	// HERE WE HAVE :
	//  All template (category) names in TemplateNames
	//  2 Arrays mapping  Package names with corresponding Templates in SubPackages+MainPackages
	//  the complete array of include file names in IncludeFiles
	//
	// We must now create package folders on destination path
	// and copy include files there, changing #include directives if necessary

	// NOW WE MUST COPY ALL INCLUDE FILES, PATCHING THE INCLUDE DIRECTIVES
	// WE PUT THEM ON CHILD OF OpenCascade FOLDER IN ROOT DEST
	for(int i = 0 ; i < IncludeFiles.GetCount() ; i++)
	{
		String First, Second;
		if(!SplitCascadeName(IncludeFiles[i], First, Second))
		  continue;
		String Source = CasRoot + "/inc/" + IncludeFiles[i];
		int iSubPackage = SubPackages.Find(First);
		String MainPackage = MainPackages[iSubPackage];
		String DestPath = DestRoot + "/OpenCascade/" + MainPackage + "/" + First;
		RealizeDirectory(DestPath);
		String Dest = DestPath + "/" + Second;
		CopyPatch(CasRoot, Source, Dest, SubPackages, MainPackages);
	}

	//////////////////////////////////////////////////////////////////////////////////////////////
	// NOW WE MUST DEAL WITH SOURCE FILES
	// WE PUT THEM IN CHILD OF OpenCascadeSrc/src FOLDER IN ROOT DEST
	
	// AT FIRST, WE GET A LIST OF SOURCE FILE FOLDERS....
	Array<String> SourceFolders = GetSourceFolders(CasRoot);
	
	// NOW WE PROCESS EACH SOURCE FOLDER
	for(int i = 0 ; i < SourceFolders.GetCount() ; i++)
	{
		// WE CREATE THE SOURCE FULL PATH
		String SourcePath = CasRoot + "/src/" + SourceFolders[i] + "/";
		
		// WE CREATE THE DEST FULL PATH AND THE FOLDER, IF NOT THERE
		String DestPath = DestRoot + "/OpenCascadeSrc/src/" + SourceFolders[i] + "/";
		RealizeDirectory(DestPath);
		
		// WE'RE DONE WITH FOLDERS, NOW WE DEAL WITH FILES ON EACH FOLDER
		Array<String> SourceFiles = GetDirectoryContent(CasRoot + "/src/" + SourceFolders[i] + "/*.*", false);
		for(int j = 0 ; j < SourceFiles.GetCount() ; j++)
		{
			// WE GET THE FILENAME
			String SourceFile = SourceFiles[j];
			
			// SKIP NON-SOURCE FILES
			if(!IsSourceFile(SourceFile))
				continue;
			
			// WE CREATE SOURCE AND DEST FULL PATHS AND COPY/PATCH THE FILE
			String Source = SourcePath + SourceFile;
			String Dest = DestPath + SourceFile;
			CopyPatch(CasRoot, Source, Dest, SubPackages, MainPackages);
						
		}
	}
	
	//////////////////////////////////////////////////////////////////////////////////////////////
	// NOW WE MUST DEAL WITH DRV FILES
	// WE PUT THEM IN CHILD OF OpenCascadeSrc/drv FOLDER IN ROOT DEST
	
	// AT FIRST, WE GET A LIST OF DRV FILE FOLDERS....
	Array<String> DrvFolders = GetDrvFolders(CasRoot);
	
	// NOW WE PROCESS EACH DRV FOLDER
	for(int i = 0 ; i < DrvFolders.GetCount() ; i++)
	{
		// WE CREATE THE SOURCE FULL PATH
		String SourcePath = CasRoot + "/drv/" + DrvFolders[i] + "/";
		
		// WE CREATE THE DEST FULL PATH AND THE FOLDER, IF NOT THERE
		String DestPath = DestRoot + "/OpenCascadeSrc/drv/" + SourceFolders[i] + "/";
		RealizeDirectory(DestPath);
		
		// WE'RE DONE WITH FOLDERS, NOW WE DEAL WITH FILES ON EACH FOLDER
		Array<String> DrvFiles = GetDirectoryContent(CasRoot + "/drv/" + DrvFolders[i] + "/*.*", false);
		for(int j = 0 ; j < DrvFiles.GetCount() ; j++)
		{
			// WE GET THE FILENAME
			String DrvFile = DrvFiles[j];
			
			// SKIP NON-SOURCE FILES
			if(!IsSourceFile(DrvFile))
				continue;
			
			// WE CREATE SOURCE AND DEST FULL PATHS AND COPY/PATCH THE FILE
			String Source = SourcePath + DrvFile;
			String Dest = DestPath + DrvFile;
			CopyPatch(CasRoot, Source, Dest, SubPackages, MainPackages);
		}
	}

	return true;

} // END PackageCascade()

CONSOLE_APP_MAIN
{
	int argc = CommandLine().GetCount();
	const Vector<String>& argv = CommandLine();
	String CasRoot;

	if(argc == 1)
	{
		// CHECKS FOR DESTINATION FOLDER
		if(!FileSystemInfo().FolderExists(argv[0]))
		{
			Cout() << "Destination folder '" << argv[0] << "' don't exist";
			exit(1);
		}
		
		// CHECKS FOR EXISTING OPENCASCADE
		CasRoot = GetCasRoot();
		if(CasRoot == "")
		{
			Cout() << "OpenCASCADE folder not found\nPlease check OpenCASCADE setup\n";
			exit(1);
		}

		PackageCascade(CasRoot, argv[0]);
		
		CreatePackages(CasRoot, argv[0]);
		
		SetExitCode(0);
	}
	else
	{
		Cout() << "USAGE : PackCascade <dest root path>\n";
		exit(1);
	}
}
