#include "CsvComparator.h"
#include "Tcc/Tcc.h"
#include "csvComparaisonCode.h"

#include <stdio.h>
#include <iostream>
#include <sstream>

typedef union{
	CDT_UINT    _uint;
	CDT_INT     _int;
	CDT_FLOAT   _float;
	CDT_STRING  _string;
	CDT_BOOL    _bool;
} CsvElements;


// ===========================================================================================
// ===========================================================================================
//
//				StringTok  functions
//
// ===========================================================================================
// ===========================================================================================
void StringTok(const String& input,  Vector<String>& tokens, const char separator )
{
	int currPos = 0;
	int nextPos = 0;
	const int length = input.GetLength();
	nextPos = input.Find( separator, currPos );
	while ( (currPos < length) && (nextPos>=0) )
	{
		String lineString =  String(input.Begin() + currPos, nextPos-currPos );
		if (lineString.GetLength() > 0) 
		{
			tokens << lineString;
		}
		currPos = nextPos + 1;
		nextPos = input.Find( separator, currPos );
	}

	// CASE: end of string
	if ( (nextPos < 0) && (currPos < length) )
	{
		String lineString =  String(input.Begin() + currPos, length-currPos );
		tokens << lineString;
	}
}


void StringTokTok(const String& input,  TokenContainer& tokens , const char separators[2])
{
	const char separator = separators[0];
	Vector<String> lineTokens;
	int currPos = 0;
	int nextPos = 0;
	const int length = input.GetLength();
	nextPos = input.Find( separator, currPos );
	while ( (currPos < length) && (nextPos>=0) )
	{
		String lineString =  String(input.Begin() + currPos, nextPos-currPos );
		currPos = nextPos + 1;
		nextPos = input.Find( separator, currPos );
		StringTok( lineString, lineTokens, separators[1] );
		tokens << lineTokens;
		lineTokens.Clear();
	}

	// CASE: end of string
	if ( (nextPos < 0) && (currPos < length) )
	{
		String lineString =  String(input.Begin() + currPos, length-currPos );
		StringTok( lineString, lineTokens, separators[1]);
		tokens << lineTokens;
	}
}


void tokenContainer2tokArray(int nbRows, int nbColumns, const char***& tokArray, TokenContainer& tokContainer)
{
	int rowOffset = tokContainer.GetCount() - nbRows; // To manage TITLE line
	tokArray = new const char**[nbColumns];
	for (int col=0; col<nbColumns; col++)
	{
		tokArray[col] = new const char*[nbRows];
		for (int row=0; row<nbRows; row++)
		{
			tokArray[col][row] = tokContainer[row+rowOffset][col].Begin();
		}
	}
}


void freetokArray( int nbColumns, const char***& tokArray)
{
	for (int col=0; col<nbColumns; col++)
	{
		if (tokArray[col])	delete [] tokArray[col];
		tokArray[col] = 0;
	}
	if (tokArray) delete [] tokArray;
	tokArray = 0;
}

// ===========================================================================================
// ===========================================================================================
//
//				CsvComparator class
//
// ===========================================================================================
// ===========================================================================================

CsvComparator::CsvComparator(String csvRef, String CompareCode, String csvResult)
: refTokensArray(0)
, resultTokensArray(0)
{
	code = LoadFile( CompareCode );
	refFile = LoadFile( csvRef );
	resultFile = LoadFile( csvResult );
	
	CtrlLayout(*this, "CsvComparator  Tool" );
	CtrlLayout(tabGrid);
	tabGrid.SizePos();
	CtrlLayout(tabCharts);
	tabCharts.SizePos();

	tabs.Add(tabGrid, "DATA" );
	tabs.Add(tabCharts, "CHARTS" );
	
	Sizeable();
}

CsvComparator::~CsvComparator()
{
	freetokArray(nbColumns, refTokensArray);
	freetokArray(nbColumns, resultTokensArray);
}

// FUNCTION TYPE DEFINITIONS
typedef ColumnDataTypeEnum (*GetDataTypeFctType)(int);
typedef int (*GetNbColumnsFctType)();
typedef void (*GetParsingSeparatorsFctType)(char*, char*);






#define BAD_ROW_COLOR  Color(225, 196, 196)// Color(255, 174, 159) //Color(255, 95, 95)

template <class T>
String ValToQtfErrString(T val, T refval)
{
	//[*@6$(255.212.0)1 val][$(255.212.0)1  `=! refval]]
	String res = "\1[*@6$(255.212.0)1 ";
	res << val << "][$(255.212.0)1  `=! "<< refval;
	return res;
}

template <class T >
bool processColumn(Tcc& tcc, int colNum, int nbRows, const char*** resultStringValues, const char*** refStringValues, GridCtrl& grid)
{
	LOG(String("Processing column: ") << colNum);
	T columnData[nbRows];
	T refColumnData[nbRows];

	// CONVERT COLUMN DATA
	for (int rowNum=0; rowNum<nbRows; rowNum++)
	{
		std::istringstream istr(resultStringValues[colNum][rowNum]);
		istr >> columnData[rowNum];
		std::istringstream refistr(refStringValues[colNum][rowNum]);
		refistr >> refColumnData[rowNum];
	}
	String fctNameStr="testCol_";
	fctNameStr << colNum;
	typedef bool (*DoTestFctType)  (int, T*, T*, const char***, const char***);
	DoTestFctType doTest = (DoTestFctType)tcc.GetSymbol( fctNameStr.Begin() );

	bool res = true; // No errors: ref and values are identical
	// PROCESS COLUMN DATA
	for (int rowNum=0; rowNum<nbRows; rowNum++)
	{
		if ( !doTest(rowNum, columnData, refColumnData, resultStringValues, refStringValues ) )
		{
			res = false;
			grid.GetRow(rowNum).Bg(BAD_ROW_COLOR);
			grid( rowNum, colNum+1 ) = ValToQtfErrString(columnData[rowNum], refColumnData[rowNum]);
		}
		LOG(String("[") << colNum << ", " << rowNum << "]  doTest(" << columnData[rowNum] << ", " << refColumnData[rowNum] << " ) ==> " << res);
	}
	return res;
}

template <>
bool processColumn<CDT_STRING>(Tcc& tcc, int colNum, int nbRows, const char*** resultStringValues, const char*** refStringValues, GridCtrl& grid)
{
	LOG(String("Processing STRING column: ") << colNum);
	String fctNameStr = "testCol_";
	fctNameStr << colNum;
	typedef bool (*DoTestFctType)  (int, CDT_STRING*, CDT_STRING*, const char***, const char***);
	DoTestFctType doTest = (DoTestFctType)tcc.GetSymbol(fctNameStr.Begin());

	bool res = true; // No errors: ref and values are identical
	// PROCESS COLUMN DATA
	for (int rowNum=0; rowNum<nbRows; rowNum++)
	{
		if ( !doTest(rowNum, resultStringValues[colNum], refStringValues[colNum], resultStringValues, refStringValues) )
		{
			res = false;
			grid.GetRow(rowNum).Bg(BAD_ROW_COLOR);
			grid( rowNum, colNum+1 ) = ValToQtfErrString(resultStringValues[colNum][rowNum], refStringValues[colNum][rowNum]);
			
			
			StaticTextCtrl* ctrl = (StaticTextCtrl*)( grid.GetCtrl(rowNum, colNum+1) );
			if (ctrl != 0)
			{
				ctrl->SetInk(Yellow());
				ctrl->SetFont(StdFont().Bold());
			}
		}
		LOG(String("[") << colNum << ", " << rowNum << "]  doTest(" << resultStringValues[colNum][rowNum] << ", " << refStringValues[colNum][rowNum] << " ) ==> " << res);
	}
	return res;
}


static void MakeStaticTextCtrl(One<Ctrl>& ctrl)
{
	ctrl.Create<StaticTextCtrl>();
}

CsvComparator::ErrorCodes CsvComparator::doCompare()
{
	ErrorCodes returnCode = FAILED_TO_EXECUTE;
	
	try
	{
		Tcc tcc;
		LOG("Start Processing");
		
		tcc.AddIncludePath("/usr/include");
		tcc.AddLibraryPath("/usr/lib");
	
	
		tcc.Compile(code);
		LOG("compile done");
		tcc.Link();
		LOG("Link done");
		GetDataTypeFctType getDataTypeFct =  (GetDataTypeFctType)tcc.GetSymbol("getDataType");

		char separators[2] = { '\n', '	' };
		GetParsingSeparatorsFctType getParsingSeparators =  (GetParsingSeparatorsFctType)tcc.GetSymbol("getParsingSeparators");
		if (getDataType)
		{
			// modifications des separateurs par defaut
			getParsingSeparators(&(separators[0]), &(separators[1]));
			LOG("Overriding PARSING SEPARATORS");
		}

		StringTokTok(refFile, refTokens, separators);
		StringTokTok(resultFile, resultTokens, separators);
		
		
		nbRows = refTokens.GetCount()-1; // first line is TITLE
		nbColumns = refTokens[1].GetCount();


		// Remplissage du tableaux
		tabGrid.grid.Absolute();
		tabGrid.grid.AddColumn("Count", 50).Fg(Blue());

		
		for (int c=0; c<nbColumns; c++)
		{
			if ( getDataTypeFct(c) != CDTE_NOT_PROCESSED)
			{
				tabGrid.grid.AddColumn( resultTokens[0][c], 100 ).Ctrls(MakeStaticTextCtrl);
			}
			else
			{
				tabGrid.grid.AddColumn( resultTokens[0][c], 100 ).Fg(Gray());
			}
		}

		for (int r=0; r<nbRows; r++)
		{
			tabGrid.grid.Add();
			tabGrid.grid(r,0) = r;

			for (int c=0; c<nbColumns; c++)
			{
				tabGrid.grid( r, c+1 ) = resultTokens[r+1][c];
			}
			if (r & 0x4) tabGrid.grid.GetRow(r).Bg(WhiteGray);
		}
		
		
		tabGrid.grid.AutoHideHorzSb(false);
		
		if ( nbRows != resultTokens.GetCount()-1 )
		{
			TRACE_ERROR("REF and RESULT files have a different number of ROWS");
			returnCode = NBR_ROWS_IS_DIFF;
		}
		else if( nbColumns != resultTokens[1].GetCount() )
		{
			TRACE_ERROR("REF and RESULT files have a different number of COLUMNS");
			returnCode = NBR_COL_IS_DIFF;
		}
		else
		{
			tokenContainer2tokArray( nbRows, nbColumns, refTokensArray, refTokens );
			tokenContainer2tokArray( nbRows, nbColumns, resultTokensArray, resultTokens );
	
		
			bool res = true;
			bool res2 = true;
			LOG(String("Testing ...") << nbColumns << " columns    " << nbRows << " rows" );
			for (int colNum=0; colNum<=nbColumns; colNum++)
			{
				switch ( getDataTypeFct(colNum) )
				{
					case CDTE_NOT_PROCESSED:
						res2=true;
						break;
		
					case CDTE_UINT:
						res2 = processColumn<CDT_UINT>(tcc, colNum, nbRows, resultTokensArray, refTokensArray, tabGrid.grid);
						break;
		
					case CDTE_INT:
						res2 = processColumn<CDT_INT>(tcc, colNum, nbRows, resultTokensArray, refTokensArray, tabGrid.grid);
						break;
		
					case CDTE_FLOAT:
						res2 = processColumn<CDT_FLOAT>(tcc, colNum, nbRows, resultTokensArray, refTokensArray, tabGrid.grid);
						break;
		
					case CDTE_STRING:
						res2 = processColumn<CDT_STRING>(tcc, colNum, nbRows, resultTokensArray, refTokensArray, tabGrid.grid);
						break;
		
					case CDTE_BOOL:
						res2 = processColumn<CDT_BOOL>(tcc, colNum, nbRows, resultTokensArray, refTokensArray, tabGrid.grid);
						break;
				}
				if ( !res2 ) tabGrid.grid.GetColumn(colNum+1).Fg(Red);
				res &= res2;
			}
		
		
			
			if ( res == true)
			{
				TRACE_INFO( "col and refCol are EQUAL");
				returnCode = FILES_ARE_IDENTICAL;
			}
			else
			{
				TRACE_INFO("col and refCol are DIFFERENT");
				returnCode = FILES_ARE_DIFERENT;
			}
		}
		LOG("End of testing ...");
	}
	catch(Exc e)
	{
		returnCode = FAILED_TO_EXECUTE;
		TRACE_ERROR("EXECUTION FAILED: " << e);
	}
	return returnCode;
}


GUI_APP_MAIN
{
	int argc = CommandLine().GetCount();
	if ( argc != 3 )
	{
		TRACE_INFO(" USAGE: CsvComparator  ref.csv  compareCode.c  result.csv");
		TRACE_INFO("        ref.csv:       reference file");
		TRACE_INFO("        compareCode.c: code used to compare the csv files");
		TRACE_INFO("        result.csv:    file to compare with the reference file");
		TRACE_INFO("    Returns:   0: files are identical");
		TRACE_INFO("               1: files are different");
		TRACE_INFO("              -1: one of the files does not exist");
		TRACE_INFO("              -2: ref and result file have different number of rows");
		TRACE_INFO("              -3: ref and result file have different number of columns");
		TRACE_INFO("              -4: one of the files is malformed");
		TRACE_INFO("              -5: Failed to execute for some reason");
		TRACE_INFO("              -6: missing parameters");
		SetExitCode(CsvComparator::BAD_PARAMETER_NUMBER);
	}
	else
	{
		const Vector<String>& argv = CommandLine();

		String refCsv = argv[0];
		String compareCode = argv[1];
		String resultCsv = argv[2];

		if (GetFileDirectory(refCsv).IsEmpty())         refCsv = NativePath(GetCurrentDirectory() + "/" + refCsv);
		if (GetFileDirectory(compareCode).IsEmpty())    compareCode = NativePath(GetCurrentDirectory() + "/" + compareCode);
		if (GetFileDirectory(resultCsv).IsEmpty())      resultCsv = NativePath(GetCurrentDirectory() + "/" + resultCsv);

		LOG( refCsv );
		LOG( compareCode );
		LOG( resultCsv );
		
		if (FileExists(refCsv) && FileExists(compareCode) && FileExists(resultCsv))
		{
			CsvComparator comparator(refCsv, compareCode, resultCsv);//.Run();
			SetExitCode(comparator.doCompare());
			comparator.Run();
		}
		else
		{
			String errMsg;
			if (!FileExists(refCsv)) errMsg << refCsv + "   DOES NOT EXIST \n";
			if (!FileExists(compareCode)) errMsg << compareCode + "   DOES NOT EXIST \n";
			if (!FileExists(resultCsv)) errMsg << resultCsv + "   DOES NOT EXIST \n";
			TRACE_ERROR(errMsg);
			SetExitCode(CsvComparator::FILE_DOES_NOT_EXIST);
		}
	}
}



