#include "JFSlider.h"

#define			BORDER_SIZE			5
#define			BORDER1				BORDER_SIZE
#define			BORDER2				( 2 * BORDER_SIZE )

#define IMAGECLASS JFSliderImg
#define IMAGEFILE <JFControls/JFSlider.iml>
#include <Draw/iml.h>

// The slider is divided into the following zones
// Full = the width for a vertical slider / height for a horiz slider
// Half  = half size
// Quart = quarter the size
// Eighth= eighth of the size
int	Full, Half, Quart, Eighth;

JFSlider::JFSlider()
: m_nMin(0)
, m_nMax(100)
, m_nStep(1)
, m_bRound_step(true)
, m_bJump(false)
, m_bUseCustomThumbs( 0 )
, m_nMajorTicks( 10 )
, m_nMinorTicks( 2 )
, mFillColor(LtCyan())
, mSliderType(CONTROL)
{
	SetValue( 0 );		// set the thumb to zero position
	Transparent();		// make the control transparent
	NoWantFocus();		// Do not want focus at this time
}

// Check if the slider is vertically oriented
bool JFSlider::IsVert() const
{
	return GetSize().cx < GetSize().cy;
}

// return x for Hslider or y for Vslider
int  JFSlider::HoVe(int  x, int  y) const
{
	return IsVert() ? y : x;
}

// return &x for Hslider or &y for Vslider
int& JFSlider::HoVeR(int& x, int& y) const
{
	return IsVert() ? y : x;
}

// Draw the slider on the canvas w
void JFSlider::Paint(Draw& w)
{
	Size size = GetSize();

	// set up the parameters for drawing the slider
	if(IsVert()) 
	    // Vertical slider
		// build up the values
		Full = size.cx;
	else
		Full = size.cy;
	
	Half  = Full/2;
	Quart = Half/2;
	Eighth = Quart/2;
	
	// draw gradations
	for( 	int i = Min(); 
			( (m_nMajorTicks > 0) || (m_nMinorTicks > 0)) && ( i <= Max() ) ;
			i += ( m_nMinorTicks == 0 ? m_nMajorTicks : m_nMinorTicks ) ) {
		
		// convert the slider value to position on canvas
		int nPos = SliderToClient( i );
		
		// draw ticks as defined
		if( ( m_nMajorTicks != 0 ) && ( i % m_nMajorTicks ) == 0 )
			DrawTick( w, MAJOR, (HOVE)HoVe( HORZ, VERT ), nPos, i );
		else if( ( m_nMinorTicks != 0 ) && ( i % m_nMinorTicks ) == 0 )
			DrawTick( w, MINOR, (HOVE)HoVe( HORZ, VERT ), nPos, i );
	}

	if(IsVert()) 
	{   // Vertical slider
		// Draw a frame for the fill
		DrawBorder(w, Half+Quart, BORDER1,
					  Quart, size.cy - BORDER2,
					  BlackBorder());
		
		// draw the fill where bottom is the min value
		w.DrawRect(Half+Quart+1, SliderToClient(m_vValues[0])+1,
				   Quart-2, size.cy-SliderToClient(m_vValues[0])-BORDER1-2,
				   mFillColor);

		// show the Thumb only if the slider is a CONTROL
		if (mSliderType == CONTROL)
		{
			// Draw thumbs for each set of data
			w.DrawImage(
	/*x*/	Half+Quart ,
	/*y*/	SliderToClient(m_vValues[0]) - Eighth,
	/*cx*/	Quart,
	/*cy*/	Quart,
			HasCapture() || HasFocus() ? ( m_bUseCustomThumbs ? m_vThumbImgsFocus[0] : CtrlsImg::kLA() ) : ( m_bUseCustomThumbs ? m_vThumbImgs[0] : CtrlsImg::kLA() ) 
			);
		}
	}
	else 
	{ 	// Draw Line Border, L T R B
		DrawBorder(w, BORDER1, Half+Quart, size.cx-BORDER2, Quart, BlackBorder);
		w.DrawRect(BORDER1+1, Half+Quart+1, SliderToClient(m_vValues[0])-2, Quart-2, mFillColor);
		
		// draw thumbs
		if (mSliderType == CONTROL)
		{
			w.DrawImage(
			SliderToClient(m_vValues[0]),Half+Quart,
			Quart,	Quart,
		    HasCapture() || HasFocus() ? (m_bUseCustomThumbs ? m_vThumbImgsFocus[0] : CtrlsImg::kUA()) : (m_bUseCustomThumbs ? m_vThumbImgs[0] : CtrlsImg::kUA() ));
		}
	}
	if(HasFocus())		DrawFocus(w, size);
}


// draw a tickmark at nPos on canvas
void	JFSlider::DrawTick( Draw &w, MAJORMINOR Type, HOVE Orientation, int nPos, int nVal )
{
	Size size = GetSize();
	int		nMajorWidth = 3;

	if( Orientation == HORZ ) {
		if( Type == MAJOR ) {
			
			w.DrawLine(	nPos+BORDER1, Half, 
						nPos+BORDER1, Half+Quart,
						nMajorWidth );
			
			Size sz = GetTextSize( AsString(nVal), StdFont() );
			int nTextPos = nPos - (int)( sz.cx / 2.0f + 0.5f );
			nTextPos = min( max( 0, nTextPos ), size.cx - sz.cx );
			w.DrawText( nTextPos, BORDER1, AsString( nVal ) );
		}
		else {
			w.DrawLine( nPos+BORDER1, Half+Eighth, nPos+BORDER1, Half+Quart );
		}
	}
	else {// vert
		if( Type == MAJOR ) 
		{
			w.DrawLine( Half, max( nPos, nMajorWidth >> 1 ), Half+Quart,  max( nPos, nMajorWidth >> 1 ), nMajorWidth );
			
			Size sz = GetTextSize( AsString(nVal), StdFont() );
			int nTextPos = nPos - (int)( sz.cy / 2.0f + 0.5f );
			nTextPos = min( max( 0, nTextPos ), size.cy - sz.cy );
			w.DrawText( BORDER1, nTextPos, AsString( nVal ) );
		}
		else 
		{
			w.DrawLine(Half+Eighth, nPos, Half+Quart, nPos);
		}
	}
}

// handle keys addressed to the slider (GotFocussed)
// only if it is a CONTROL
bool JFSlider::Key(dword key, int repcnt)
{
	// No thumb action for indicator
	if (mSliderType == INDICATOR)	return Ctrl::Key(key, repcnt);

	if(IsEditable())
		switch(key) {
		case K_LEFT:
		case K_UP:
			Dec();
			return true;
		case K_RIGHT:
		case K_DOWN:
			Inc();
			return true;
		}
	return Ctrl::Key(key, repcnt);
}

// handle the left mouse button if the slider is s CONTROL
void JFSlider::LeftDown(Point pos, dword keyflags)
{
	if(!IsEditable())  return;
	
	// No thumb action for indicator
	if (mSliderType == INDICATOR)	return;
	
	SetWantFocus();
	
	int thumbPos = SliderToClient(m_vValues[0]);
	int p = HoVe(pos.x, pos.y);
	int nHalfThumb = HoVe(m_bUseCustomThumbs ? m_vThumbImgs[0].GetSize().cx : CtrlImg::hthumb().GetSize().cx,
			 m_bUseCustomThumbs ? m_vThumbImgs[0].GetSize().cy : CtrlImg::vthumb().GetSize().cy) >> 1;
			 
	if(IsNull(thumbPos)) {
		SetValue( ClientToSlider(p) );
		WhenSlideFinish();
		UpdateActionRefresh();
	}
	// Did we click on the thumb?
	else if( ( p >= ( thumbPos - nHalfThumb ) ) && 
		( p < ( thumbPos + nHalfThumb ) ) )
		SetCapture();
	
	else if( m_bJump ) 
	{
		m_vValues[0] = ClientToSlider(p);
		WhenSlideFinish();
		UpdateActionRefresh();		
	}
	else 
	{	// on either side of the thumb position, let the slider Inc or Dec
		if( ( ( p < thumbPos) && (m_nMin == Min() ) ) || ( (p > thumbPos) && ( m_nMin == Max() ) ) )
			if (IsVert())	Inc(); else Dec();
		else
			if (IsVert())	Dec(); else Inc();
	}
	
	Refresh();
}

void JFSlider::LeftRepeat(Point p, dword f)
{
	if(!HasCapture())
		LeftDown(p, f);
}

void JFSlider::LeftUp(Point pos, dword keyflags)
{
	if (HasCapture())
		WhenSlideFinish();
	Refresh();
}

void JFSlider::MouseMove(Point pos, dword keyflags)
{
	if(HasCapture()) {
		
		// for vertical slider, make the bottom end min
		int n = ClientToSlider(HoVe(pos.x, GetSize().cy-pos.y));
		if(n != m_vValues[0]) {
			SetValue( n );
			UpdateActionRefresh();
		}
	}
}

// set the slider value within its' range
void JFSlider::SetData(const Value& v)
{
	int i = v;
	if(!IsNull(i))
		i = minmax(i, Min(), Max() );

	if(i != m_vValues[0]) {
		SetValue( i );
		UpdateRefresh();
	}
}

// read the slider[n] value
Value JFSlider::GetData( int nIndex ) const
{
	return m_vValues[nIndex];
}

// default read of slider value
Value  JFSlider::GetData() const
{
	return m_vValues[0];
}

// Restrict slider to Min, Max values
JFSlider& JFSlider::MinMax(int _min, int _max)
{
	if(m_nMin != _min || m_nMax != _max) {
		m_nMin = _min;
		m_nMax = _max;
		if(!IsNull(m_vValues[0])) {
			int v = minmax(m_vValues[0], Min(), Max());
			if(m_vValues[0] != v) {
				SetValue( v );
				Update();
			}
		}
		Refresh();
	}
	return *this;
}

// CONVERT SLIDER VALUES TO Canvas coordinates
int JFSlider::SliderToClient(int v) const
{
	if(IsNull(v))		return Null;
	
	Size size = GetSize();

	// restrict v to range of slider values
	v = minmax(v, Min(), Max());

	if( v < 0 )
		v = iscalefloor(v - m_nMin, HoVe(size.cx,size.cy)-BORDER2, m_nMax - m_nMin);
	else
		v = iscaleceil(v - m_nMin, HoVe(size.cx, size.cy)-BORDER2, m_nMax - m_nMin);
	
	if (IsVert())	v = (size.cy)-v-BORDER1;
	return v;
}

// convert canvas coordinates to slider value
int JFSlider::ClientToSlider(int p) const
{
	return minmax(m_nMin + iscale(p-BORDER1, m_nMax - m_nMin, HoVe(GetSize().cx , GetSize().cy )- BORDER2),
	                           Min(), Max());

}

// decrement the slider value
void JFSlider::Dec()
{
	int n = m_vValues[0];
	if(IsNull(m_vValues[0]))
		n = Max();
	else
	if(n > Min()) {
		if(m_bRound_step && m_nStep > 1)
			n = idivfloor(n - 1, m_nStep) * m_nStep;
		else
			n -= m_nStep;
		if(n < Min())
			n = Min();
	}
	if(n != m_vValues[0]) {
		SetValue( n );
		WhenSlideFinish();
		UpdateActionRefresh();
	}
}

// increment
void JFSlider::Inc()
{
	int n = m_vValues[0];
	if(IsNull(m_vValues[0]))
		n = Min();
	else
	if(n < Max()) {
		if(m_bRound_step && m_nStep > 1)
			n = idivceil(n + 1, m_nStep) * m_nStep;
		else
			n += m_nStep;
		if(n > Max())
			n = Max();
	}
	if(n != m_vValues[0]) {
		SetValue( n );
		WhenSlideFinish();
		UpdateActionRefresh();
	}
}

// add another slider
JFSlider&	JFSlider::AddOutCtrl( Ctrl* c )
{
	m_vctrlOutput.Add( c );
	
	return *this;
}

// set the value for the slider position
int	JFSlider::SetValue( int n, int nIndex /*= 0 */ )
{
	if( m_vValues.At(nIndex) != n )
	{
		m_vValues.At(nIndex) = n;
	
		if( m_vctrlOutput.GetCount() > nIndex )
			m_vctrlOutput[nIndex]->SetData( n );
		
		UpdateRefresh();
	}
	
	return n;
}

void JFSlider::GotFocus()
{
	Refresh();
}

void JFSlider::LostFocus()
{
	Refresh();
}

JFSlider::~JFSlider() {}
