#include "UppGL.h"

GLTransform::GLTransform()
{
	dirty = true;
	Reset();
}

void GLTransform::Reset(float x, float y, float z)
{
	transform = Mat4f();
	SetPosition(x, y, z);
}

void GLTransform::Apply()
{
	glLoadMatrixf(&transform[0].x);
	glTranslatef(-position.x, -position.y, -position.z);
	dirty = false;
}

void GLTransform::Rotate(float deg, const Vec3f &v)
{
	transform *= glm::rotate(deg, v);
	dirty = true;
}

void GLTransform::SetPosition(const Vec3f &p)
{
	position = p;
	dirty = true;
}

void GLTransform::LookAt(const Vec3f &pos, const Vec3f &target, const Vec3f &up)
{
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	gluLookAt(pos.x, pos.y, pos.z, target.x, target.y, target.z, up.x, up.y, up.z);
	glGetFloatv(GL_MODELVIEW_MATRIX, &transform[0].x);
	glPopMatrix();
	
	transform[3].x = 0.0f;	
	transform[3].y = 0.0f;
	transform[3].z = 0.0f;
	transform[3].w = 1.0f;
	SetPosition(pos);
}

/*
** GLLight
*/

GLLight::GLLight()
{
	ambient = Vec4f(0.85f, 0.85f, 0.85f, 1.0f);
	diffuse = Vec4f(1.0f, 1.0f, 1.0f, 1.0f);
	specular  = Vec4f(0.9f,  0.0f, 0.0f, 1.0f);	
	brightness = Vec4f(1.0f, 1.0f, 1.0f, 1.0f);
	ix = 0;
}

void GLLight::SetLight() const
{
	Vec3f pos = Position();
	Vec4f pos4 = Vec4f(pos.x, pos.y, pos.z, 1.0f);
	glLightfv(GL_LIGHT0+ix, GL_POSITION, &pos4.x);
	SetLightColors();
}

void GLLight::SetLightColors() const
{
	Vec4f temp;
	temp = ambient*brightness[AMBIENT];
	temp.w = ambient.w;
    glLightfv(GL_LIGHT0+ix, GL_AMBIENT, &temp.x);
    temp = diffuse*brightness[DIFFUSE];
    temp.w = diffuse.w;
    glLightfv(GL_LIGHT0+ix, GL_DIFFUSE, &temp.x);
    temp = specular*brightness[SPECULAR];
    temp.w = specular.w;
    glLightfv(GL_LIGHT0+ix, GL_SPECULAR, &temp.x);		
}

/*
** Camera
*/

Camera::Camera(float x, float y, float z) 
{
	Reset();

	dragsensitivity 	= 0.5f;
	keysensitivity		= 1.0f;		
	wheelsensitivity	= 0.1f;
	shiftmult			= 5.0f;
	
	nearplane 	= 1.0f;
	farplane 	= 1000.0f;
	fov 		= 45.0f;
	
	Reset(0.0f, 0.0f, 0.0f);
}

Camera::~Camera() {}

void Camera::Serialize(Stream &s)
{
	s % transform % position;
}

void Camera::Backup()
{
	StringStream s;
	s % *this;
	backup = s;
}

void Camera::Restore()
{
	if (backup.IsEmpty()) return Reset();
	StringStream s(backup);
	s % *this;
}

bool Camera::OnMouseDrag(Point p, dword keyflags)
{
	bool out = MouseDrag(p-drag, keyflags); 
	drag = p;
	return out;
}

bool Camera::OnMouseWheel(Point p, int zdelta, dword keyflags)
{
	return MouseWheel(p, zdelta, keyflags); 
}
