class GLTransform
{
protected:
	bool 		dirty;
	Vec3f		position;		
	Mat4f 		transform;
	
	//Vec3f DirVector(int i) const							{ return Vec3f(transform[i].x, transform[i].y, transform[i].z); }	
	Vec3f DirVector(int i) const							{ const float *m = &transform[0].x; return Vec3f(m[i], m[i+4], m[i+8]); }		
public:	
	GLTransform();
	
	void 	Apply();
	const 	Mat4f &Transform() const						{ return transform; }
	bool	IsDirty() const									{ return dirty; }
		
	void 	Reset(float x=0.0f, float y=0.0f, float z=0.0f);
	void 	SetPosition(const Vec3f &p);
	void 	SetPosition(float x, float y, float z)			{ SetPosition(Vec3f(x, y, z)); }
	void	LookAt(const Vec3f &pos, const Vec3f &target, const Vec3f &up);
	
	void 	Move(float x, float y, float z)					{ Move(Vec3f(x, y, z)); }
	void 	Move(const Vec3f &v)							{ SetPosition(Position() + v); }
	void	MoveRight(float x)								{ Move(Right()*x); }
	void	MoveUp(float y)									{ Move(Up()*y); }
	void	MoveForward(float z)							{ Move(Forward()*z); }

	void 	Rotate(float deg, float x, float y, float z)	{ return Rotate(deg, Vec3f(x, y, z)); }
	void 	Rotate(float deg, const Vec3f &v);	
	void 	Pitch(float deg)								{ Rotate(deg, Right()); }
	void 	Yaw(float deg)									{ Rotate(deg, Up()); }
	void	Roll(float deg)									{ Rotate(deg, Forward()); }	
	
	Vec3f Right() const 									{ return DirVector(0); }
	Vec3f Up() const 										{ return DirVector(1); }
	Vec3f Forward() const 									{ return DirVector(2); }
	Vec3f Position() const 									{ return position; }
};

class GLLight : public GLTransform
{
	Vec4f ambient;
	Vec4f diffuse;
	Vec4f specular;
	Vec4f brightness;
	int ix;

public:	
	enum {
		AMBIENT,
		DIFFUSE,
		SPECULAR
	};
	
public:
	GLLight();

	void 	SetLight() const;
	void 	SetLightColors() const;
	
	GLLight &SetNumber(int _ix)					{ ix = _ix; return *this; }
	int		 Number() const						{ return ix; }
	
	GLLight &SetBrightness(int type, float b) 	{ brightness[type] = b; return *this; }
	float	 Brightness(int type) const			{ return brightness[type]; }
	
	GLLight &SetAmbient(float r, float g, float b, float a = 1.0f) { return SetAmbient(Vec4f(r, g, b, a)); }
	GLLight &SetAmbient(Vec4f v)				{ ambient = v; return *this; }
	GLLight &SetAmbientBrightness(float b) 		{ SetBrightness(AMBIENT, b); return *this; }

	GLLight &SetDiffuse(float r, float g, float b, float a = 1.0f) { return SetDiffuse(Vec4f(r, g, b, a)); }
	GLLight &SetDiffuse(Vec4f v)				{ diffuse = v; return *this; }
	GLLight &SetDiffuseBrightness(float b) 		{ SetBrightness(DIFFUSE, b); return *this; }

	GLLight &SetSpecular(float r, float g, float b, float a = 1.0f) { return SetSpecular(Vec4f(r, g, b, a)); }
	GLLight &SetSpecular(Vec4f v)				{ specular = v; return *this; }
	GLLight &SetSpecularBrightness(float b) 	{ SetBrightness(SPECULAR, b); return *this; }

	const Vec4f	&	Ambient() const				{ return ambient; }
	float			AmbientBrightness() const	{ return Brightness(AMBIENT); }

	const Vec4f	&	Diffuse() const				{ return ambient; }
	float			DiffuseBrightness() const	{ return Brightness(DIFFUSE); }

	const Vec4f	&	Specular() const			{ return ambient; }
	float			SpecularBrightness() const	{ return Brightness(SPECULAR); }
};

class Camera : public GLTransform, public Moveable<Camera> {
private:
	String 		backup;
	Point 		drag;
protected:
	float keysensitivity;
	float dragsensitivity;
	float wheelsensitivity;
	float shiftmult;
	float nearplane, farplane, fov;

	virtual bool MouseDrag(Point dif, dword keyflags) 				{ return false; };
	virtual bool MouseWheel(Point p, int zdelta, dword keyflags) 	{ return false; };
public:
	Camera(float x=0.0f, float y=0.0f, float z=0.0f);
	~Camera();

	virtual void 	SetView()							{ Apply(); }
	
	virtual void Serialize(Stream &s);
	void Backup();
	void Restore();
	
	Camera& 	SetKeySensitivity(float v)  			{ keysensitivity = v; return *this; }
	Camera& 	SetDragSensitivity(float v)  			{ dragsensitivity = v; return *this; }
	Camera& 	SetWheelSensitivity(float v)  			{ wheelsensitivity = v; return *this; }
	Camera&		SetShiftMult(float v)					{ shiftmult = v; return *this; } 
	float 		GetKeySensitivity() const  				{ return keysensitivity; }
	float 		GetDragSensitivity() const  			{ return dragsensitivity; }
	float 		GetWheelSensitivity() const  			{ return wheelsensitivity; }
	float		GetShiftMult() const					{ return shiftmult; } 

	Camera& SetNearPlane(float v)			{ nearplane = v; return *this; }
	Camera& SetFarPlane(float v)			{ farplane = v; return *this; }
	Camera& SetFOV(float v)					{ fov = v; return *this;; }
	float GetNearPlane() const				{ return nearplane; }
	float GetFarPlane() const				{ return farplane; }
	float GetFOV() const					{ return fov; }

	virtual bool Key(dword key, int count) 				{ return false; }
	bool OnDragStart(Point p)							{ drag = p; return false; }
	bool OnMouseDrag(Point p, dword keyflags);
	bool OnMouseWheel(Point p, int zdelta, dword keyflags);	
};

class CallbackCamera : public Camera {
protected:
	virtual bool MouseDrag(Point dif, dword keyflags)				{ return WhenMouseDrag(Vec2f(((float)dif.x) * dragsensitivity, ((float)dif.y) * dragsensitivity), keyflags); }
	virtual bool MouseWheel(Point p, int zdelta, dword keyflags)	{ return WhenMouseWheel(((float)zdelta)*wheelsensitivity, keyflags); }
public:
	Gate2<Vec2f, dword> 	WhenMouseDrag;
	Gate2<float, dword> 	WhenMouseWheel;
	Gate2<dword, float> 	WhenKey;

	virtual bool Key(dword key, int count)							{ return WhenKey(key, ((float)count)*keysensitivity); }
};