typedef GLhandleARB ShaderHandle;

struct ShaderProgramData : public ManagedResourceData
{
	struct Uniform {
		GLint addr;
		
		Uniform() : addr(-1) {}
		virtual void Set(int val) { NEVER(); }
		virtual void Set(float val) { NEVER(); }
		virtual void Set(const Vec3f &val) { NEVER(); }
		void 		 SetAddr(ShaderHandle id, const String &name);
		virtual void Reset() = 0;
	};
	struct IntUniform : public Uniform
	{
		int v;
		virtual void Set(int val);
		virtual void Reset()		{ Set(v); }		
	};
	struct FloatUniform : public Uniform
	{
		float v;
		virtual void Set(float val);
		virtual void Reset()		{ Set(v); }		
	};
	struct Vec3Uniform : public Uniform
	{
		Vec3f v;		
		virtual void Set(const Vec3f &val);
		virtual void Reset()		{ Set(v); }		
	};
		
	ShaderHandle				id;
	ArrayMap<String, Uniform>	uniform;
	String 						vertexfile;
	String 						fragmentfile;
	
	ShaderProgramData() : id(0) {}
	virtual void Free()			{ if (id) glDeleteObjectARB(id); id = 0; }
};

#ifndef flagDUMMY_SHADERPROGRAM
class ShaderProgram : public ManagedResourceNeverNull<ShaderProgramData>
{
	static GLuint 							currentid;
	static VectorMap<String, ShaderHandle>	vertshader;
	static VectorMap<String, ShaderHandle>	fragshader;
	static Vector<String>					defines;
	static bool								definesdirty;
	static String 							definestring;	
	
	friend class ShaderManager;
public:
	void 		Activate() const				{ if (Get()->id != currentid) { glUseProgramObjectARB(Get()->id); currentid = Get()->id; } }
	void 		ActivateAlways() const			{ glUseProgramObjectARB(Get()->id); currentid = Get()->id; }	
	static void Deactivate() 					{ glUseProgramObjectARB(NULL); currentid = 0; }
	
	bool 	Load(const String &vert, const String &frag);
	
	void	ResetUniforms() const;
	bool	SetUniform(String name, int val) const;
	bool	SetUniform(String name, float val) const;
	bool	SetUniform(String name, const Vec3f &val) const;
	int 	UniformCount() const				{ ASSERT(!IsEmpty()); return Get()->uniform.GetCount(); }

	String 	VertexFile() const					{ return Get()->vertexfile; }
	String 	FragmentFile() const				{ return Get()->fragmentfile; }
public:
	static void AddDefine(String def)			{ defines.Add(def); definesdirty = true; }
	static void SetDefine(String def, bool set)	{ definesdirty = true; set ? AddDefine(def) : RemoveDefine(def); }
	static Vector<String> &GetDefines()			{ return defines; }
	static void RemoveDefine(String def);
	static bool ReloadAll();
private:
	String 			CreateProgramName(const String &vert, const String &frag);
	bool			PreProcess(String &s, String name);
	bool			CreateShaderProgram(const String &vert, const String &frag);
	
	ShaderHandle	LoadShaderProgram(const String &vert, const String &frag);
	ShaderHandle	CreateVertexShader(const String &name);
	ShaderHandle	CreateFragmentShader(const String &name);
	ShaderHandle	CreateShaderProgram(ShaderHandle vert, ShaderHandle frag);
	ShaderHandle	CreateShaderPart(const String &name, GLenum type);
};
#else
#ifndef _DEBUG
#error DUMMY_SHADER ENABLED
#endif

class ShaderProgram : public ManagedResourceNeverNull<ShaderProgramData>
{
	friend class ShaderManager;
public:
	void 		Activate() const				{ }
	void 		ActivateAlways() const			{ }
	static void Deactivate() 					{ }
	
	bool 	Load(const String &vert, const String &frag) { return true; }
	
	void	ResetUniforms() const;
	bool	SetUniform(String name, int val) const  { return true; }
	bool	SetUniform(String name, float val) const { return true; }
	bool	SetUniform(String name, const Vec3f &val) const { return true; }
	int 	UniformCount() const				{ return 0; }

	String 	VertexFile() const					{ return "vertexfile"; }
	String 	FragmentFile() const				{ return "fragmentfile"; }
public:
	static void AddDefine(String def)			{ }
	static void SetDefine(String def, bool set)	{ }
	static Vector<String> &GetDefines()			{ }
	static void RemoveDefine(String def)		{}
	static bool ReloadAll()						{ return true; }
	bool	 IsNullInstance() const				{ return false; }
};
#endif