#include "UppGL.h"

#ifdef PLATFORM_WIN32
#define RGBA_FIX(q) Swap((q)->r, (q)->b);
#endif

#ifdef PLATFORM_POSIX
#ifdef CPU_BE
#define RGBA_FIX(q) { Swap((q)->a, (q)->b); Swap((q)->b, (q)->g); Swap((q)->r, (q)->g); }
#else
#define RGBA_FIX(q) Swap((q)->r, (q)->b);
#endif
#endif

#include <plugin/png/png.h>
#include <plugin/jpg/jpg.h>
#include <plugin/bmp/bmp.h>
#include <plugin/gif/gif.h>

void TextureData::Deactivate(int unit)
{	
	glActiveTexture(GL_TEXTURE0 + unit); 
	glBindTexture(GL_TEXTURE_2D, 0); 
	currentid.At(unit) = 0;
}

bool Texture::Load(const String &name)
{
	if (SetResource(name)) 
		return true;
	if (forceque)
		return QueFile(name);
	int id = LoadTexture(name);
	if (!id)
		return false;
	TextureData &t = CreateResource(name);
	t.id = id;
	return true;	
}

void Texture::Disable(int unit)
{ 
	glActiveTexture(GL_TEXTURE0 + unit); 
	glDisable(GL_TEXTURE_2D); 
}

void Texture::Enable(int unit)
{ 
	glActiveTexture(GL_TEXTURE0 + unit); 
	glEnable(GL_TEXTURE_2D); 
}

bool Texture::FromImage(Image img, String name)
{
	if (SetResource(name)) 
		return true;	
	// Create texture
	if (!GLEE_EXT_bgra)
		RGBAFormat(img);
	GLuint id = CreateTexture(img);	
	if (!id)
		GLError("Unable to create GL Color Texture");
	TextureData &t = CreateResource(name);
	t.id = id;
	return true;
}

bool Texture::SetColor(const RGBA &rgba)
{
	String name = AsString(Color(rgba));
	if (SetResource(name)) 
		return true;	
	Image img = CreateColorImage(rgba);
	// Create texture
	GLuint id = CreateTexture(img);
	if (!id)
		GLError("Unable to create GL Color Texture");
	TextureData &t = CreateResource(name);
	t.id = id;
	return true;	
}

bool Texture::QueFile(const String &name)
{
	TextureData &t = CreateResource(name);
	t.id = TextureData::TEXTURE_UNLOADED;
	return true;
}

bool Texture::ProcessQue()
{
	#ifdef _MULTITHREADED
	#error MAKE ME MT!
	#endif
	bool success = true;
	Texture dummy;
	for (int i = 0; i < ResourceCount(); i++) {
		TextureData &t = GetResource(i);	
		if (t.IsUnloaded()) {
			int id = dummy.LoadTexture(t.Name());
			if (!id)	
				success = false;
			else
				t.id = id;
		}
	}
	return success;
}

GLuint Texture::LoadTexture(const String &name)
{
	// Load Image
	GLuint id = 0;
	String file = GetFullPath(TEXTURE_PATH, name);
	/*Image img = StreamRaster::LoadFileAny(file);*/ // This loads images upside down because of the coord-difference between Windows and GL
	Image img = LoadImageAny(file);
	if (!IsNull(img)) {
		if (!GLEE_EXT_bgra)
			RGBAFormat(img);
		id = CreateTexture(img);
		if (!id)
			GLError(Format("Unable to create GL Texture for %s", file));
		else
			return id;
	}
	return 0;
}

GLuint Texture::CreateTexture(Image &img)
{
	ASSERT(wglGetCurrentContext());
	GLuint texnum = 0;
	Size sz = img.GetSize();

	glGenTextures(1, &texnum);
	glBindTexture(GL_TEXTURE_2D, texnum);
	   
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	// TRY: GL_ARB_texture_mirrored_repeat 

    if (anisotropic > 1)
    {
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT,
            anisotropic);
    }
	
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, sz.cx, sz.cy, 0, GLEE_EXT_bgra ? GL_BGRA_EXT : GL_RGBA, GL_UNSIGNED_BYTE, ~img);
	gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGBA, sz.cx, sz.cy, GLEE_EXT_bgra ? GL_BGRA_EXT : GL_RGBA, GL_UNSIGNED_BYTE, ~img);
	
	// This is a bit dodgy, but we have to init this somewhere with a gl context
	if (!num_texture_units)
		glGetIntegerv(GL_MAX_TEXTURE_UNITS, &num_texture_units);
	return texnum;		
}

Size Texture::GetMaxSize()
{
	GLint tsz; 
	glGetIntegerv(GL_MAX_TEXTURE_SIZE, &tsz);
	return Size(tsz, tsz);
}

int Texture::GetMaxAnisotropic()
{
	int result = 1;
	if (_GLEE_EXT_texture_filter_anisotropic)
        glGetIntegerv(GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT, &result);
	return result;
}

void Texture::SetAnisotropic(int level) 
{
	anisotropic = minmax(level, 0, GetMaxAnisotropic());
	if (!GLEE_EXT_texture_filter_anisotropic)
		GLLog("Anisotropic filtering not supported");
	else
		GLLog(Format("Anisotropic filtering level %d", anisotropic));
}

Image Texture::CreateColorImage(RGBA color)
{
	RGBA_FIX(&color);
	return CreateImage(Size(16, 16), color);
}

void RGBAFormat(Image &img)
{
	ImageBuffer ib(img);
	RGBA *eoi = ~ib + ib.GetLength();
	for (RGBA *q = ~ib; q < eoi; q++)
		RGBA_FIX(q);
	img = ib;
}

Image LoadImageAny(const String &file)
{
	String ext = ToUpper(GetFileExt(file));
	if (!FileExists(file)) {
		GLError(Format("Image file %s does not exist", file));
		return Image();
	}
	FileIn fin(file);
	if (!fin.IsOpen()) {
		GLError(Format("Could not open file for read: %s", file));		
		return Image();
	}
	if (ext == ".PNG") {
		RasterFlip<PNGRaster> r;
		if (!r.Open(fin)) {
			GLError(Format("Unrecognised image format: %s", file));
			return Image();
		}
		return r.GetImageFlip();
	}
	else if (ext == ".JPG" || ext == ".JPEG") {
		RasterFlip<JPGRaster> r;
		if (!r.Open(fin)) {
			GLError(Format("Unrecognised image format: %s", file));
			return Image();
		}
		return r.GetImageFlip();
	}
	else if (ext == ".GIF") {
		RasterFlip<GIFRaster> r;
		if (!r.Open(fin)) {
			GLError(Format("Unrecognised image format: %s", file));
			return Image();
		}
		return r.GetImageFlip();
	}
	else if (ext == ".BMP") {
		RasterFlip<BMPRaster> r;
		if (!r.Open(fin)) {
			GLError(Format("Unrecognised image format: %s", file));
			return Image();
		}
		return r.GetImageFlip();
	}
	GLError(Format("Unrecognised image extension: %s", file));
	return Image();
}


Vector<GLuint> TextureData::currentid;
bool Texture::forceque = false;
int	 Texture::anisotropic = 1;
int  Texture::num_texture_units = 0;