struct TextureData : public ManagedResourceData
{
	enum { TEXTURE_NULL = 0, TEXTURE_UNLOADED = 0xDEADBEEF };
	GLuint 					id;
	static Vector<GLuint> 	currentid;

	TextureData() : id(0) 	{}	
	
	GLuint Id()	const		{ return id; }
	
	bool IsLoaded() const	{ return (id != TEXTURE_NULL && id != TEXTURE_UNLOADED); }
	bool IsUnloaded() const	{ return (id == TEXTURE_UNLOADED); }
	
	inline void	 ActivateAlways(int unit = 0) const;
	inline void  Activate(int unit = 0) const;
	static void  Deactivate(int unit);
	
	virtual void Free()		{ if (IsLoaded()) glDeleteTextures(1, &id); id = 0; }
};

class Texture : public ManagedResourceNeverNull<TextureData> {
	static bool				forceque;
	static int				anisotropic;
	static int 				num_texture_units;		
public:
    // NOT THREADSAFE
    bool		Load(const String &name);
    bool		FromImage(Image img, String name);
	bool		SetColor(const RGBA &rgba);
    // THREADSAFE
    bool		QueFile(const String &name);
    static bool	ProcessQue();
	static void AlwaysQue(bool v = true)	{ forceque = v; }

	static void Enable(int unit = 0);
	static void Disable(int unit = 0);
	static void Deactivate(int unit = 0)	{ TextureData::Deactivate(unit); }
	static void DeactivateAll()				{ for (int i = 0; i < TextureData::currentid.GetCount(); i++) Deactivate(i); }

	static Size	GetMaxSize();
	static int  GetNumUnits()				{ return num_texture_units; }
	static int	GetMaxAnisotropic();
	static void SetAnisotropic(int level);

	void		ActivateAlways(int unit = 0) const 	{ Get()->ActivateAlways(unit); }
	void 		Activate(int unit = 0) const		{ Get()->Activate(unit); }

	GLuint		Id() const							{ return Get() ? Get()->Id() : 0; }

	bool operator<(const Texture &t) const			{ return Get() < t.Get(); }
	bool operator!=(const Texture &t) const			{ return Get() != t.Get(); }
	bool operator==(const Texture &t) const			{ return Get() == t.Get(); }
private:
	GLuint 		CreateTexture(Image &img);
	Image		CreateColorImage(RGBA color);
	GLuint		LoadTexture(const String &name);	
};

void 		RGBAFormat(Image &img);	
Image 		LoadImageAny(const String &file);

void TextureData::Activate(int unit) const
{
	ASSERT(!IsUnloaded());
	if (currentid.At(unit, 0) != id) {
		glActiveTexture(GL_TEXTURE0 + unit);
		glBindTexture(GL_TEXTURE_2D, id);
		currentid[unit] = id;
	}
}
void TextureData::ActivateAlways(int unit) const
{
	ASSERT(!IsUnloaded());
	glActiveTexture(GL_TEXTURE0 + unit);
	glBindTexture(GL_TEXTURE_2D, id);
	currentid.At(unit) = id;	
}

/* 
**	Custom Image Loading
*/

template <class T>
struct RasterFlip : public T
{
	Image GetImageFlip() {
		Size sz = GetSize();
		return GetImageFlip(0, 0, sz.cx, sz.cy);
	}	
	Image  GetImageFlip(int x, int y, int cx, int cy);	
};

template <class T>
Image RasterFlip<T>::GetImageFlip(int x, int y, int cx, int cy)
{
	// Read raster backwards to invert image at no cost
	Size size = GetSize();
	y = minmax(y, 0, size.cy);
	int yy = minmax(y + cy, y, size.cy);
	x = minmax(x, 0, size.cx);
	cx = minmax(x + cx, x, size.cx) - x;
	ImageBuffer b(cx, yy - y);
	RGBA* t = b + b.GetLength() - cx; // Start from last row
	int y0 = y;
	while(y < yy) {
		memcpy(t, ~GetLine(y) + x, cx * sizeof(RGBA));
		t -= cx; // Previous row
		y++;
	}
	Info f = GetInfo();
	b.SetHotSpot(f.hotspot - Point(x, y0));
	if(size.cx && size.cy)
		b.SetDots(Size(f.dots.cx * cx / size.cx, f.dots.cy * cy / size.cy));
	return IsError() ? Image() : Image(b);
}

