#include "UppGL.h"

#define ANG2RAD 3.14159265358979323846f/360.0f 

Frustum::Frustum() 
{
	points.SetCount(8);
	lines.SetCount(8);
}

Frustum::~Frustum() {}

void Frustum::SetProjection(float _angle, float _ratio, float _nearD, float _farD) {

	// store the information
	ratio = _ratio;
	angle = _angle * ANG2RAD;
	nearD = _nearD;
	farD = _farD;

	// compute width and height of the near and far plane sections
	tang = tan(angle);
	sphereFactorY = 1.0f/cos(angle);//tang * sin(this->angle) + cos(this->angle);

	double anglex = atan(tang*ratio);
	sphereFactorX = (float)(1.0f/cos(anglex)); //tang*ratio * sin(anglex) + cos(anglex);

	nh = nearD * tang;
	nw = nh * ratio; 

	fh = farD * tang;
	fw = fh * ratio;
}

void Frustum::SetCamera(const Vec3f &position, const Vec3f &_x, const Vec3f &_y, const Vec3f &_z) {

	Vec3f dir, nc, fc;

	camPos = position;

	Z = glm::normalize(_z);
	X = glm::normalize(_x);
	Y = glm::normalize(_y);
}

void Frustum::CalcPoints()
{

	points.Add(Vec3f(ftl.x, ftl.y, ftl.z));
	points.Add(Vec3f(ftr.x, ftr.y, ftr.z));
	points.Add(Vec3f(fbl.x, fbl.y, fbl.z));
	points.Add(Vec3f(fbr.x, fbr.y, fbr.z));	
}

void Frustum::CalcNearPoints()
{
	// compute the center of the near plane
	Vec3f nc = camPos - Z * nearD;
	
	// compute the 4 corners of the frustum
	ntl = nc + Y * nh - X * nw;
	ntr = nc + Y * nh + X * nw;
	nbl = nc - Y * nh - X * nw;
	nbr = nc - Y * nh + X * nw;	
	
	points[4] = Vec3f(ntl.x, ntl.y, ntl.z);
	points[5] = Vec3f(ntr.x, ntr.y, ntr.z);
	points[6] = Vec3f(nbl.x, nbl.y, nbl.z);
	points[7] = Vec3f(nbr.x, nbr.y, nbr.z);	
}

void Frustum::CalcFarPoints()
{
	// compute the center of far plane
	Vec3f fc = camPos + Z * farD;	
	points[0] = fc + Y * fh - X * fw;
	points[1] = fc + Y * fh + X * fw;
	points[2] = fc - Y * fh - X * fw;	
	points[3] = fc - Y * fh + X * fw;	
}

void Frustum::CalcLines()
{
	// TODO: Avoid sqrts by caching lengths
	CalcFarPoints();
	lines[0].Set(camPos, points[0]);
	lines[1].Set(camPos, points[1], lines[0].length);
	lines[2].Set(camPos, points[2], lines[0].length);
	lines[3].Set(camPos, points[3], lines[0].length);
	lines[4].Set(points[0], points[1]);
	lines[5].Set(points[1], points[3]);
	lines[6].Set(points[2], points[0]);
	lines[7].Set(points[3], points[2]);
}

int Frustum::ZPlaneIntersect(Vector<Vec2f> &pt, float z)
{
	int cnt = 0;
	int ix;
	if (z == lines[0].start.z) z += 0.0005f; 
	for (ix = 0; ix < 4; ix++) {
		if (ZLinePlaneInstersect(lines[ix], z, pt[cnt]))
			cnt++;
	}
	if (cnt == 4) return 4;
	for (ix; ix < 8; ix++) {
		if (ZLinePlaneInstersect(lines[ix], z, pt[cnt]))
			cnt++;
	}	
	ASSERT(cnt != 1 && cnt != 2);
	return cnt;
}

bool Frustum::ZLinePlaneInstersect(const Line &line, float z, Vec2f &out)
{
//	if ((a.z <= b.z && z < a.z && z > b.z) || (a.z > b.z && z < b.z && z > a.z))
//		return false;
	if (line.zdiv == 0.0f) return false;
	float delta = (z - line.start.z) * line.zdiv;
	if (delta < 0.0f || delta > line.length) return false;
	out.x = line.start.x + line.unit.x * delta;
	out.y = line.start.y + line.unit.y * delta;
	return true;
}

void Frustum::CalcPlanes()
{
	const Vec3f &p = camPos;
	// compute the center of the near and far planes
	Vec3f nc = camPos - Z * nearD;
	Vec3f fc = camPos - Z * farD;	

	// compute the six planes
	// the function set3Points asssumes that the points
	// are given in counter clockwise order
	pl[TOP].Set3Points(ntr, ntl, ftl);
	pl[BOTTOM].Set3Points(nbl, nbr, fbr);
	pl[LEFT].Set3Points(ntl, nbl, fbl);
	pl[RIGHT].Set3Points(nbr, ntr, fbr);
//	pl[NEARP].set3Points(ntl, ntr, nbr);
//	pl[FARP].set3Points(ftr, ftl, fbl);

	pl[NEARP].SetNormalAndPoint(-Z, nc);
	pl[FARP].SetNormalAndPoint(Z, fc);

	Vec3f aux, normal;

	aux = (nc + Y*nh) - p;
	normal = aux * X;
	pl[TOP].SetNormalAndPoint(normal, nc+Y*nh);

	aux = (nc - Y*nh) - p;
	normal = X * aux;
	pl[BOTTOM].SetNormalAndPoint(normal, nc-Y*nh);
	
	aux = (nc - X*nw) - p;
	normal = aux * Y;
	pl[LEFT].SetNormalAndPoint(normal, nc-X*nw);

	aux = (nc + X*nw) - p;
	normal = Y * aux;
	pl[RIGHT].SetNormalAndPoint(normal, nc+X*nw);
}

int Frustum::PointInFrustum(const Vec3f &p) const 
{
	float pcz,pcx,pcy,aux;
	// compute vector from camera position to p
	Vec3f v = p-camPos;
	// compute and test the Z coordinate
	pcz = glm::dot(v, -Z);
	if (pcz > farD || pcz < nearD)
		return OUTSIDE;
	// compute and test the Y coordinate
	pcy = glm::dot(v, Y);
	aux = (float)(pcz * tang);
	if (pcy > aux || pcy < -aux)
		return OUTSIDE;
	// compute and test the X coordinate
	pcx = glm::dot(v, X);
	aux = aux * ratio;
	if (pcx > aux || pcx < -aux)
		return OUTSIDE;
	return INSIDE;
}

int Frustum::SphereInFrustum(const Vec3f &p, float radius) const
{
	float d1,d2;
	float az,ax,ay,zz1,zz2;

	Vec3f v = p-camPos;

	az = glm::dot(v, -Z);
	if (az > farD + radius || az < nearD-radius)
		return OUTSIDE;

	ax = glm::dot(v, X);
	zz2 = (float)(az * tang);
	zz1 = zz2 * ratio;
	d1 = sphereFactorX * radius;
	if (ax > zz1+d1 || ax < -zz1-d1)
		return OUTSIDE;

	ay = glm::dot(v, Y);
	d2 = sphereFactorY * radius;
	if (ay > zz2+d2 || ay < -zz2-d2)
		return OUTSIDE;

	if (az > farD - radius || az < nearD+radius || ay > zz2-d2 
		|| ay < -zz2+d2 || ax > zz1-d1 || ax < -zz1+d1)
		return INTERSECT;
	return INSIDE;
}

int Frustum::InclusiveSphereInFrustum(const Vec3f &p, float radius) const
{
	float d1,d2;
	float az,ax,ay,zz1,zz2;

	Vec3f v = p-camPos;

	az = glm::dot(v, -Z);
	if (az > farD + radius || az < nearD-radius)
		return OUTSIDE;

	ax = glm::dot(v, X);
	zz2 = (float)(az * tang); // For y
	zz1 = zz2 * ratio; // For x
	d1 = sphereFactorX * radius;
	if (ax > zz1+d1 || ax < -zz1-d1)
		return OUTSIDE;

	ay = glm::dot(v, Y);
	zz2 = (float)(az * tang);
	d2 = sphereFactorY * radius;
	if (ay > zz2+d2 || ay < -zz2-d2)
		return OUTSIDE;

	if (az <= farD && az >= nearD && ay <= zz2 && ay >= -zz2 && ax <= zz1 && ax >= -zz1)
		return INSIDE;
	else if (az > farD - radius || az < nearD+radius || ay > zz2-d2 
		|| ay < -zz2+d2 || ax > zz1-d1 || ax < -zz1+d1)
		return INTERSECT;
	return INSIDE;	
}

void Frustum::DrawPoints() const
{
	glBegin(GL_POINTS);
		glVertex3f(ntl.x, ntl.y, ntl.z);
		glVertex3f(ntr.x, ntr.y, ntr.z);
		glVertex3f(nbl.x, nbl.y, nbl.z);
		glVertex3f(nbr.x, nbr.y, nbr.z);

		glVertex3f(ftl.x, ftl.y, ftl.z);
		glVertex3f(ftr.x, ftr.y, ftr.z);
		glVertex3f(fbl.x, fbl.y, fbl.z);
		glVertex3f(fbr.x, fbr.y, fbr.z);
	glEnd();
}

void Frustum::DrawLines() const 
{
	glBegin(GL_LINES);
		for (int i = 0; i < 8; i++) {
			Vec3f p = lines[i].start + lines[i].unit * lines[i].length;
			Vec3f s = lines[i].start;
			glVertex3fv(&lines[i].start.x);
			glVertex3fv(&p.x);
		}
	glEnd();
	return;
	
	glBegin(GL_LINE_LOOP);
	//near plane
		glVertex3f(ntl.x, ntl.y, ntl.z);
		glVertex3f(ntr.x, ntr.y, ntr.z);
		glVertex3f(nbr.x, nbr.y, nbr.z);
		glVertex3f(nbl.x, nbl.y, nbl.z);
	glEnd();

	glBegin(GL_LINE_LOOP);
	//far plane
		glVertex3f(ftr.x, ftr.y, ftr.z);
		glVertex3f(ftl.x, ftl.y, ftl.z);
		glVertex3f(fbl.x, fbl.y, fbl.z);
		glVertex3f(fbr.x, fbr.y, fbr.z);
	glEnd();

	glBegin(GL_LINE_LOOP);
	//bottom plane
		glVertex3f(nbl.x, nbl.y, nbl.z);
		glVertex3f(nbr.x, nbr.y, nbr.z);
		glVertex3f(fbr.x, fbr.y, fbr.z);
		glVertex3f(fbl.x, fbl.y, fbl.z);
	glEnd();

	glBegin(GL_LINE_LOOP);
	//top plane
		glVertex3f(ntr.x, ntr.y, ntr.z);
		glVertex3f(ntl.x, ntl.y, ntl.z);
		glVertex3f(ftl.x, ftl.y, ftl.z);
		glVertex3f(ftr.x, ftr.y, ftr.z);
	glEnd();

	glBegin(GL_LINE_LOOP);
	//left plane
		glVertex3f(ntl.x, ntl.y, ntl.z);
		glVertex3f(nbl.x, nbl.y, nbl.z);
		glVertex3f(fbl.x, fbl.y, fbl.z);
		glVertex3f(ftl.x, ftl.y, ftl.z);
	glEnd();

	glBegin(GL_LINE_LOOP);
	// right plane
		glVertex3f(nbr.x, nbr.y, nbr.z);
		glVertex3f(ntr.x, ntr.y, ntr.z);
		glVertex3f(ftr.x, ftr.y, ftr.z);
		glVertex3f(fbr.x, fbr.y, fbr.z);
	glEnd();
}


void Frustum::DrawPlanes() const
{
	glBegin(GL_QUADS);
	//near plane
		glVertex3f(ntl.x, ntl.y, ntl.z);
		glVertex3f(ntr.x, ntr.y, ntr.z);
		glVertex3f(nbr.x, nbr.y, nbr.z);
		glVertex3f(nbl.x, nbl.y, nbl.z);

	//far plane
		glVertex3f(ftr.x, ftr.y, ftr.z);
		glVertex3f(ftl.x, ftl.y, ftl.z);
		glVertex3f(fbl.x, fbl.y, fbl.z);
		glVertex3f(fbr.x, fbr.y, fbr.z);

	//bottom plane
		glVertex3f(nbl.x, nbl.y, nbl.z);
		glVertex3f(nbr.x, nbr.y, nbr.z);
		glVertex3f(fbr.x, fbr.y, fbr.z);
		glVertex3f(fbl.x, fbl.y, fbl.z);

	//top plane
		glVertex3f(ntr.x, ntr.y, ntr.z);
		glVertex3f(ntl.x, ntl.y, ntl.z);
		glVertex3f(ftl.x, ftl.y, ftl.z);
		glVertex3f(ftr.x, ftr.y, ftr.z);

	//left plane

		glVertex3f(ntl.x, ntl.y, ntl.z);
		glVertex3f(nbl.x, nbl.y, nbl.z);
		glVertex3f(fbl.x, fbl.y, fbl.z);
		glVertex3f(ftl.x, ftl.y, ftl.z);

	// right plane
		glVertex3f(nbr.x, nbr.y, nbr.z);
		glVertex3f(ntr.x, ntr.y, ntr.z);
		glVertex3f(ftr.x, ftr.y, ftr.z);
		glVertex3f(fbr.x, fbr.y, fbr.z);

	glEnd();

}

void Frustum::DrawNormals() const
{
	Vec3f a,b;

	glBegin(GL_LINES);
		// near
		a = (ntr + ntl + nbr + nbl) * 0.25f;
		b = a + pl[NEARP].normal;
		glVertex3f(a.x, a.y, a.z);
		glVertex3f(b.x, b.y, b.z);

		// far
		a = (ftr + ftl + fbr + fbl) * 0.25f;
		b = a + pl[FARP].normal;
		glVertex3f(a.x, a.y, a.z);
		glVertex3f(b.x, b.y, b.z);

		// left
		a = (ftl + fbl + nbl + ntl) * 0.25f;
		b = a + pl[LEFT].normal;
		glVertex3f(a.x, a.y, a.z);
		glVertex3f(b.x, b.y, b.z);
		
		// right
		a = (ftr + nbr + fbr + ntr) * 0.25f;
		b = a + pl[RIGHT].normal;
		glVertex3f(a.x, a.y, a.z);
		glVertex3f(b.x, b.y, b.z);
		
		// top
		a = (ftr + ftl + ntr + ntl) * 0.25f;
		b = a + pl[TOP].normal;
		glVertex3f(a.x, a.y, a.z);
		glVertex3f(b.x, b.y, b.z);
		
		// bottom
		a = (fbr + fbl + nbr + nbl) * 0.25f;
		b = a + pl[BOTTOM].normal;
		glVertex3f(a.x, a.y, a.z);
		glVertex3f(b.x, b.y, b.z);
	glEnd();
}


Plane::Plane(const Vec3f &v1, const Vec3f &v2, const Vec3f &v3)
{
	Set3Points(v1,v2,v3);
}

Plane::Plane(void)
{
}

Plane::~Plane()
{
}

void Plane::Set3Points(const Vec3f &v1, const Vec3f &v2, const Vec3f &v3)
{
	Vec3f aux1, aux2;

	aux1 = v1 - v2;
	aux2 = v3 - v2;

	normal = glm::cross(aux2, aux1);

	normal = glm::normalize(normal);
	point = v2;
	d = -glm::dot(normal, point);	
}

void Plane::SetNormalAndPoint(const Vec3f &_normal, const Vec3f &_point)
{
	normal = glm::normalize(_normal);
	point = _point;
	d = -glm::dot(normal, point);		
}

void Plane::SetCoefficients(float a, float b, float c, float _d)
{
	// set the normal vector
	normal = Vec3f(a, b, c);
	//compute the lenght of the vector
	float l = glm::length(normal);
	// normalize the vector
	normal /= l;
	// and divide d by th length as well
	d = _d/l;	
}

float Plane::Distance(const Vec3f &p)
{
	return (d + glm::dot(normal, p));	
}

